package ohd.hseb.hefs.mefp.models.parameters;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.models.MEFPParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.log.QuestionableLog;
import ohd.hseb.hefs.mefp.models.temperature.TemperatureParameterEstimationModel;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.cfsv2.CFSv2ForecastSource;
import ohd.hseb.hefs.mefp.sources.gefs.GEFSForecastSource;
import ohd.hseb.hefs.mefp.sources.historical.HistoricalForecastSource;
import ohd.hseb.hefs.mefp.sources.historical.ProcessedHistoricalBinaryFileTools;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.OneSetParameterValues;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.about.AboutFile;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.TarTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

/**
 * Handles all parameters for a model execution: algorithm parameters and source-specific parameters. It is assumed that
 * the data type in the LocationAndDataTypeIdentifier provided fully specifies the model to use. The
 * AlgorithmModelParameters is then determined based on the LocationAndDataTypeIdentifier, along with an IO handler to
 * handle reading/writing parameters.<br>
 * <br>
 * Whenever this is constructed, the model will be given {@link EstimationControlOptions}. This is to ensure that it any
 * model contained within this will have the requisite estimation model parameters in place.
 * 
 * @author hank.herr
 */
@AboutFile("version/hefsplugins_config.xml")
public class MEFPFullModelParameters extends FullModelParameters
{
    private static final Logger LOG = LogManager.getLogger(MEFPFullModelParameters.class);

    private final List<TimeSeriesArray> _historicalTimeSeries = new ArrayList<TimeSeriesArray>();

    private final QuestionableLog _questionableParameterLog;

    /**
     * Records run scenarios for each forecast source, which currently only includes if the events must be read in and
     * stored for a source. If a source has an entry in this map, it is assumed that the model will be run for that
     * source. If the entry is true, then events must be loaded when reading parameters; events are not read in if false
     * (allows for faster reading).
     */
    private Map<MEFPForecastSource, Boolean> _operationalSourceToLoadEventsMap = null;

    //This is called during parameter estimation.
    public MEFPFullModelParameters(final LocationAndDataTypeIdentifier identifier,
                                   final EstimationControlOptions estimationControlOptions,
                                   final List<? extends ForecastSource> forecastSourcesOrderedBasedOnParameterFile)
    {
        super(identifier, estimationControlOptions, forecastSourcesOrderedBasedOnParameterFile);
        _questionableParameterLog = new QuestionableLog(forecastSourcesOrderedBasedOnParameterFile);
        this.getModel().setEstimationCtlOptions(getEstimationControlOptions()); //Make sure the model knows about the estimation control parameters.
    }

    //This is called via the estimated parameters file handler.
    public MEFPFullModelParameters(final LocationAndDataTypeIdentifier identifier,
                                   final List<? extends ForecastSource> forecastSourcesOrderedBasedOnParameterFile)
    {
        super(identifier, forecastSourcesOrderedBasedOnParameterFile);
        _questionableParameterLog = new QuestionableLog(forecastSourcesOrderedBasedOnParameterFile);
        this.getModel().setEstimationCtlOptions(getEstimationControlOptions()); //Make sure the model knows about the estimation control parameters.
    }

    //This is called via the adapter.
    /**
     * This is a barebones constructor. It calls {@link FullModelParameters#FullModelParameters(List)}, without
     * knowledge of the identifier. Without the identifier, the model cannot be set. This must be handled later before
     * the model is used. This should only be called when the parameters are about to be read in from the parameter file
     * (the reading process initializes required parameters).
     * 
     * @param forecastSourcesOrderedBasedOnParameterFile
     */
    public MEFPFullModelParameters(final List<? extends ForecastSource> forecastSourcesOrderedBasedOnParameterFile)
    {
        super(forecastSourcesOrderedBasedOnParameterFile);
        _questionableParameterLog = new QuestionableLog(forecastSourcesOrderedBasedOnParameterFile);
    }

    /**
     * Prepares all source model parameters for calculation. Currently it calls {@link #prepareSourceModelParameters()}.
     * Is there anything else that needs to be called? Should this be removed? TODO
     */
    public void prepareParametersForComputation()
    {
        prepareSourceModelParameters();

        //Anything else?
    }

    /**
     * Currently, this just passes the call down to
     * {@link #prepareSourceModelParametersForComputationOfOneSource(List, MEFPForecastSource)}. If anything is added to
     * {@link #prepareParametersForComputation()}, this may need to be updated to include the added stuff as well.
     */
    public void prepareParametersForComputationOfOneSource(final List<MEFPForecastSource> availableSources,
                                                           final MEFPForecastSource estimatedSource)
    {
        prepareSourceModelParametersForComputationOfOneSource(availableSources, estimatedSource);
    }

    /**
     * Prepares the source model parameters for the estimation of a single source. The existing source model parameters
     * must be initialized from a parameter file. This will keep the source model parameters for any source for which
     * parameters are NOT to be estimated AND that is a currently available source. In other words, if the forecast
     * source is no longer used, it will be removed from the parameters.
     * 
     * @param availableSources Source available for parameter estimation at this time; as specified through the MEFPPE.
     * @param estimatedSource The source to be estimated, which must be in the list of available sources.
     */
    private void prepareSourceModelParametersForComputationOfOneSource(final List<MEFPForecastSource> availableSources,
                                                                       final MEFPForecastSource estimatedSource)
    {
        final List<SourceModelParameters> existingSourceModelParameters = new ArrayList<>(getOrderedSourceModelParameters());

        setOrderedForecastSources(availableSources);
        initializeSourceModelParameters();

        //Loop over all initialized source model parameters...
        for(int sourceParmIndex = 0; sourceParmIndex < getOrderedSourceModelParameters().size(); sourceParmIndex++)
        {
            SourceModelParameters srcParms = getOrderedSourceModelParameters().get(sourceParmIndex);
            SourceModelParameters matchingOldSrcParms = null;

            //If the current src parameters are NOT for the source to be estimated, then see if a match can be found among
            //the old src parms.
            if(!srcParms.getForecastSource().equals(estimatedSource))
            {
                for(final SourceModelParameters oldSrcParms: existingSourceModelParameters)
                {
                    if(oldSrcParms.getForecastSource().equals(srcParms.getForecastSource()))
                    {
                        matchingOldSrcParms = oldSrcParms;
                        break;
                    }
                }
            }

            //In any case, set the applicable model.  This must be done before the setupForStoringParametersAndEvents method is called below
            //when matchOldSrcParms is null.
            ((MEFPSourceModelParameters)srcParms).setApplicableModel(getModel());

            //If a match was found among the old parameters, then move them over and reset the srcParms appropriately.
            if(matchingOldSrcParms != null)
            {
                getOrderedSourceModelParameters().set(sourceParmIndex, matchingOldSrcParms);
                srcParms = getOrderedSourceModelParameters().get(sourceParmIndex);
            }
            //If not, or if the source is to be estimated, just set the number of forecast lead days.
            else
            {
                //Initialize so that the source has no parameters estimated for it.  It will then need to be overridden when reading in 
                //or estimating parameters to set it to be non-zero.  Call the setup method to ensure all required objects are initialized.
                ((MEFPSourceModelParameters)srcParms).setupForStoringParametersAndEvents(0);
            }
        }

        //Clear out the logs
        getQuestionableParameterLog().removeAllEntriesForSource(estimatedSource);
    }

    /**
     * Only used operationally, when generating ensembles. Clear the {@link #_operationalSourceToLoadEventsMap}.
     */
    public void clearOperationalRunScenarios()
    {
        _operationalSourceToLoadEventsMap = null;
    }

    /**
     * Adds an operational run time scenario for the given source. This is used during parameter reading to only read
     * needed information.
     * 
     * @param source The source which will be used during ensemble generation.
     * @param loadEvents Identifies if canonical events must be loaded for the source.
     */
    public void addOperationalRunScenario(final MEFPForecastSource source, final boolean loadEvents)
    {
        if(_operationalSourceToLoadEventsMap == null)
        {
            _operationalSourceToLoadEventsMap = Maps.newHashMap();
        }
        _operationalSourceToLoadEventsMap.put(source, loadEvents);
    }

    public QuestionableLog getQuestionableParameterLog()
    {
        return _questionableParameterLog;
    }

    /**
     * This is only used in testing and is called via the source specific get methods, like
     * {@link #getRFCSourceModelParameters()}. DO NOT CALL THIS OUTSIDE OF TESTING!!!
     */
    private int getOrderIndexOfSource(final MEFPForecastSource source)
    {
        return getOrderedForecastSources().indexOf(source);
    }

    /**
     * Short-hand that should only be used in testing, because in reality the list of sources may vary.
     */
    public MEFPSourceModelParameters getRFCSourceModelParameters()
    {
        return (MEFPSourceModelParameters)getOrderedSourceModelParameters().get(getOrderIndexOfSource(new RFCForecastSource()));
    }

    /**
     * Short-hand that should only be used in testing, because in reality the list of sources may vary.
     */
    public MEFPSourceModelParameters getGEFSSourceModelParameters()
    {
        return (MEFPSourceModelParameters)getOrderedSourceModelParameters().get(getOrderIndexOfSource(new GEFSForecastSource()));
    }

    /**
     * Short-hand that should only be used in testing, because in reality the list of sources may vary.
     */
    public MEFPSourceModelParameters getCFSv2SourceModelParameters()
    {
        return (MEFPSourceModelParameters)getOrderedSourceModelParameters().get(getOrderIndexOfSource(new CFSv2ForecastSource()));
    }

    /**
     * Short-hand that should only be used in testing, because in reality the list of sources may vary.
     */
    public MEFPSourceModelParameters getClimSourceModelParameters()
    {
        return (MEFPSourceModelParameters)getOrderedSourceModelParameters().get(getOrderIndexOfSource(new HistoricalForecastSource()));
    }

    /**
     * @return The single historical time series contained in {@link #_historicalTimeSeries}.
     * @throws IllegalStateException if more than one time series is contained within {@link #_historicalTimeSeries}.
     */
    public TimeSeriesArray getSingleHistoricalTimeSeries()
    {
        if(_historicalTimeSeries.size() > 1)
        {
            throw new IllegalStateException("Attempt to acquire a single historical time series from parameters when there are "
                + _historicalTimeSeries.size() + " many.");
        }
        return _historicalTimeSeries.get(0);
    }

    public List<TimeSeriesArray> getHistoricalTimeSeries()
    {
        return _historicalTimeSeries;
    }

    /**
     * This is public because it might be reasonable for the parameters to be read without historical time series.
     * Making it public allows for something external to read in that time series instead and set it for these
     * parameters (this is what we will do when reading binary data; the sbin files will be read for the historical time
     * series and put in here to execute the ensemble generation).
     */
    public void addHistoricalTimeSeries(final TimeSeriesArray historicalTimeSeries)
    {
        _historicalTimeSeries.add(historicalTimeSeries);
    }

    /**
     * This is public because it might be reasonable for the parameters to be read without historical time series.
     * Making it public allows for something external to read in that time series instead and set it for these
     * parameters (this is what we will do when reading binary data; the sbin files will be read for the historical time
     * series and put in here to execute the ensemble generation).
     */
    public void setHistoricalTimeSeries(final TimeSeriesArrays historicalTS)
    {
        _historicalTimeSeries.clear();
        _historicalTimeSeries.addAll(TimeSeriesArraysTools.convertTimeSeriesArraysToList(historicalTS));
    }

    /**
     * @return {@link OneSetParameterValues} instance corresponding to the provided source, day of year, and event.
     */
    public OneSetParameterValues getSourceEstimatedModelParameterValues(final ForecastSource source,
                                                                        final int dayOfYear,
                                                                        final CanonicalEvent event)
    {
        final int canonicalEventIndex = this.getSourceModelParameters(source).getComputedEvents().indexOf(event);
        if(canonicalEventIndex < 0)
        {
            throw new IllegalArgumentException("Event '" + event.toString()
                + "' is not in the list of computed events for the source " + source.getName());
        }
        return getSourceEstimatedModelParameterValues(source, dayOfYear, canonicalEventIndex);
    }

    @Override
    public MEFPAlgorithmModelParameters getAlgorithmModelParameters()
    {
        return (MEFPAlgorithmModelParameters)super.getAlgorithmModelParameters();
    }

    @Override
    public MEFPEstimationControlOptions getEstimationControlOptions()
    {
        return (MEFPEstimationControlOptions)super.getEstimationControlOptions();
    }

    @Override
    public List<MEFPForecastSource> getOrderedForecastSources()
    {
        return ListTools.convertCollection(super.getOrderedForecastSources(), (MEFPForecastSource)null);
    }

    @Override
    public MEFPParameterEstimationModel getModel()
    {
        return (MEFPParameterEstimationModel)super.getModel();
    }

    @Override
    public MEFPSourceModelParameters getSourceModelParameters(final ForecastSource source)
    {
        return (MEFPSourceModelParameters)super.getSourceModelParameters(source);
    }

    @Override
    public MEFPSourceModelParameters getSourceModelParameters(final int sourceIndex)
    {
        return (MEFPSourceModelParameters)super.getSourceModelParameters(sourceIndex);
    }

    @Override
    public OneSetParameterValues getSourceEstimatedModelParameterValues(final ForecastSource source,
                                                                        final int dayOfYear,
                                                                        final int secondaryIndex)
    {
        final OneSetParameterValues values = this.getModel().initializeOneSetParameterValues(source,
                                                                                             dayOfYear,
                                                                                             secondaryIndex);
        this.putValuesInto(values);
        return values;
    }

    @Override
    protected ParameterEstimationModel buildModelInstance()
    {
        final MEFPParameterEstimationModel model = MEFPTools.determineParameterEstimationModel(getIdentifier().getParameterIdType());
        model.setEstimationCtlOptions(getEstimationControlOptions());
        return model;
    }

    @Override
    protected EstimationControlOptions buildEstimationControlOptions()
    {
        return MEFPTools.constructEstimationControlOptions(getIdentifier().getParameterIdType(),
                                                           ListTools.convertCollection(this.getOrderedForecastSources(),
                                                                                       (MEFPForecastSource)null));
    }

    @Override
    protected boolean isSourceModelParameterEntryFile(final String entryName)
    {
        return entryName.contains("SourceModelParameters");
    }

    @Override
    public void prepareEstimationOptionsForComputationOfOneSource(final EstimationControlOptions currentRunTimeOptions,
                                                                  final ForecastSource estimatedSource)
    {
        //Copy the old source estimation options.
        final MEFPEstimationControlOptions clonedRunTimeOptions = (MEFPEstimationControlOptions)currentRunTimeOptions.clone();

        //Copy the model parameters from the old to the new.
        clonedRunTimeOptions.getModelParameters().copyFrom(getEstimationControlOptions().getModelParameters());

        //For each run-time source, copy the old options if they exist, unless this is the estimated source.
        for(final MEFPForecastSource source: clonedRunTimeOptions.getForecastSources())
        {
            if(source.equals(estimatedSource))
            {
                continue;
            }

            //If it exists...
            if(Iterables.contains(getEstimationControlOptions().getForecastSources(), source))
            {
                clonedRunTimeOptions.getSourceControlOptions(source)
                                    .copyFrom(getEstimationControlOptions().getSourceControlOptions(source));
            }
            //Otherwise, use the existing, but set the number of days to be 0.  This will occur if the source
            //is new relative to the parameter file and is NOT to be estimated now.
            else
            {
                clonedRunTimeOptions.getSourceControlOptions(source).setEnabled(false);
            }
        }

        //Cloned run-time options now represents the new options for the estimated parameters
        getEstimationControlOptions().copyFrom(clonedRunTimeOptions);
    }

    @Override
    protected void prepareSourceModelParameters()
    {
        initializeSourceModelParameters();

        //The number of days is source type specific.
        for(final SourceModelParameters srcPars: getOrderedSourceModelParameters())
        {
            ((MEFPSourceModelParameters)srcPars).setApplicableModel(getModel());

            //Initialize so that the source has no parameters estimated for it.  It will then need to be overridden when reading in 
            //or estimating parameters to set it to be non-zero.
            ((MEFPSourceModelParameters)srcPars).setNumberOfForecastLeadDays(0);
        }

        //This is only useful during operational/hindcasting reading.  If this is called during parameter estimation,
        //The _operationa*Map is not used.
        if(_operationalSourceToLoadEventsMap != null)
        {
            for(final MEFPForecastSource source: _operationalSourceToLoadEventsMap.keySet())
            {
                //The source model parameters can be null IF the source is included/allowed/available from a user defined forecastSourcesDefinition.xml file 
                //BUT not it is not available in the parameter file.
                if(getSourceModelParameters(source) != null)
                {
                    getSourceModelParameters(source).setOperationalEventsWillBeLoaded(_operationalSourceToLoadEventsMap.get(source));
                }
            }
        }
    }

    @Override
    protected void writeAdditionalOutputToTarArchive(final TarArchiveOutputStream stream) throws Exception
    {
        final TimeSeriesArrays ts = TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(getHistoricalTimeSeries());

        //Write the bin file.
        TarTools.addByteEntryToTarArchive(stream,
                                          "historicalTimeSeries.bin",
                                          TimeSeriesArraysTools.writeDataToByteArray(getHistoricalTimeSeries(), false));

        //Now write the XML file, since it is human readable.
        TarTools.addTimeSeriesEntryToTarArchive(stream, "historicalTimeSeries.xml", ts);

        //Write the questionable log.
        TarTools.addXMLEntryToTarArchive(stream, "questionableParameterLog.xml", _questionableParameterLog, false);
    }

    @Override
    protected void readAdditionalInputFromTarArchive(final File tarInputFile,
                                                     final TarArchiveInputStream tarStream,
                                                     final boolean operationalRead) throws Exception
    {
        TimeSeriesArrays ts = null;

        //If the historical time series bin file exists (needed for backward compatibility)
        if(TarTools.isCurrentEntry(tarStream, "historicalTimeSeries.bin"))
        {
            //Read in precipitation historical ts bin file.
            if(this.getIdentifier().isPrecipitationDataType())
            {
                final TimeSeriesArray template = new TimeSeriesArray(ProcessedHistoricalBinaryFileTools.prepareMAPHeader(getIdentifier()));
                ts = TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(TimeSeriesArraysTools.readDataFromBinStream(template,
                                                                                                                                 tarStream));
                if(ts.size() != 1)
                {
                    throw new Exception("Invalid number of precipitation time series found in the historical time sereis binary file: "
                        + ts.size() + "; expected 1");
                }
            }
            //Read in temperature historical ts bin file, and set the second ts to be TMAX -- ASSUMES TMIN IS ALWAYS FIRST!
            else
            {
                final TimeSeriesArray template = new TimeSeriesArray(ProcessedHistoricalBinaryFileTools.prepareTMINHeader(getIdentifier()));
                ts = TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(TimeSeriesArraysTools.readDataFromBinStream(template,
                                                                                                                                 tarStream));
                if(ts.size() != 2)
                {
                    throw new Exception("Invalid number of temperature time series found in the historical time series binary file: "
                        + ts.size() + "; expected 2");
                }
                ((DefaultTimeSeriesHeader)ts.get(TemperatureParameterEstimationModel.TMAX_HISTORICAL_TIME_SERIES_INDEX)
                                            .getHeader()).setParameterId(ParameterId.TMAX.name());
                ((DefaultTimeSeriesHeader)ts.get(TemperatureParameterEstimationModel.TMAX_HISTORICAL_TIME_SERIES_INDEX)
                                            .getHeader()).setParameterName(ParameterId.TMAX.name());
            }
        }

        //If ts is null, then the historical time series bin file did not exist.  Read in the XML file which must always exist.  Note that
        //to read in the XML file, the file must be passed through.  It cannot read time series XML directly from the stream.
        if(ts == null)
        {
            ts = TarTools.readTimeSeriesFromTarFile(tarInputFile, true, "historicalTimeSeries.xml");
            TimeSeriesArraysTools.removeAnnoyingFileDescriptions(ts); //To make unit tests pass and since it does no harm.
        }

        //Set the historical time series appropriately.
        setHistoricalTimeSeries(TimeSeriesArraysTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(ts));

        //Only read in questionable stuff if this is not an operational read.
        if(!operationalRead)
        {
            try
            {
                TarTools.readXMLFromTarFile(tarInputFile,
                                            true,
                                            "questionableParameterLog.xml",
                                            _questionableParameterLog,
                                            false);
            }
            catch(final Exception e)
            {
                LOG.debug("Unable to find entry for the questionable parameter entry log (possibly an older format parameter file); skipping the log.");
            }
        }
    }

    @Override
    public List<Integer> generateDaysOfTheYearForWhichToEstimateParameters()
    {
        return this.getEstimationControlOptions()
                   .getModelParameters()
                   .generateDaysOfTheYearForWhichToEstimateParameters();
    }

    @Override
    public int findNearestComputationalDay(final int desiredDay)
    {
        return this.getEstimationControlOptions().getModelParameters().findNearestComputationalDay(desiredDay);
    }

    @Override
    protected boolean needParametersForSource(final boolean operationalRun, final ForecastSource source)
    {
        return (!operationalRun) || (_operationalSourceToLoadEventsMap.get(source) != null);
    }
}
