package ohd.hseb.hefs.mefp.models.parameters;

import java.io.IOException;
import java.text.DecimalFormat;
import java.util.Arrays;

import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.utils.xml.TableWriterRowSkipChecker;
import ohd.hseb.hefs.utils.xml.TableXMLReader;
import ohd.hseb.hefs.utils.xml.TableXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.io.EndianConvertingInputStream;

/**
 * Extension of OneTypeParameterValues stores the values for a single ModelParameterType in a map of day of year to
 * canonical event to value. The hefsutils XML {@link TableXMLWriter} and {@link TableXMLReader} utilties are used to
 * write the parmameter values for the days of year and events. If
 * {@link OneTypeParameterValues#getDaysOfTheYearForWhichToReadWriteParameters()} returns a non-empty list, then that is
 * used to specify which days are read or written.
 * 
 * @author hank.herr
 */
public class MEFPOneTypeParameterValues extends OneTypeParameterValues
{
    private final double[][] _valuesByDayAndEventIndex;

    /**
     * Used for determining the canonical event corresponding to the second parameter of getValue. Only records
     * canonical events that apply to the data source for which parameters are stored.
     */
    private final CanonicalEventList _usedCanonicalEventList;

    /**
     * @param type {@link ModelParameterType} being stored.
     * @param usedEventList Events for which parameter values must be stored.
     */
    public MEFPOneTypeParameterValues(final ModelParameterType type, final CanonicalEventList usedEventList)
    {
        super(type);
        _usedCanonicalEventList = usedEventList;
        _valuesByDayAndEventIndex = new double[365][_usedCanonicalEventList.size()];
        for(int i = 0; i < _valuesByDayAndEventIndex.length; i++)
        {
            Arrays.fill(_valuesByDayAndEventIndex[i], Double.NaN);
        }
    }

    /**
     * @param type {@link ModelParameterType} being stored.
     * @param fullCanonicalEventList Full list of events, of which only eventCount many will be stored and kept here.
     * @param eventCount The number of events for which parameters will be stored. In most cases, this is not the full
     *            list but some number based on the number of days used for a forecast source.
     */
    public MEFPOneTypeParameterValues(final ModelParameterType type,
                                      final CanonicalEventList fullCanonicalEventList,
                                      final int eventCount)
    {
        this(type, fullCanonicalEventList.subList(eventCount));
    }

    /**
     * @return True if the event is contained within the list of used canonical events, which are those events for which
     *         parameters were computed.
     */
    public boolean wereParametersComputedForEvent(final CanonicalEvent event)
    {
        return _usedCanonicalEventList.contains(event);
    }

    /**
     * @param dayOfYear Starts counting at 1, not zero.
     */
    public double getValue(final Integer dayOfYear, final CanonicalEvent event)
    {
        return getValue(dayOfYear, _usedCanonicalEventList.indexOf(event));
    }

    /**
     * @param dayOfYear Starts counting at 1, not zero.
     */
    public void setValue(final Integer dayOfYear, final CanonicalEvent event, final double value)
    {
        _valuesByDayAndEventIndex[dayOfYear - 1][_usedCanonicalEventList.indexOf(event)] = value;
    }

    /**
     * @param dayOfYear Starts counting at 1, not zero.
     */
    public void readFromFile(final int dayOfYear, final EndianConvertingInputStream stream) throws Exception
    {
        final int count = stream.readIntSwap();
        if(count != 4 * getNumberOfIndicesWithComputedValuesPerDay())
        {
            throw new ParameterIOException("Number of bytes to read, " + count
                + ", must be equal to 4x the number of canonical events with values, "
                + getNumberOfIndicesWithComputedValuesPerDay() + ".");
        }

        for(int i = 0; i < count / 4; i++)
        {
            final float value = readValue(stream);
//CAN BE USED FOR TESTING
//            if((getType() instanceof AvgParameterType)
//                && (getType().getOptionalExtraIdentifier().toString().equalsIgnoreCase("Forecasts")))
//            {
//                System.out.println("####>>     " + dayOfYear + " -- " + i + " -- " + events.get(i).toString() + " = "
//                    + value);
//            }
            setValue(dayOfYear, _usedCanonicalEventList.get(i), value);

        }

        final int endCount = stream.readIntSwap();
        if(count != endCount)
        {
            throw new ParameterIOException("Number of bytes to read before the block, " + count
                + ", did not equal the number at the end of the block, " + endCount + ".");
        }
    }

    /**
     * The value is read in based on the parameter type value type class.
     * 
     * @param stream
     * @return The read in value cast to a float, if not already a float.
     * @throws IOException
     */
    public float readValue(final EndianConvertingInputStream stream) throws IOException
    {
        if(getType().getValueTypeClass() == Integer.class)
        {
            return stream.readIntSwap();
        }
        return stream.readFloatSwap();
    }

    @Override
    public boolean areAllParametersMissingForDay(final int dayOfYear)
    {
        for(int i = 0; i < _valuesByDayAndEventIndex[dayOfYear - 1].length; i++)
        {
            if(!Double.isNaN(_valuesByDayAndEventIndex[dayOfYear - 1][i]))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean areAllParameterValuesMissing()
    {
        for(int i = 0; i < _valuesByDayAndEventIndex.length; i++)
        {
            for(int j = 0; j < _valuesByDayAndEventIndex[i].length; j++)
            {
                if(!Double.isNaN(_valuesByDayAndEventIndex[i][j]))
                {
                    return false;
                }
            }
        }
        return true;
    }

    @Override
    public double getSmallestValue()
    {
        double returnValue = Double.POSITIVE_INFINITY;
        for(int dayIndex = 0; dayIndex < _valuesByDayAndEventIndex.length; dayIndex++)
        {
            for(int eventIndex = 0; eventIndex < _valuesByDayAndEventIndex[dayIndex].length; eventIndex++)
            {
                final double value = _valuesByDayAndEventIndex[dayIndex][eventIndex];
                if(value < returnValue)
                {
                    returnValue = value;
                }
            }
        }
        return returnValue;
    }

    @Override
    public double getLargestValue()
    {
        double returnValue = Double.NEGATIVE_INFINITY;
        for(int dayIndex = 0; dayIndex < _valuesByDayAndEventIndex.length; dayIndex++)
        {
            for(int eventIndex = 0; eventIndex < _valuesByDayAndEventIndex[dayIndex].length; eventIndex++)
            {
                final double value = _valuesByDayAndEventIndex[dayIndex][eventIndex];
                if(value > returnValue)
                {
                    returnValue = value;
                }
            }
        }
        return returnValue;
    }

    @Override
    public double getValue(final int dayOfYear, final int eventIndex)
    {
        return _valuesByDayAndEventIndex[dayOfYear - 1][eventIndex];
    }

    @Override
    public int getNumberOfDaysOfYearWithValues()
    {
        return _valuesByDayAndEventIndex.length;
    }

    @Override
    public int getNumberOfIndicesWithComputedValuesPerDay()
    {
        return _usedCanonicalEventList.size();
    }

    @Override
    public void setValue(final int dayOfYear, final int eventIndex, final double value)
    {
        _valuesByDayAndEventIndex[dayOfYear - 1][eventIndex] = value;
    }

    @Override
    public XMLReader getReader()
    {
        final XMLString typeClass = new XMLString("typeClass");
        final XMLString typeParameterId = new XMLString("typeParameterId");
        final XMLString typeOptionalExtraID = new XMLString("typeExtraID");
        final XMLInteger typeParameterIndex = new XMLInteger("typeParameterIndex");
        final XMLString typeValueTypeClass = new XMLString("typeValueClass");

        //The finalize method makes sure that the read in model parameter type info matches getType()'s return.
        final TableXMLReader reader = new TableXMLReader("parameterValues",
                                                         "dayOfYearValues",
                                                         "eventValue",
                                                         _valuesByDayAndEventIndex)
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                super.finalizeReading();

                //Check the type.
                if(!getType().getClass().getName().equals(typeClass.get()))
                {
                    throw new XMLReaderException("Read in model parameter type class, " + typeClass.get()
                        + ", does not match expected, " + getType().getClass().getName() + ".");
                }
                if(!getType().getParameterId().equals(typeParameterId.get()))
                {
                    throw new XMLReaderException("Read in model parameter type parameter id, " + typeParameterId.get()
                        + ", does not match expected, " + getType().getParameterId() + ".");
                }
                if((getType().getOptionalExtraIdentifier() != null)
                    && (!getType().getOptionalExtraIdentifier().toString().equals(typeOptionalExtraID.get())))
                {
                    throw new XMLReaderException("Read in model parameter type optional extra id, "
                        + typeOptionalExtraID.get() + ", does not match expected, "
                        + getType().getOptionalExtraIdentifier().toString() + ".");
                }
                if(getType().getParameterIndex() != typeParameterIndex.get())
                {
                    throw new XMLReaderException("Read in model parameter type parameter index, "
                        + typeParameterIndex.get() + ", does not match expected, " + getType().getParameterIndex()
                        + ".");
                }
                if(!getType().getValueTypeClass().getName().equals(typeValueTypeClass.get()))
                {
                    throw new XMLReaderException("Read in model parameter type value class, "
                        + typeValueTypeClass.get() + ", does not match expected, "
                        + getType().getValueTypeClass().getName() + ".");
                }
            }
        };

        //Specify the days of the year for reading, if any.
        for(int i = 0; i < this.getDaysOfTheYearForWhichToReadWriteParameters().size(); i++)
        {
            //Subtract one to convert it to a row index from a day of the year.
            reader.addRowToRead(getDaysOfTheYearForWhichToReadWriteParameters().get(i) - 1);
        }

        reader.addAttribute(typeClass, true);
        reader.addAttribute(typeParameterId, true);
        reader.addAttribute(typeOptionalExtraID, false);
        reader.addAttribute(typeParameterIndex, true);
        reader.addAttribute(typeValueTypeClass, true);
        return reader;
    }

    @Override
    public XMLWriter getWriter()
    {
        //Create the table writer and copy the days-of-the-year over to it, shifting them to start counting at 0.
        final TableXMLWriter writer = new TableXMLWriter("parameterValues",
                                                         "dayOfYearValues",
                                                         "eventValue",
                                                         _valuesByDayAndEventIndex);
        if(!getDaysOfTheYearForWhichToReadWriteParameters().isEmpty())
        {
            writer.clearRowsToWrite();
            for(final Integer dayOfYear: getDaysOfTheYearForWhichToReadWriteParameters())
            {
                writer.addRowToWrite(dayOfYear - 1);
            }
        }
        writer.setUseDelimiter(true);
        writer.setNumberFormatter(new DecimalFormat("0.00000"));

        //Do not output empty rows (i.e., where all numbers are NaN) in order to save space.
        writer.setSkipChecker(new TableWriterRowSkipChecker()
        {
            @Override
            public boolean skipRow(final Object oneRow)
            {
                final double[] row = (double[])oneRow;
                for(int i = 0; i < row.length; i++)
                {
                    if(!Double.isNaN(row[i]))
                    {
                        return false;
                    }
                }
                return true;
            }
        });

        writer.addAttribute(new XMLString("typeClass", this.getType().getClass().getName()), true);
        writer.addAttribute(new XMLString("typeParameterId", getType().getParameterId()), true);
        if(getType().getOptionalExtraIdentifier() != null)
        {
            writer.addAttribute(new XMLString("typeExtraID", getType().getOptionalExtraIdentifier().toString()), true);
        }
        writer.addAttribute(new XMLInteger("typeParameterIndex", getType().getParameterIndex()), true);
        writer.addAttribute(new XMLString("typeValueClass", getType().getValueTypeClass().getName()), true);
        return writer;
    }
}
