package ohd.hseb.hefs.mefp.models.parameters;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.models.MEFPParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTRhoParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.RhoParameterType;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.TarTools;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.vars.XMLEnum;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLLong;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.misc.HCalendar;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;

/**
 * Generic MEFP source model parameters can be used directly by forecast sources or can be overridden if needed.
 * Generally, it should only be necessary to override for climatology forecast sources, in order to call
 * {@link #removeDistributionOfFcsts()}.<br>
 * <br>
 * It also includes full methods for I/O both to a large, overall, XML file ({@link #getReader()} and
 * {@link #getWriter()}) and broken down into smaller chunks within a tar ball (
 * {@link #readFromTarArchive(File, boolean)} and {@link #writeToTarArchive(TarArchiveOutputStream, boolean)}. Within a
 * tar archive, the parameters have been broken down as follows:<br>
 * <br>
 * *.meta.xml: Meta information about the estimation of the parameters.<br>
 * <br>
 * *.values.xml: The parameter values themselves.<br>
 * <br>
 * *.[data type].events.xml: The canonical events computed and used in estimating the parameters. The event files are
 * broken down by data type ("precip" or "tmin"/"tmax"), and each files contains both observed and forecast events.<br>
 * <br>
 * Private methods are provided to create the XML readers and writers necessary for each of the three types of files.
 * 
 * @author hank.herr
 */
public class MEFPSourceModelParameters extends SourceModelParameters
{
    /**
     * This attribute includes a finalize that preps the source model parameters for reading based on the number of
     * forecast days. It initializes the parameters mapping and the event value arrays, after determining the number of
     * applicable canonical events.
     */
    private final XMLInteger _numberOfForecastLeadDays = new XMLInteger("numberOfForecastDays", 0, 0, null)
    {
        @Override
        public void finalizeReading() throws XMLReaderException
        {
            super.finalizeReading();
            MEFPSourceModelParameters.this.initializeParametersForUse();
        }
    };

    /**
     * Records the number of canonical events applicable to this source based on {@link #_numberOfForecastLeadDays}.
     */
    private final XMLInteger _numberOfCanonicalEvents = new XMLInteger("numberOfCanonicalEvents", 0, 0, null);

    /**
     * The first time in millis for which an archive forecast, or reforecast, was found, and for which events were
     * calculated and used to estimate parameters. This is not used in the models, but it may be useful from a user
     * point of view, so I'll keep it.
     */
    private final XMLLong _initialTimeOfComputedCanonicalEvents = new XMLLong("initialTimeOfComputedCanonicalEvents",
                                                                              Long.MAX_VALUE);

    /**
     * The last time in millis for which an archive forecast, or reforecast, was found, and for which events were
     * calculated and used to estimate parameters. This is not used in the models, but it may be useful from a user
     * point of view, so I'll keep it.
     */
    private final XMLLong _lastTimeOfComputedCanonicalEvents = new XMLLong("lastTimeOfComputedCanonicalEvents",
                                                                           Long.MAX_VALUE);

    /**
     * This is computed as {@link #_lastTimeOfComputedCanonicalEvents} - {@link #_initialTimeOfComputedCanonicalEvents}
     * + 1. So, it really isn't necessary here, but it will be kept because it is a useful number of know for a user,
     * even if it is not used in the model.<br>
     * <br>
     * Turns out this will be used: If it is positive, then the source had parameters estimated for it. If its 0 or -1,
     * then the value was either never changed or made negative because of no canonical events provided to the
     * {@link #determineTimesOfComputedCanonicalEvents(SourceCanonicalEventValues)}. In either case, parameters could
     * not be estimated. "numberOfCalibrationReforeastDays" is misspelled in the parameter files, so don't change it!
     */
    private final XMLInteger _numberOfCalibrationReforecastDays = new XMLInteger("numberOfCalibrationReforeastDays",
                                                                                 0,
                                                                                 0,
                                                                                 null);

    /**
     * Records the canonical event values.
     */
    private SourceCanonicalEventValues[] _sourceEventValues = null;

    /**
     * Tags to include specifying the data type (e.g., "precipitation", "tmin", "tmax") of data in the XML.
     */
    private String[] _dataTypeTagPrefix = null;

    /**
     * Events for which parameters must be computed.
     */
    private CanonicalEventList _computedEvents;

    /**
     * Flag indicates if only observed parameters should be recorded. If true, parameters are recorded for both forecast
     * and observed. It uses
     * {@link MEFPParameterEstimationModel#getObservationParametersRequiredForModel(ohd.hseb.hefs.pe.sources.ForecastSource)}
     * to determine parameters to expect/store if this flag is true. It uses
     * {@link MEFPParameterEstimationModel#getAllParametersRequiredForModel(ohd.hseb.hefs.pe.sources.ForecastSource)} if
     * this parameter if alse.
     */
    private final boolean _onlyRecordObservedParameters;

    /**
     * Used to acquire the parameters that will be stored.
     */
    private MEFPParameterEstimationModel _applicableModel = null;

    /**
     * Records the distribution used for the observed data; only applicable to precipitation.
     */
    private final XMLEnum<DistributionType> _distributionOfObs = new XMLEnum<DistributionType>(DistributionType.class,
                                                                                               "distributionOfObs",
                                                                                               DistributionType.WEIBULL);

    /**
     * Records the distribution used for the forecast data; only applicable to precipitation. If the source does not
     * include forecasts, such as for historical/climatology, then this variable should be set to null by calling #
     * {@link MEFPSourceModelParameters#removeDistributionOfFcsts()}.
     */
    private XMLEnum<DistributionType> _distributionOfFcsts = new XMLEnum<DistributionType>(DistributionType.class,
                                                                                           "distributionOfFcsts",
                                                                                           DistributionType.WEIBULL);

    /**
     * This flag is only used operationally when reading source parameters. If true, then the events will need to be
     * read in as well as the parameters, because EPT requires events.
     */
    private boolean _operationalEventsWillBeLoaded = false;

    /**
     * Constructor includes all information necessary to initialize and read/write parameters. Note that
     * {@link #setApplicableModel(MEFPParameterEstimationModel)} must be called after the constructor and before any
     * meaningful work with parameters or event values occurs. Otherwise, the storage objects cannot be initialized.
     * 
     * @param identifier Identifier for which parameters are being created.
     * @param source The forecast source.
     * @param algorithmParameters The algorithm model parameters which should already have been established.
     * @param onlyRecordObservedValues True if these parameters only include observed parameters; false to include all
     *            parameters.
     */
    public MEFPSourceModelParameters(final LocationAndDataTypeIdentifier identifier,
                                     final MEFPForecastSource source,
                                     final AlgorithmModelParameters algorithmParameters,
                                     final boolean onlyRecordObservedValues)
    {
        super(identifier, source, algorithmParameters);
        _onlyRecordObservedParameters = onlyRecordObservedValues;
    }

    /**
     * Call to initialize the source model parameters for usage (reading or computing), after the algorithm model
     * parameters have been read. This is called when {@link MEFPFullModelParameters#prepareSourceModelParameters()} is
     * called.
     * 
     * @param numForecastDays The number of forecast days for this source.
     */
    public void setupForStoringParametersAndEvents(final int numForecastDays)
    {
        setNumberOfForecastLeadDays(numForecastDays);
        initializeParametersForUse();
    }

    /**
     * This presumes that the {@link #_numberOfForecastLeadDays} has been set, somehow. It sets up the storage for
     * parameters and canonical events, if {@link #_storeCanonicalEvents} is true.
     */
    private void initializeParametersForUse()
    {
        //Determine the number of applicable canonical events
        setNumberOfCanonicalEvents(getAlgorithmModelParameters().getFullListOfEventsInOrder()
                                                                .determineNumberOfApplicableEvents(getNumberOfForecastLeadDays()));
        _computedEvents = getAlgorithmModelParameters().getFullListOfEventsInOrder()
                                                       .subList(getNumberOfCanonicalEvents());

        //Which model parameter types are used is based on if the source uses observation parameter types only.
        if(_onlyRecordObservedParameters)
        {
            initializeModelParameterValuesStorage(_applicableModel.getObservationParametersRequiredForModel(getForecastSource()));
        }
        else
        {
            initializeModelParameterValuesStorage(_applicableModel.getAllParametersRequiredForModel(getForecastSource()));
        }

        //Initialize the source canonical event values for reading and usage.  Note the XML tag name prefixes passed in.
        if(getIdentifier().isPrecipitationDataType())
        {
            initializeSourceEventValuesStorage(1, new String[]{"precip"});
        }
        else
        {

            initializeSourceEventValuesStorage(2, new String[]{"tmin", "tmax"});
        }
    }

    /**
     * Initializes {@link #_sourceEventValues} for storing values. Note that {@link #_computedEvents} must be set before
     * this is called, and it is set in {@link #initializeModelParameterTypeValues(List, CanonicalEventList)}.
     * 
     * @param numberOfDataTypes The number of data types.
     * @param dataTypeTagPrefixes The prefix to use for each data as an XML tag. E.g.: "precip","tmin","tmax".
     */
    private void initializeSourceEventValuesStorage(final int numberOfDataTypes, final String[] dataTypeTagPrefixes)
    {
        _sourceEventValues = new SourceCanonicalEventValues[numberOfDataTypes];
        _dataTypeTagPrefix = new String[numberOfDataTypes];
        for(int i = 0; i < numberOfDataTypes; i++)
        {
            _sourceEventValues[i] = getForecastSource().constructSourceCanonicalEventValues(_computedEvents);
            _dataTypeTagPrefix[i] = dataTypeTagPrefixes[i];
        }
    }

    /**
     * Calls {@link #addParameterValues(ModelParameterType, ohd.hseb.hefs.pe.model.OneTypeParameterValues)} for each
     * type of parameter.
     * 
     * @param types Types for which parameter values are computed.
     */
    private void initializeModelParameterValuesStorage(final List<ModelParameterType> types)
    {
        for(final ModelParameterType type: types)
        {
            super.addParameterValues(type, new MEFPOneTypeParameterValues(type, _computedEvents));
        }
    }

    /**
     * Calls {@link #determineTimesOfComputedCanonicalEvents(SourceCanonicalEventValues)} passing in first element in
     * {@link #_sourceEventValues}.
     */
    public void determineTimesOfComputedCanonicalEvents()
    {
        determineTimesOfComputedCanonicalEvents(_sourceEventValues[0]);
    }

    /**
     * Sets {@link #_initialTimeOfComputedCanonicalEvents} and {@link #_lastTimeOfComputedCanonicalEvents} based on the
     * provided event values. Call {@link #determineTimesOfComputedCanonicalEvents()} to set it based on
     * {@link #_sourceEventValues} already in this object.
     * 
     * @param values
     */
    public void determineTimesOfComputedCanonicalEvents(final SourceCanonicalEventValues values)
    {
        if(values == null)
        {
            _initialTimeOfComputedCanonicalEvents.set(Long.MAX_VALUE);
            _lastTimeOfComputedCanonicalEvents.set(Long.MAX_VALUE);
            _numberOfCalibrationReforecastDays.set(-1);
        }
        else
        {
            _initialTimeOfComputedCanonicalEvents.set(_sourceEventValues[0].getForecastValues()
                                                                           .getFirstComputationalTime());
            _lastTimeOfComputedCanonicalEvents.set(_sourceEventValues[0].getForecastValues().getLastcomputationalTime());
            _numberOfCalibrationReforecastDays.set((int)((_lastTimeOfComputedCanonicalEvents.get() - _initialTimeOfComputedCanonicalEvents.get()) / (24L * HCalendar.MILLIS_IN_HR)));
        }
    }

    /**
     * This method is what is used to create the component list of {@link XMLVariable} instances to be read and written
     * from and to XML files. It is called as part of the methods {@link #getWriter()}, {@link #getReader()},
     * {@link #getTarReaderForMetaInformation()}, and {@link #getTarWriterForMetaInformation()}. Those methods call this
     * via wrappers {@link #buildListOfMetaInformationXMLReadables()} and
     * {@link #buildListOfMetaInformationXMLWritables()}.<br>
     * <br>
     * Override this method as needed to add or remove components to read from or write to XML files.
     * 
     * @return List of the {@link XMLVariable} instances that contain meta information.
     */
    protected List<XMLVariable> buildListOfMetaInformationXMLComponentVariables()
    {
        final List<XMLVariable> components = new ArrayList<XMLVariable>();
        components.add(_numberOfForecastLeadDays);
        components.add(_numberOfCalibrationReforecastDays);
        components.add(_numberOfCanonicalEvents);
        components.add(_initialTimeOfComputedCanonicalEvents);
        components.add(_lastTimeOfComputedCanonicalEvents);
        if(this.getIdentifier().isPrecipitationDataType())
        {
            components.add(_distributionOfObs);
            if(_distributionOfFcsts != null)
            {
                components.add(_distributionOfFcsts);
            }
        }
        return components;
    }

    /**
     * @return {@link #buildListOfMetaInformationXMLComponentVariables()} converted to {@link XMLReadable}. This is
     *         necessary because the canonical event values will eventually be added to the returned list, and they are
     *         not {@link XMLVariable}s.
     */
    private List<XMLReadable> buildListOfMetaInformationXMLReadables()
    {
        final List<XMLReadable> results = ListTools.convertCollection(buildListOfMetaInformationXMLComponentVariables(),
                                                                      (XMLReadable)null);

        //Read in the events into a dummy list, but then check the results after reading to make sure it is valid.  
        //The best way to do that is to override informChanged, which is called whenever the events are done reading 
        //(during finalization).
        final CanonicalEventList readInComputedEvents = new CanonicalEventList(getIdentifier().isPrecipitationDataType())
        {
            @Override
            public void informChanged()
            {
                super.informChanged();
                //Call getComputedEvents so that, if you want to see _computedEvents when in debug mode, step into this method call.
                if(!this.equals(getComputedEvents()))
                {
                    throw new IllegalStateException("Inconsistent parameters: read in computed canonical events does "
                        + "not equal those determined based on number of forecast days.");
                }
            }
        };
        results.add(readInComputedEvents); //Computed events is not an XMLVariable, so it must be added separately.
        return results;
    }

    /**
     * @return {@link #buildListOfMetaInformationXMLComponentVariables()} converted to {@link XMLWritable}. This is
     *         necessary because the canonical event values will eventually be added to the returned list, and they are
     *         not {@link XMLVariable}s.
     */
    private List<XMLWritable> buildListOfMetaInformationXMLWritables()
    {
        final List<XMLWritable> results = ListTools.convertCollection(buildListOfMetaInformationXMLComponentVariables(),
                                                                      (XMLWritable)null);
        results.add(_computedEvents); //Computed events is not an XMLVariable, so it must be added separately.
        return results;
    }

    public void setApplicableModel(final MEFPParameterEstimationModel applicableModel)
    {
        _applicableModel = applicableModel;
    }

    /**
     * @return The events for which parameters are computed. It is a list sorted according to the standard way to sort
     *         events and containing both modulation and base events.
     */
    public CanonicalEventList getComputedEvents()
    {
        return _computedEvents;
    }

    /**
     * The other wrapped versions of this method should be called: {@link #getTMAXSourceEventValues()},
     * {@link #getTMINSourceEventValues()}, {@link #getTemperatureSourceEventValues(boolean)}, and
     * {@link #getPrecipitationSourceEventValues()}.
     * 
     * @param dataTypeIndex 0 for precipitation or tmin, 1 for tmax.
     * @return The event values.
     */
    public SourceCanonicalEventValues getSourceEventValues(final int dataTypeIndex)
    {
        if(_sourceEventValues == null)
        {
            return null;
        }
        return this._sourceEventValues[dataTypeIndex];
    }

    public SourceCanonicalEventValues getTMINSourceEventValues()
    {
        return getSourceEventValues(0);
    }

    public SourceCanonicalEventValues getTMAXSourceEventValues()
    {
        return getSourceEventValues(1);
    }

    public SourceCanonicalEventValues getPrecipitationSourceEventValues()
    {
        return getSourceEventValues(0);
    }

    public double getPrecipitationRhoParameterValue(final int dayOfYear, final CanonicalEvent event)
    {
        return getParameterValues(new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)).getValue(dayOfYear,
                                                                                                                          event);
    }

    public double getEPTPrecipitationRhoParameterValue(final int dayOfYear, final CanonicalEvent event)
    {
        return getParameterValues(new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)).getValue(dayOfYear,
                                                                                                                             event);
    }

    public double getTMINRhoParameterValue(final int dayOfYear, final CanonicalEvent event)
    {
        return getParameterValues(new RhoParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID)).getValue(dayOfYear, event);
    }

    public double getTMAXRhoParameterValue(final int dayOfYear, final CanonicalEvent event)
    {
        return getParameterValues(new RhoParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID)).getValue(dayOfYear, event);
    }

    public SourceCanonicalEventValues getTemperatureSourceEventValues(final boolean tmaxFlag)
    {
        if(tmaxFlag)
        {
            return getTMAXSourceEventValues();
        }
        return getTMINSourceEventValues();
    }

    public int getNumberOfForecastLeadDays()
    {
        return _numberOfForecastLeadDays.get();
    }

    public void setNumberOfForecastLeadDays(final int days)
    {
        _numberOfForecastLeadDays.set(days);
    }

    public int getNumberOfCalibrationReforecastDays()
    {
        return this._numberOfCalibrationReforecastDays.get();
    }

    public void setNumberOfCalibrationReforecastDays(final int days)
    {
        this._numberOfCalibrationReforecastDays.set(days);
    }

    public int getNumberOfCanonicalEvents()
    {
        return _numberOfCanonicalEvents.get();
    }

    public void setNumberOfCanonicalEvents(final int numberOfCanonicalEvents)
    {
        this._numberOfCanonicalEvents.set(numberOfCanonicalEvents);
    }

    public long getInitialTimeOfComputedCanonicalEvents()
    {
        return _initialTimeOfComputedCanonicalEvents.get();
    }

    public void setInitialTimeOfComputedCanonicalEvents(final long initialTimeOfComputedCanonicalEvents)
    {
        _initialTimeOfComputedCanonicalEvents.set(initialTimeOfComputedCanonicalEvents);
    }

    public long getLastTimeOfComputedCanonicalEvents()
    {
        return _lastTimeOfComputedCanonicalEvents.get();
    }

    public void setLastTimeOfComputedCanonicalEvents(final long lastTimeOfComputedCanonicalEvents)
    {
        _lastTimeOfComputedCanonicalEvents.set(lastTimeOfComputedCanonicalEvents);
    }

    public DistributionType getDistributionOfObs()
    {
        return _distributionOfObs.get();
    }

    public void setDistributionOfObs(final DistributionType distributionOfCFSObs)
    {
        this._distributionOfObs.set(distributionOfCFSObs);
    }

    /**
     * @return The forecast {@link DistributionType}. You may want to call {@link #hasDistributionOfFcstsBeenRemoved()}
     *         prior to using this method to detect if the subclass instance uses that variable.
     * @throws IllegalStateException if the {@link #_distributionOfFcsts} variable has been nulled out via
     *             {@link #removeDistributionOfFcsts()}.
     */
    public DistributionType getDistributionOfFcsts()
    {
        if(_distributionOfFcsts == null)
        {
            throw new IllegalStateException("Attempt to access distribution of forecasts which was removed by a subclass.");
        }
        return _distributionOfFcsts.get();
    }

    /**
     * You may want to call {@link #hasDistributionOfFcstsBeenRemoved()} prior to using this method to detect if the
     * subclass instance uses that variable.
     * 
     * @param distributionOfCFSFcsts What to set the value of the variable to. Null is allowed.
     * @throws IllegalStateException if the {@link #_distributionOfFcsts} variable has been nulled out via
     *             {@link #removeDistributionOfFcsts()}.
     */
    public void setDistributionOfFcsts(final DistributionType distributionOfCFSFcsts)
    {
        if(_distributionOfFcsts == null)
        {
            throw new IllegalStateException("Attempt to access distribution of forecasts which was removed by a subclass.");
        }
        this._distributionOfFcsts.set(distributionOfCFSFcsts);
    }

    /**
     * Removes the {@link #_distributionOfFcsts} variable, nulling it out.
     */
    public void removeDistributionOfFcsts()
    {
        _distributionOfFcsts = null;
    }

    /**
     * @return True if the {@link #_distributionOfFcsts} variable has been nulled out via
     *         {@link #removeDistributionOfFcsts()}.
     */
    public boolean hasDistributionOfFcstsBeenRemoved()
    {
        return (_distributionOfFcsts == null);
    }

    /**
     * Call to specify if events need to be loaded for these parameters when next read.
     */
    public void setOperationalEventsWillBeLoaded(final boolean b)
    {
        _operationalEventsWillBeLoaded = b;
    }

    /**
     * Adds a sourceClass and sourceId XML element to the writer.
     * 
     * @param writer The writer to which to add the attributes.
     */
    private void addAttributesToWriter(final CompositeXMLWriter writer)
    {
        writer.addAttribute(new XMLString("sourceClass", getForecastSource().getClass().getName()), true);
        writer.addAttribute(new XMLString("sourceId", getForecastSource().getSourceId()), false);
    }

    /**
     * @param tagName Tag name of composite XML element to create.
     * @param componentsToAdd Components to add to the element.
     * @param includeParameterValueReadingFromSuperClass If true, then the parameter values, for which XML is created in
     *            {@link SourceModelParameters#getReader()}, will be included in the element at the end.
     * @return A {@link CompositeXMLReader} which includes a sourceClass attribute, which is checked upon reading to
     *         confirm it matches this class.
     */
    private CompositeXMLReader buildBaseReaderWithSourceChecking(final String tagName,
                                                                 final List<XMLReadable> componentsToAdd,
                                                                 final boolean includeParameterValueReadingFromSuperClass)
    {
        //The source class will NOT be initialized with a default value.  It MUST be read from XML and match this source.
        final XMLString sourceClass = new XMLString("sourceClass");
        final XMLString sourceId = new XMLString("sourceId");

        //The base reader must use the componentsToAdd first and check the sourceAttribute to make sure
        //it matches.
        final CompositeXMLReader reader = new CompositeXMLReader(tagName, true, componentsToAdd)
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                //This should only happen for versions equal to or before HEFS 1.2.1.  For those versions, the source
                //id was fixed to the source's class prefix.  Hence, we'll just set it to be fixed here.
                if(sourceId.get() == null)
                {
                    sourceId.set(MEFPSourceModelParameters.this.getForecastSource().getSourceId());
                }
                if(!sourceClass.get().equals(MEFPSourceModelParameters.this.getForecastSource().getClass().getName())
                    || !sourceId.get().equals(MEFPSourceModelParameters.this.getForecastSource().getSourceId()))
                {
                    throw new XMLReaderException("The source data handler class " + sourceClass.get() + " with id "
                        + sourceId.get() + " is not what is expected, "
                        + MEFPSourceModelParameters.this.getForecastSource().getClass().getName() + " with id "
                        + MEFPSourceModelParameters.this.getForecastSource().getSourceId() + ".");
                }
            }
        };
        if(includeParameterValueReadingFromSuperClass)
        {
            reader.addComponents(super.getReader().getComponents());
        }
        reader.addAttribute(sourceClass, true);
        reader.addAttribute(sourceId, false);
        return reader;
    }

    private CompositeXMLWriter getTarWriterForMetaInformation()
    {
        final List<XMLWritable> componentsToAdd = this.buildListOfMetaInformationXMLWritables();
        final CompositeXMLWriter writer = new CompositeXMLWriter("sourceModelParametersInformation", componentsToAdd);
        addAttributesToWriter(writer);
        return writer;
    }

    private CompositeXMLWriter getTarWriterForParameterValues()
    {
        final CompositeXMLWriter baseWriter = super.getWriter();
        final CompositeXMLWriter writer = new CompositeXMLWriter("sourceModelParameterValues",
                                                                 new ArrayList<XMLWritable>());
        writer.addComponents(baseWriter.getComponents());
        addAttributesToWriter(writer);
        return writer;
    }

    private CompositeXMLWriter getTarWriterForEvents(final int index)
    {
        final List<XMLWritable> componentsToAdd = new ArrayList<XMLWritable>();
        componentsToAdd.add(_sourceEventValues[index].getObservationsWriter(this._dataTypeTagPrefix[index]
            + "ObservedValues"));
        componentsToAdd.add(_sourceEventValues[index].getForecastsWriter(this._dataTypeTagPrefix[index]
            + "ForecastValues"));
        final CompositeXMLWriter writer = new CompositeXMLWriter("sourceCanonicalEventValues", componentsToAdd);
        addAttributesToWriter(writer);
        return writer;
    }

    private CompositeXMLReader getTarReaderForMetaInformation()
    {
        final List<XMLReadable> componentsToAdd = this.buildListOfMetaInformationXMLReadables();
        return buildBaseReaderWithSourceChecking("sourceModelParametersInformation", componentsToAdd, false);
    }

    private CompositeXMLReader getTarReaderForParameterValues()
    {
        return buildBaseReaderWithSourceChecking("sourceModelParameterValues", new ArrayList<XMLReadable>(), true);
    }

    /**
     * This assumes that {@link #_sourceEventValues} is not null and index is valid.
     * 
     * @param index
     * @return Reader for the events specified by index.
     */
    private CompositeXMLReader getTarXMLReaderForEvents(final int index)
    {
        final List<XMLReadable> componentsToAdd = new ArrayList<XMLReadable>();
        componentsToAdd.add(_sourceEventValues[index].getObservationsReader(this._dataTypeTagPrefix[index]
            + "ObservedValues"));
        componentsToAdd.add(_sourceEventValues[index].getForecastsReader(this._dataTypeTagPrefix[index]
            + "ForecastValues"));
        return buildBaseReaderWithSourceChecking("sourceCanonicalEventValues", componentsToAdd, false);
    }

    @Override
    public void validateParameters() throws Exception
    {
        throw new Exception("Cannot validate MEFP source parameters yet!"); // Validation needed
    }

    /**
     * @return True if parameters were estimated for the source; false otherwise. This works off the
     *         {@link #getNumberOfCalibrationReforecastDays()}, which must be positive if parameters were successfully
     *         estimated.
     */
    @Override
    public boolean wereParametersEstimatedForSource()
    {
        return getNumberOfCalibrationReforecastDays() > 0;
    }

    @Override
    protected MEFPAlgorithmModelParameters getAlgorithmModelParameters()
    {
        return (MEFPAlgorithmModelParameters)super.getAlgorithmModelParameters();
    }

    @Override
    public MEFPOneTypeParameterValues getParameterValues(final ModelParameterType type)
    {
        return (MEFPOneTypeParameterValues)super.getParameterValues(type);
    }

    @Override
    public MEFPForecastSource getForecastSource()
    {
        return (MEFPForecastSource)super.getForecastSource();
    }

    @Override
    public String getLegendEntryForEnsembleMember(final int memberIndex)
    {
        final CanonicalEventList fullListOfEventsInOrder = (getAlgorithmModelParameters()).getFullListOfEventsInOrder();
        return "Period: " + fullListOfEventsInOrder.get(memberIndex).getStartLeadPeriod() + " - "
            + fullListOfEventsInOrder.get(memberIndex).getEndLeadPeriod();
    }

    @Override
    public int getNumberOfEnsembleMembersForDiagnosticDisplay()
    {
        return getNumberOfCanonicalEvents();
    }

    @Override
    public String getColumnNameForDaysOfYearIndex()
    {
        return "# Days";
    }

    @Override
    public String getColumnNameForEnsembleSizeIndex()
    {
        return "# Events";
    }

    @Override
    public String getLegendTitle()
    {
        if(this.getIdentifier().isPrecipitationDataType())
        {
            return "Canonical Event\n(6h Periods)";
        }
        else
        {
            return "Canonical Event\n(24h Periods)";
        }
    }

    @Override
    public CompositeXMLWriter getWriter()
    {
        final List<XMLWritable> componentsToAdd = this.buildListOfMetaInformationXMLWritables();

        //Add the source canonical event values for all data types if the array is not null, meaning the 
        //initialize method was called.
        if(_sourceEventValues != null)
        {
            for(int i = 0; i < this._sourceEventValues.length; i++)
            {
                componentsToAdd.add(_sourceEventValues[i].getObservationsWriter(this._dataTypeTagPrefix[i]
                    + "ObservedValues"));
                componentsToAdd.add(_sourceEventValues[i].getForecastsWriter(this._dataTypeTagPrefix[i]
                    + "ForecastValues"));
            }
        }

        final CompositeXMLWriter baseWriter = super.getWriter();
        final CompositeXMLWriter writer = new CompositeXMLWriter(baseWriter.getXMLTagName(), componentsToAdd);
        writer.addComponents(baseWriter.getComponents());
        addAttributesToWriter(writer);
        return writer;
    }

    @Override
    public CompositeXMLReader getReader()
    {
        final List<XMLReadable> componentsToAdd = buildListOfMetaInformationXMLReadables();

        //Add the source canonical event values for all data types if the array is not null, meaning the 
        //initialize method was called.
        if(_sourceEventValues != null)
        {
            for(int i = 0; i < this._sourceEventValues.length; i++)
            {
                componentsToAdd.add(_sourceEventValues[i].getObservationsReader(this._dataTypeTagPrefix[i]
                    + "ObservedValues"));
                componentsToAdd.add(_sourceEventValues[i].getForecastsReader(this._dataTypeTagPrefix[i]
                    + "ForecastValues"));
            }
        }

        return buildBaseReaderWithSourceChecking("sourceModelParameters", componentsToAdd, true);
    }

    /**
     * Called to write events to a binary file within the tar file.
     */
    //Performance Enhancement 1: Binary file I/O for events
    private byte[] writeEventsByteArray(final int sourceIndex) throws Exception
    {
        final ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
        final DataOutputStream dataStream = new DataOutputStream(byteStream);
        _sourceEventValues[sourceIndex].getObservedValues().writeToByteStream(dataStream);
        _sourceEventValues[sourceIndex].getForecastValues().writeToByteStream(dataStream);
        return byteStream.toByteArray();
    }

    /**
     * Called to read events from a binary file within the tar file.
     */
    private void readEventsFromBinaryTarEntry(final TarArchiveInputStream tarStream,
                                              final String entryName,
                                              final int sourceIndex) throws Exception
    {
        try
        {
            final TarArchiveEntry inputEntry = TarTools.gotoTarArchiveEntry(tarStream, entryName);
            if(inputEntry == null)
            {
                throw new Exception("Expected entry " + entryName + " was not found.");
            }
            final BufferedInputStream bufStream = new BufferedInputStream(tarStream);
            final DataInputStream dataStream = new DataInputStream(bufStream);
            _sourceEventValues[sourceIndex].getObservedValues().readFromByteStream(dataStream);
            _sourceEventValues[sourceIndex].getForecastValues().readFromByteStream(dataStream);
        }
        catch(final Exception e)
        {
            final Exception relayed = new Exception("Unable to parse entry " + entryName + ": " + e.getMessage());
            relayed.setStackTrace(e.getStackTrace());
            throw relayed;
        }
    }

    /**
     * @return True if the entry file is the name of the first entry for a source.
     */
    @Override
    public String buildFirstSourceParametersFileEntryName()
    {
        return buildParameterFileNamePrefix() + ".meta." + XMLTools.getXMLFileExtension(false);
    }

    /**
     * The flag performanceMode is used to indicate if binary time series file writing will be done. In all cases,
     * parameters will not be written to fastInfoset files which appear to be slower to read through a gzipped tarball
     * than regular XML.
     */
    @Override
    public void writeToTarArchive(final TarArchiveOutputStream outputStream, final boolean performanceMode) throws Exception
    {
        TarTools.addXMLEntryToTarArchive(outputStream,
                                         buildFirstSourceParametersFileEntryName(),
                                         getTarWriterForMetaInformation(),
                                         false);
        TarTools.addXMLEntryToTarArchive(outputStream,
                                         buildParameterFileNamePrefix() + ".values."
                                             + XMLTools.getXMLFileExtension(false),
                                         getTarWriterForParameterValues(),
                                         false);

//See message below in readFromTarArchive.
//        for(int dayOfYear = 1; dayOfYear <= 365; dayOfYear++)
//        {
//            System.err.println("####>> writing -- " + dayOfYear);
//            if(!areAllParametersMissing(dayOfYear))
//            {
//                this.clearDaysOfYearToReadWriteParameters();
//                this.addDayOfYearToReadWriteParameters(dayOfYear);
//                TarTools.addXMLEntryToTarArchive(outputStream, getClass().getSimpleName() + ".values.day" + dayOfYear
//                    + "." + XMLTools.getXMLFileExtension(fastInfoset), getTarWriterForParameterValues(), fastInfoset);
//            }
//        }

        //This should never fail if the parameters are properly estimated.  The _sourceEventValues attribute should
        //always be initialized.
        if(_sourceEventValues != null)
        {
            for(int i = 0; i < this._sourceEventValues.length; i++)
            {
                if(!performanceMode)
                {
                    TarTools.addXMLEntryToTarArchive(outputStream,
                                                     buildParameterFileNamePrefix() + "." + this._dataTypeTagPrefix[i]
                                                         + ".events." + XMLTools.getXMLFileExtension(false),
                                                     getTarWriterForEvents(i),
                                                     false);
                }
                else
                {
                    //Performance improvement 1: Write the events to a binary file.
                    TarTools.addByteEntryToTarArchive(outputStream, buildParameterFileNamePrefix() + "."
                        + this._dataTypeTagPrefix[i] + ".events.bin", writeEventsByteArray(i));
                }
            }
        }
    }

    /**
     * @return A prefix string to add to relevant parameter files names. It is the source id plus
     *         'SourceModelParameters'.
     */
    private String buildParameterFileNamePrefix()
    {
        return this.getForecastSource().getSourceId() + "SourceModelParameters";
    }

    /**
     * The flag performanceMode is used to indicate if binary time series file writing was done. This method assumes
     * that the entries in the stream are in the correct order. It also assumes that the current entry is the first
     * entry associated with this source. If that entry is NOT the meta file entry, then it assumes that this source is
     * newer than the parameter file, meaning that parameters could not be estimated so that its number of lead days
     * should be 0.<br>
     * <br>
     * This method must be called with the current entry set to the meta file entry.<br>
     * <br>
     * NOTE: The meta file is ALWAYS read regardless of the skipSourceRead flag.
     */
    @Override
    public void readFromTarArchive(final TarArchiveInputStream tarStream,
                                   final boolean operationalRead,
                                   final boolean skipSourceRead) throws Exception
    {
        //If the meta file entry is not present, then we assume two possibilities.  (a) the source is an old source,
        //meaning the parameters are old.  In that case, skip the files associated with current parameter file source
        //and check the next to match this instead.  (b) the source is a new source and the parameter file is old.  In
        //that case, set the number of days for the source to be 0 and leave this method so that the next source is
        //read.  
        //TODO Different tact now... Source is not know when this is read and is determined from the content of the file.
//        final String prefix = buildParameterFileNamePrefix();
//        final String entryName = prefix + ".meta." + XMLTools.getXMLFileExtension(false);
//        while(!TarTools.isCurrentEntry(tarStream, entryName))
//        {
//            String currentPrefix = ((OHDWrapperTarArchiveInputStream)tarStream).getOHDCurrentEntry().getName();
//            currentPrefix = currentPrefix.substring(0, currentPrefix.indexOf("."));
//
//            //Source in parameter file has been removed in new version of code.  Skip the set of files and
//            //try again.
//            if(MEFPTools.isOldSourceParameterFilePrefix(currentPrefix))
//            {
//                TarTools.gotoNextEntry(tarStream);
//            }
//            //This source is too new for the parameter files.  Set this source to not be used (i.e., number
//            //of forecast days to be 0) and leave this method so that the next source is tried.
//            else
//            {
//                setNumberOfForecastLeadDays(0);
//                return;
//            }
//        }
        final String entryName = TarTools.getCurrentEntryName(tarStream);
        if(!entryName.endsWith(".meta." + XMLTools.getXMLFileExtension(false)))
        {
            throw new Exception("Expected a tar file entry ending with '" + ".meta."
                + XMLTools.getXMLFileExtension(false) + "', but found the entry " + entryName);
        }

        //Start reading... Always read the meta information regardless of the skipSourceRead method.  Some necessary
        //structure are put in place during this process.  If the source is not yet set, this will put it in place.
        TarTools.readXMLFromTarStream(tarStream, true, entryName, getTarReaderForMetaInformation(), false);

        //There is always one values file for each source.  Either read it or skip it appropriately.
        if(!skipSourceRead)
        {
            TarTools.readXMLFromTarStream(tarStream,
                                          true,
                                          buildParameterFileNamePrefix() + ".values."
                                              + XMLTools.getXMLFileExtension(false),
                                          getTarReaderForParameterValues(),
                                          false);
        }
        else
        {
            TarTools.gotoNextEntry(tarStream);
        }

//XXX The following will read data for just the day of the year parameter value file.  This has proven to be not much better than reading
//the full size parameter file, but only extracting one day.  I guess the number of files in a tar ball affects read speed.  
//Note that because tar navigating allows forward movement but not backward, we need to step through each entry and check the names
//one by one and we have to assume there are a specific number of files in order for the source event values reading to work.
//        for(int dayOfYear = 1; dayOfYear <= 365; dayOfYear++)
//        {
//            final String fileName = getClass().getSimpleName() + ".values.day" + dayOfYear + "."
//                + XMLTools.getXMLFileExtension(fastInfoset);
//            final TarArchiveEntry entry = tarStream.getNextTarEntry();
//            if(!entry.getName().equals(fileName))
//            {
//                throw new Exception("Unexpected entry, " + entry.getName() + ", when " + fileName + " is expected.");
//            }
//            if((this.getDaysOfTheYearForWhichToReadWriteParameters().isEmpty())
//                || (getDaysOfTheYearForWhichToReadWriteParameters().contains(dayOfYear)))
//            {
//                this.clearDaysOfYearToReadWriteParameters();
//                this.addDayOfYearToReadWriteParameters(dayOfYear);
//                try
//                {
//                    XMLTools.readXMLFromStream(tarStream, fastInfoset, getTarReaderForParameterValues());
//                }
//                catch(final Exception e)
//                {
//                    final Exception relayed = new Exception("Unable to parse entry " + entry.getName() + ": "
//                        + e.getMessage());
//                    relayed.setStackTrace(e.getStackTrace());
//                    throw relayed;
//                }
//            }
//        }

        //Read in the source event values only if the storage structure is initialized for reading. 
        //Note that the files for all sources must always exist in the tarball, even if the source was not estimated
        //(i.e., number of days was 0 or it was disabled via checkbox).
        if(_sourceEventValues != null)
        {
            for(int i = 0; i < this._sourceEventValues.length; i++)
            {
                //Events are read if (1) the source is NOT skipped and (2) either this is not an operational run or,
                //if it is, the _operationalEventWillBeLoaded flag is true.
                if((!skipSourceRead) && (!operationalRead || (operationalRead && this._operationalEventsWillBeLoaded)))
                {
                    //There is only one event file, but it may be an xml or bin file.
                    final TarArchiveEntry entry = tarStream.getNextTarEntry();

                    //Parms were written in regular mode (XML)...
                    if(entry.getName().equals(buildParameterFileNamePrefix() + "." + this._dataTypeTagPrefix[i]
                        + ".events." + XMLTools.getXMLFileExtension(false)))
                    {
                        TarTools.readXMLFromTarStream(tarStream,
                                                      true,
                                                      entry.getName(),
                                                      getTarXMLReaderForEvents(i),
                                                      false);
                    }
                    //Parms were written in performance mode (bin)...
                    else if(entry.getName().equals(buildParameterFileNamePrefix() + "." + this._dataTypeTagPrefix[i]
                        + ".events.bin"))
                    {
                        readEventsFromBinaryTarEntry(tarStream, entry.getName(), i);
                    }

                    //Unexpected file...
                    else
                    {
                        throw new Exception("Expected either a bin or xml events entry for "
                            + getClass().getSimpleName() + " but found " + entry.getName());
                    }
                }
                else
                {
                    //Skip the event file.
                    TarTools.gotoNextEntry(tarStream);
                }
            }
        }

        //Move on to the next entry prepping it for reading.  Not sure if this is needed, but it seems to work.
        TarTools.gotoNextEntry(tarStream);

    }
}
