package ohd.hseb.hefs.mefp.models.precipitation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ohd.hseb.hefs.mefp.models.ForecastEnsembleCalculator;
import ohd.hseb.hefs.mefp.models.MEFPModelTools;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.dist.types.NormalDist;
import ohd.hseb.hefs.utils.tools.SamplingTools;
import ohd.hseb.hefs.utils.tools.ThreadedRandomTools;

/**
 * The code below is not properly, independently unit tested. Rather, it is tested via the
 * {@link PrecipitationEnsembleGenerationModelTest}. The reason is because it is too difficult to gather the parameters
 * correctly for unit testing of this model. Hence, that framework serves primarily as a data gather mechanism. We then
 * turn on the EPT option for RFC and GFS, use stratified sampling instead of random sampling, and compare the forecast
 * ensembles generated here with those created via the Fortran/C code. If all cases match, we're set.
 * 
 * @author hankherr
 */
public class EPTPrecipitationForecastEnsembleCalculator implements ForecastEnsembleCalculator
{
    private static final Logger LOG = LogManager.getLogger(EPTPrecipitationForecastEnsembleCalculator.class);

//    private static final int MAX_NUM_SAMPLE_POINTS = 365 * 4 * 70;

    /**
     * In the Fortran code, the minimum sample size for ensemble generation, here, is the not the same for parameter
     * estimation (7).
     */
    private static final int MIN_SAMPLE_SIZE_1 = 14;
    private static final int MIN_SAMPLE_SIZE_2 = 5; // A size that is small yet won't break down parameter estimation. LW

    private final MEFPForecastSource _source;
    private final MEFPFullModelParameters _fullParameters;
    private final int _dayOfYear;

    //private final boolean _stratifiedSampling;
    private final SamplingTools.SamplingTechnique _stratifiedSampling;
    private final Double _thresholdToIncludeSamplingLogMessages;
    private final DistributionType _distributionType; //d_type
    private final double _noRainThreshold; //ptresh

    //XXX this was used to specify the optimization flag
    //private final int _optimizationFlag = 0; //adj

    private final StandardCanonicalEventValuesGatherer _eventGatherer;

    /**
     * Only call this during testing!!!
     * 
     * @param type
     * @param noRainThreshold
     */
    protected EPTPrecipitationForecastEnsembleCalculator(final DistributionType type,
                                                         final double noRainThreshold,
                                                         final SamplingTools.SamplingTechnique stratifiedSampling)                                                         
    {
        _source = null;
        _fullParameters = null;
        _dayOfYear = -1;
        _noRainThreshold = noRainThreshold;
        _distributionType = type;
        _eventGatherer = null;
        _stratifiedSampling = stratifiedSampling;
        _thresholdToIncludeSamplingLogMessages = null;
    }

    /**
     * @param source The forecast source for which to generate an ensemble.
     * @param fullParameters The model parameters.
     * @param dayOfYear The day of the year for which to generate an ensemble.
     * @param eventValues The reforecast canonical event values.
     */
    public EPTPrecipitationForecastEnsembleCalculator(final MEFPForecastSource source,
                                                      final MEFPFullModelParameters fullParameters,
                                                      final int dayOfYear,
                                                      final SourceCanonicalEventValues eventValues,                                                      
                                                      final SamplingTools.SamplingTechnique stratifiedSampling,
                                                      final Double thresholdToIncludeSamplingLogMessages)
    {
        _source = source;
        _fullParameters = fullParameters;
        _dayOfYear = dayOfYear;
        _stratifiedSampling = stratifiedSampling;
        _thresholdToIncludeSamplingLogMessages = thresholdToIncludeSamplingLogMessages;

        //The control parameters must be used to construct the canonical event values gatherer.  The other option is to put all 7+ needed control options
        //in the source model parameters and refer to them, but that seems like a lot of redundancy for little gain.  
        final MEFPPrecipitationModelControlOptions controlOptions =
                                                                  (MEFPPrecipitationModelControlOptions)fullParameters.getEstimationControlOptions()
                                                                                                                      .getModelParameters();

        //Standard mechanism for building an eptGatherer.  Note that this will gather more events than is necessary, but it should add a negligible 
        //amount of computation time.  The gathering is done in the exact same manner as for parameeter estimation.
        _eventGatherer =
                       MEFPModelTools.constructEPTCanonicalEventValuesGatherer(source,
                                                                               eventValues,
                                                                               controlOptions,
                                                                               fullParameters.getEstimationControlOptions()
                                                                                             .getSourceControlOptions(_source));

        //Gather the events for the day of the year specified in the parameters all at once.
        _eventGatherer.gatherEventValues(dayOfYear);

        //Pull options that were used in estimation from the control parameters stored in the MEFPFullModelParamters.
        _distributionType = controlOptions.getEPT().getEPTDistribution();
        _noRainThreshold = controlOptions.getEPT().getEPTPrecipThreshold();
//XXX Removing for now        _optimizationFlag = controlOptions.getEPT().getEPTOptimizationOption();
    }

    /**
     * Calls
     * {@link #calculateForecastEnsemble(CanonicalEvent, double, int, PrecipitationOneSetParameterValues, List, List)}.
     * It determines the list of forecast and observed canonical events based on the specified event. It also loads the
     * {@link PrecipitationOneSetParameterValues} to used based on the event and the known day of the year.
     * 
     * @param evt {@link CanonicalEvent} for which to generate a forecast ensemble.
     * @param canonicalEventValue The event's computed value.
     * @param numberOfEnsembleMembers The number of members to generate.
     * @return An array specifying the forecast ensemble.
     * @throws Exception
     */
    @Override
    public double[] calculateForecastEnsemble(final CanonicalEvent event,
                                              final double canonicalEventValue,
                                              final int numberOfEnsembleMembers) throws Exception
    {
        final PrecipitationOneSetParameterValues parametersForEvt =
                                                                  (PrecipitationOneSetParameterValues)_fullParameters.getSourceEstimatedModelParameterValues(_source,
                                                                                                                                                             _dayOfYear,
                                                                                                                                                             event);

        //Get the events and do some basic checks.
        final List<Float> forecastEventValues = _eventGatherer.getForecastGatheredEventValues(event); //f_sample
        final List<Float> observedEventValues = _eventGatherer.getObservedGatheredEventValues(event); //o_sample
        if(forecastEventValues.isEmpty() || observedEventValues.isEmpty())
        {
            throw new Exception("No canonical event values were found for event " + event.toString() + ".");
        }

        if(forecastEventValues.size() != observedEventValues.size())
        {
            throw new IllegalStateException("For event " + event.toString()
                + ", the number of forecast events is not identical to the number of observed events ("
                + forecastEventValues.size() + " != " + observedEventValues.size() + ").");
        }

        return calculateForecastEnsemble(event,
                                         canonicalEventValue,
                                         numberOfEnsembleMembers,
                                         parametersForEvt,
                                         forecastEventValues,
                                         observedEventValues);
    }

    /**
     * @param event The canonical event currently being processed. If this is null, then
     *            {@link #_thresholdToIncludeSamplingLogMessages} better be null!
     * @param canonicalEventValue The event's computed value.
     * @param numberOfEnsembleMembers The number of members to generate.
     * @param parametersForEvt The {@link PrecipitationOneSetParameterValues} specifying the parameters to use for
     *            computations.
     * @param forecastEventValues The forecast event values used to estimate distributions.
     * @param observedEventValues The observed event values used to estimate distributions.
     * @return An array specifying the forecast ensemble.
     * @throws Exception
     */
    protected double[] calculateForecastEnsemble(final CanonicalEvent event,
                                                 final double canonicalEventValue,
                                                 final int numberOfEnsembleMembers,
                                                 final PrecipitationOneSetParameterValues parametersForEvt,
                                                 final List<Float> forecastEventValues,
                                                 final List<Float> observedEventValues) throws Exception
    {
//XXX Write all necessary information to run this method with parameters as given to files for JUnit test purposes.  
//Be sure to specify the appropriate scenarioName in the files below.
//        final Properties props = new Properties();
//        props.put("noRainThreshold", "" + _noRainThreshold);
//        props.put("distributionType", _distributionType.toString());
//        props.put("numberOfEnsembleMembers", "" + numberOfEnsembleMembers);
//        props.put("canonicalEventValue", "" + canonicalEventValue);
//        props.store(new FileOutputStream(new File("testdata/eptPrecipitationForecastEnsembleCalculator/test2_in.param.txt")),
//                    "Parameters of EPTPrecipitationForecastEnsembleCalculator");
//        NumberTools.writeNumbersToFile(new File("testdata/eptPrecipitationForecastEnsembleCalculator/test2_in.fcstvalues.txt"),
//                                       forecastEventValues);
//        NumberTools.writeNumbersToFile(new File("testdata/eptPrecipitationForecastEnsembleCalculator/test2_in.obsvalues.txt"),
//                                       observedEventValues);
//        NumberTools.writeNumbersToFile(new File("testdata/eptPrecipitationForecastEnsembleCalculator/test2_in.onesetvalues.txt"),
//                                       parametersForEvt.getValues());

        final double[] ensemble = new double[numberOfEnsembleMembers];
        Arrays.fill(ensemble, 0d);

        //In fortran this was done during the loop to populate the lists.  However, if it is already in the parameters,
        //why compute it again?
        final double p00 = parametersForEvt.getEPTBivarProbNoPrecip();
        final double p10 = parametersForEvt.getEPTFcstPoPObsZero();
        final double p01 = parametersForEvt.getEPTObsPopFcstZero();
        final double p11 = 1d - p00 - p01 - p10;

        // If the source is climatology, create an ensemble from the distribution of the observations. LW
        if(((MEFPForecastSource)parametersForEvt.getForecastSource()).isClimatologySource())
        {
            final int numberOfPositiveMembers = (int)Math.round(numberOfEnsembleMembers * (1.0d - p00));
            final ContinuousDist sDist = parametersForEvt.getEPTObsBivarMarginalDist(_distributionType);
            final List<Double> quaValues = new ArrayList<Double>();

            for(int i = 0; i < numberOfPositiveMembers; i++)
            {
                double quaValue;
                quaValue =  SamplingTools.generateSample(sDist,numberOfPositiveMembers , i, _thresholdToIncludeSamplingLogMessages,
                        "EPT Algorithm for event " + event
                        + " and resampled climatology: Generating random sample for non-zero"
                        + " precipitation amount from distribution "
                        + sDist + "... ", _stratifiedSampling);
//                if(_stratifiedSampling)
//                {
//                    // Stratified sampling with the Weibull plotting position:
//                    // LW pValue = (double)(i + 1) / (double)(numberOfPositiveMembers + 1);
//                    quaValue =
//                             SamplingTools.generateStratifiedRandomSample(sDist,
//                                                                                numberOfPositiveMembers,
//                                                                                i,
//                                                                                _thresholdToIncludeSamplingLogMessages,
//                                                                                "EPT Algorithm for event "
//                                                                                    + event
//                                                                                    + " and resampled climatology: Generating stratified random sample "
//                                                                                    + "for non-zero precipitation amount from distribution "
//                                                                                    + sDist + "... ");
//                }
//                else
//                {
//                    quaValue =
//                             SamplingTools.generateRandomSample(sDist,
//                                                                      _thresholdToIncludeSamplingLogMessages,
//                                                                      "EPT Algorithm for event " + event
//                                                                          + " and resampled climatology: Generating random sample for non-zero"
//                                                                          + " precipitation amount from distribution "
//                                                                          + sDist + "... ");
//                }

                // The quantiles computed here should be no less than _noRainThreshold in theory. If they are, 
                // set them to _noRainThreshold. LW
                if(Math.abs(quaValue) < _noRainThreshold)
                {
                    quaValue = _noRainThreshold;
                }

                quaValues.add(quaValue);
            }

            this.sortAndPutValuesAtEndOfArray(ensemble, quaValues);

            return ensemble;
        }

        //Do the counting and populate the data arrays.
        int count0 = 0;
        final List<Float> x0_y1 = new ArrayList<Float>(forecastEventValues.size());
        final List<Float> x1_y0 = new ArrayList<Float>(forecastEventValues.size());
        final List<Float> r = new ArrayList<Float>(forecastEventValues.size());
        final List<Float> s = new ArrayList<Float>(forecastEventValues.size());
        count0 = MEFPModelTools.constructPrecipitationRainNoRainLists(forecastEventValues,
                                                                      observedEventValues,
                                                                      _noRainThreshold,
                                                                      x0_y1,
                                                                      x1_y0,
                                                                      r,
                                                                      s);

        //XXX This is necessary to match Fortran, but should not be necessary in Java since both param est and ens gen call the same method to 
        //gather canonical event values and put them into rain/norain lists. /        p00 = (double)count0 / (double)forecastEventValues.size();
//        final double p10x = (double)x1_y0.size() / (double)forecastEventValues.size();
//        final double p01x = (double)x0_y1.size() / (double)forecastEventValues.size();
//        final double p11x = (double)r.size() / (double)forecastEventValues.size();

        //Sort x0_y1 if stratified sampling is to be done. Sorted x0_y1 will be used in method samplePositiveValues.
        //Also sort s.
        switch(_stratifiedSampling) 
        { 
            case STRATIFIED_RANDOM:
            case QUANTILE: 
                Collections.sort(x0_y1);
                if(s.size() < MIN_SAMPLE_SIZE_1)
                    Collections.sort(s);
                break;
        }
//        if(_stratifiedSampling == SamplingTools.SamplingTechnique.QUANTILE.STRATIFIED_RANDOM || _stratifiedSampling== SamplingTools.SamplingTechnique.QUANTILE) //add quantily
//        {
//            Collections.sort(x0_y1);
//            if(s.size() < MIN_SAMPLE_SIZE_1)
//                Collections.sort(s);
//        }

        double qpf = canonicalEventValue;
        //This sets the qpf value and, if there are no cases where the forecast event value is zero, 
        //forces it to be the _precipThresh.  This means that it cannot go into any if-statement where qpf < _precipThresh.
        if(count0 + x0_y1.size() == 0)
        {
            if(qpf < _noRainThreshold)
            {
                LOG.debug("Triggered clause indicating that a 0-forecast event was provided, but no such event "
                    + "was found during parameter estimation; setting event to be minimally positive.");
                qpf = _noRainThreshold;
            }
        }

        //If there are no cases during parameter estimation where the forecast event value is greater than the threshold, 
        //forces qpf to be zero. LW
        if(r.size() + x1_y0.size() == 0)
        {
            if(qpf >= _noRainThreshold)
            {
                LOG.debug("Triggered clause indicating that a positive-forecast event was provided, but no such event "
                    + "was found during parameter estimation; setting event to be zero.");
                qpf = 0.0d;
            }
        }

        //Generate an ensemble if qpf < threshold. 
        if(qpf < _noRainThreshold)
        {
            //NOTE that if x0_y1 is empty, that means whenever the forecast is 0, the observed is also zero.  Hence
            //we leave the ensemble array to remain all 0 (it was filled with zeros).
            if(!x0_y1.isEmpty())
            {
                final double tmp = p00 / (p00 + p01); //The conditional probability of no rain.
                final int numberOfPositiveMembers = (int)Math.round(numberOfEnsembleMembers * (1.0d - tmp));

                if(numberOfPositiveMembers > 0)
                {
                    final List<Double> quaValues =
                                                 SamplingTools.sampleFromValues(x0_y1,
                                                                                      numberOfPositiveMembers,
                                                                                      _stratifiedSampling,
                                                                                      _thresholdToIncludeSamplingLogMessages,
                                                                                      "EPT Algorithm for event "
                                                                                          + event
                                                                                          + ": Generating stratified random sample from historcal values to identify non-zero precipitation when QPF indicates no precip; "
                                                                                          + "i.e., QPF < no-rain threshold (no parmetric distribution)... ");
                    sortAndPutValuesAtEndOfArray(ensemble, quaValues);
                }
            }
        }

        //Generate an ensemble if qpf > threshold, but we don't have enough cases where they are both zero
        //to go through the standard process.
        else if(r.size() < MIN_SAMPLE_SIZE_1)
        {
            // LW: If the execution gets to this point, then r.size() will not be zero. See the change made to the following 
            // case (commented out) a few lines above.

            //if(r.size() + x1_y0.size() == 0) 
            //{
            //    throw new Exception("There were no cases where the forecast was positive in the past event data.");
            //}

            // LW final double tmp = p11 / (p11 + p10); //The conditional probability of no rain.
            // LW final int numberOfPositiveMembers = (int)Math.round(numberOfEnsembleMembers * (1.0d - tmp));
            final double tmp = p11 / (p11 + p10); //The conditional probability of rain.
            final int numberOfPositiveMembers = (int)Math.round(numberOfEnsembleMembers * tmp);

            final List<Double> quaValues =
                                         SamplingTools.sampleFromValues(s,
                                                                              numberOfPositiveMembers,
                                                                              _stratifiedSampling,
                                                                              _thresholdToIncludeSamplingLogMessages,
                                                                              "EPT Algorithm for event "
                                                                                  + event
                                                                                  + ": Generating stratified random sample to identify non-zero precipitation when QPF > no-rain threshold "
                                                                                  + "but there are not enough samples where observed and forecast are positive; that distribution is "
                                                                                  + s + "...");
            sortAndPutValuesAtEndOfArray(ensemble, quaValues);
        }

        else
        {
            int numberOfPositiveMembers = ensemble.length;
            double bx1 = 0.0d;
            double bx2 = 0.0d;
            double by1 = 0.0d;
            double by2 = 0.0d;
            boolean isStrictEPT = true;

            //If there enough samples where the forecast is positive but the obs is 0, then use a Pearson Type-III
            //distribution (Gamma with a shift) with the estimated dist parameters to compute a value dx used to
            //determine the number of positive ensemble members.
            if(x1_y0.size() >= MIN_SAMPLE_SIZE_2)
            {
//XXX Removed for now... see comment below          
//                validateAndFixDistributionParameters(parametersForEvt);
                final ContinuousDist x1y0Dist = parametersForEvt.getEPTObsZeroFcstDist(DistributionType.GAMMA);
                final double x1y0PDFofQPF = x1y0Dist.functionPDF(qpf);
                final ContinuousDist rDist = parametersForEvt.getEPTFcstIntermittencyDist(DistributionType.GAMMA);
                final double rPDFofQPF = rDist.functionPDF(qpf);

//                System.err.println("####>> values -- " + qpf + ", " + p10 + ", " + x1y0PDFofQPF + ", " + p11 + ", "
//                    + rPDFofQPF + ", " + p01);

                if(x1y0PDFofQPF > 1000.0d || Double.isNaN(x1y0PDFofQPF))
                {
                    isStrictEPT = false;
                    bx1 = p00 + p01;
                    bx2 = 1 - bx1;
                    by1 = p00 + p10;
                    by2 = 1 - by1;
                }
                else
                {
                    // Use the following scheme to ensure that numberOfPositiveMembers is unbiased in the long run:  
                    // 1. Get f=floor(n*pop) and c=ceiling(n*pop), where n is the ensemble size and pop is the PoP for that ensemble.
                    // 2. We have (a*f+b*c)/n=pop, and here a+b=1.
                    // 3. Solve the above for a, we get a = c-n*pop, assuming c-f = 1.
                    // 4. Generate a random number from U(0,1) and use a to decide whether we have a f or c.

                    final double probabilityOfPrecipitation = p11 * rPDFofQPF / (p10 * x1y0PDFofQPF + p11 * rPDFofQPF);
                    final double theoreticalSizeOfPositiveEnsembleMembers = probabilityOfPrecipitation
                        * numberOfEnsembleMembers;
                    final int floorSize = (int)Math.floor(theoreticalSizeOfPositiveEnsembleMembers);
                    final int ceilingSize = (int)Math.ceil(theoreticalSizeOfPositiveEnsembleMembers);

                    if(floorSize == ceilingSize)
                    {
                        numberOfPositiveMembers = floorSize;
                    }
                    else
                    {
                        final double probabilityOfSelectingFloorSize = ceilingSize
                            - theoreticalSizeOfPositiveEnsembleMembers;
                        numberOfPositiveMembers =
                                                ThreadedRandomTools.selectBetweenTwoValues(floorSize,
                                                                                           ceilingSize,
                                                                                           probabilityOfSelectingFloorSize);

                        // numberOfPositiveMembers = (int)Math.round(numberOfEnsembleMembers * dx);
                    }

                }
            }
            else
            {
                // The transition case for which the size of x1_y0 is small. For this case, the mixed-type meta-Gaussian
                // model is assumed to be close to the straight meta-Gaussian model, so a slightly different treatment 
                // is needed.  LW
                isStrictEPT = false;
                bx1 = p00 + p01;
                bx2 = 1 - bx1;
                by1 = p00 + p10;
                by2 = 1 - by1;
            }

            final ContinuousDist rDist = parametersForEvt.getEPTFcstBivarMarginalDist(_distributionType);
            final ContinuousDist sDist = parametersForEvt.getEPTObsBivarMarginalDist(_distributionType);
            final double rho = parametersForEvt.getEPTRho();

            //DJ's change: Record the conditional bias penalty weight, or alpha, based on the parameter mu.  
            //Compute lambda.
            double alpha = parametersForEvt.getEPTXmin();
            if (Double.isNaN(alpha))
            {
                alpha = 0; //Default is alpha = 0, which is equivalent to standard EPT algorithm.
            }
            final double lambda = ((1.0 + alpha)*rho)/(1 + alpha * rho * rho);

            double z_qpf;
            if(isStrictEPT)
            {
                z_qpf = NormalDist.STD_NORM_DIST.functionInverseCDF(rDist.functionCDF(qpf));
            }
            else
            {
                z_qpf = NormalDist.STD_NORM_DIST.functionInverseCDF(bx1 + bx2 * rDist.functionCDF(qpf));
            }

            //These values are being carried over from the C code in case we decide to use the optimization for parameter estimation.
//            final double tau1 = parametersForEvt.getEPTXmin();
//            final double mu1 = parametersForEvt.getEPTMu();
//            final double sig1 = parametersForEvt.getEPTSig();

            
            //Generate a numberOfPositiveMembers many normal deviates.
            final List<Double> quaValues = new ArrayList<Double>();
            for(int i = 0; i < numberOfPositiveMembers; i++)
            {
                double zValue =0.0d;
    	        switch(_stratifiedSampling) 
    	        { 
    	            case STRATIFIED_RANDOM:
    	            case QUANTILE: 
    	                zValue = SamplingTools.generateSample(NormalDist.STD_NORM_DIST, numberOfPositiveMembers , i, null, null,_stratifiedSampling);
    	                break;
    	            case RANDOM:
    	            	zValue = ThreadedRandomTools.getRandomNumGen().nextGaussian();
    	        }
//                if(_stratifiedSampling)// Note of 
//                {
//                    // Stratified sampling with the Weibull plotting position:
//                    // LW final double prob = (double)(i + 1) / (double)(numberOfPositiveMembers + 1);
//                    zValue = SamplingTools.generateStratifiedRandomSample(NormalDist.STD_NORM_DIST,
//                                                                                numberOfPositiveMembers,
//                                                                                i);
//                }
//                else
//                {
//                    zValue = ThreadedRandomTools.getRandomNumGen().nextGaussian();
//                }
                double w = Double.NaN;

                //This equation is pulled from the if-class commented out below assuming _optimizationFlag = 0.
                //w = rho * z_qpf + Math.sqrt(1 - Math.pow(rho, 2)) * zValue;
                w = lambda * z_qpf + Math.sqrt(1 + Math.pow(lambda, 2) - 2*lambda*rho) * zValue;

//XXX This code used to use the optimization flag:
                //Without optimization, the w values is a conditional normal using rho, z_qpf, and the random number.
//                if(_optimizationFlag == 0)
//                {
//                    w = rho * z_qpf + Math.sqrt(1 - Math.pow(rho, 2)) * zValue;
//                }
//                else
//                //I don't think the optimization mode is ever turned on in the parameter estimator for now.  
//                //If the optimization (iadj) flag is available, then the next part should be used combined with the
//                //complex optimization stuff in the pe portion.
//                {
//                    //tau1 is being set based on Xmin, which may be a poorly chosen variable name in the C code... not sure...
//                    //won't worry about it for now.  This may never be used.
//                    w = tau1 * z_qpf + mu1 + sig1 * zValue;
//                }

                final double cdfValues = NormalDist.STD_NORM_DIST.functionCDF(w);

                double quaValue;

                if(isStrictEPT)
                {
                    // The quantiles computed from cdfValues should be no less than _noRainThreshold in theory. If they are, 
                    // set them to _noRainThreshold. LW
                    quaValue = sDist.functionInverseCDF(cdfValues);
                    if(Math.abs(quaValue) < _noRainThreshold)
                    {
                        quaValue = _noRainThreshold;
                    }
                }
                else
                {
                    quaValue = sDist.functionInverseCDF((cdfValues - by1) / by2);
                    if(Math.abs(quaValue) < _noRainThreshold)
                    {
                        quaValue = 0.0d;
                    }

                }

                if((_thresholdToIncludeSamplingLogMessages != null)
                    && (cdfValues > _thresholdToIncludeSamplingLogMessages.doubleValue()))
                {
                    LOG.info("EPT Algorithm for event " + event
                        + ": After application of EPT linear function to standard normal deviate " + zValue
                        + " with QPF in normal space of " + z_qpf + ", the probability used is " + cdfValues
                        + " resulting in quantile " + quaValue + " mm sampled from distribution " + sDist + ".");
                }

                //From how the code is structured, I know that partitions are not used and qpf > 0.5 threshold.  Hence, I know that the
                //s-distribution will be used to go from cdfValues to quantile values.  So, do it...
                quaValues.add(quaValue);
            }

            //I am going to leave out the partitioning code, since it is commented out in the default release.
            //See the C code in the prototypes for more info.
            this.sortAndPutValuesAtEndOfArray(ensemble, quaValues);
        }

//XXX Write output to file for testing.  Make sure the scenarioName is correct and matches the input files output above.
//        NumberTools.writeNumbersToFile(new File("testdata/eptPrecipitationForecastEnsembleCalculator/test2_out.data.txt"),
//                                       ensemble);

        return ensemble;
    }

//    /**
//     * This implements the scheme in the C code to sample from the array of positive numbers. Currently, the first value
//     * is weighted double the rest of the values (based on C code).
//     * 
//     * @param valuesToSample The values from which to sample.
//     * @param numberOfValues The number of samples to generate.
//     * @return
//     */
//    private List<Double> samplePositiveValues(final List<Float> valuesToSample, final int numberOfValues)
//    {
//        final List<Double> quaValues = new ArrayList<Double>(numberOfValues);
//
//        //XXX The old algorithm (copied from C) would give the first element in the values double the weight of the others.
//        //This has been fixed based on the approval from Limin so that each sample is weighted equally.  Its just a matter
//        //of letting indexToUse to be the probability * the number of samples, truncated to an int.  
//        for(int i = 0; i < numberOfValues; i++)
//        {
//            double prob = 0.0d;
//            if(_stratifiedSampling)
//            {
//                // Stratified sampling with the Weibull plotting position:
//                // LW prob = (double)(i + 1) / (double)(numberOfValues + 1);
//                prob = ThreadedRandomTools.generateStratifiedRandomSampleProb(numberOfValues, i);
//            }
//            else
//            {
//                prob = ThreadedRandomTools.getRandomNumGen().nextDouble();
//            }
//
//            //This affects the ensemble generation test output.  For some reason, it does not affect the adapter output,
//            //but maybe I'm just not testing EPT yet (???).
//            final int indexToUse = (int)(prob * valuesToSample.size());
//
//            //The old algorithm, based on C code.
////            int indexToUse = (int)(prob * (valuesToSample.size() + 1)) - 1;
////            if(indexToUse < 0)
////            {
////                indexToUse = 0;
////            }
//
//            quaValues.add(valuesToSample.get(indexToUse).doubleValue());
//        }
//
//        return quaValues;
//    }

    /**
     * Puts the values at the end of the array.
     * 
     * @param arrayToAffect The array to change.
     * @param valuesToPutAtEnd The values to put at the end of the array (in sorted order).
     */
    private void sortAndPutValuesAtEndOfArray(final double[] arrayToAffect, final List<Double> valuesToPutAtEnd)
    {
        Collections.sort(valuesToPutAtEnd);
        for(int i = 0; i < valuesToPutAtEnd.size(); i++)
        {
            arrayToAffect[arrayToAffect.length - valuesToPutAtEnd.size() + i] = valuesToPutAtEnd.get(i);
        }
    }

    //XXX The method below is not used because we are converting the Pearson Type-III distribution to 
    //Gamma distribution for testing.  We may need to bring a variation of this back in the future, so I'll 
    //leave it in for now but commented out.
//    private static final double MINIMUM_DISTRIBUTION_PARAMETER = 0.000001d;
//
//    private void validateAndFixDistributionParameters(final PrecipitationOneSetParameterValues parameters)
//    {
//        //x1 y0 dist
//        if(parameters.getEPTObsZeroFcstDistParm(0) < MINIMUM_DISTRIBUTION_PARAMETER)
//        {
//            parameters.setEPTObsZeroFcstDistParm(0, 1d);
//        }
//        if(parameters.getEPTObsZeroFcstDistParm(1) < MINIMUM_DISTRIBUTION_PARAMETER)
//        {
//            parameters.setEPTObsZeroFcstDistParm(1, 2d);
//        }
//
//        //x1 y1, fcst dist
//        if(parameters.getEPTFcstBivarMarginalDistParm(0) < MINIMUM_DISTRIBUTION_PARAMETER)
//        {
//            parameters.setEPTFcstBivarMarginalDistParm(0, 1d);
//        }
//        if(parameters.getEPTFcstBivarMarginalDistParm(1) < MINIMUM_DISTRIBUTION_PARAMETER)
//        {
//            parameters.setEPTFcstBivarMarginalDistParm(1, 2d);
//        }
//    }

}
