package ohd.hseb.hefs.mefp.models.precipitation;

import static com.google.common.base.Preconditions.checkArgument;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ohd.hseb.hefs.mefp.models.MEFPModelTools;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.dist.types.NormalDist;
import ohd.hseb.util.data.DataSet;

/**
 * Calculates the parameters required for the EPT. It also provides a method to populate a
 * {@link PrecipitationOneSetParameterValues} object based on the computed parameters. That object can then be put into
 * the {@link FullModelParameters} for the appropriate day-of-year and canonical event.
 * 
 * @author hank.herr
 */
public class EPTPrecipitationParameterCalculator
{
    private static final int NUMBER_OF_DATA_SET_VARS = 3;

    /**
     * In the Fortran code, the minimum sample size for parameter estimation, here, is not the same as for ensemble
     * generation, 14.
     */
    private static final int MIN_SAMPLE_SIZE = 5; // A size that is small yet won't break down parameter estimation. LW

    //When comparing to fortran, x is fcst and y is obs.  

    private final double _fixedZeroThreshold;

    /**
     * This parameter is used to determine if the parameters are considered missing. It must only be set to a non-NaN
     * value if all parameters necessary were computed successfully. If the parameters are considered "missing" it
     * should be left as NaN.
     */
    private double _probabiliyFcst0Obs0 = Double.NaN;
    private double _probabiliyFcst0Obs1 = Double.NaN;
    private double _probabiliyFcst1Obs0 = Double.NaN;
    private ContinuousDist _fcst1Obs0FcstDistribution = null;
    private ContinuousDist _fcst1Obs1IntermittencyFcstMarginalDistribution = null;
    private ContinuousDist _fcst1Obs1FcstMarginalDistribution = null;
    private ContinuousDist _fcst1Obs1ObsMarginalDistribution = null;
    private double _pearsonCorrInNormalSpace = Double.NaN;
    private float _alpha = Float.NaN;

    private String _questionableMessage = null;

    //DEBUG
//    private final long[] _runTimes = new long[7];

    /**
     * @param fixedZeroThreshold This is a bit redundant with the pthresh argument passed into
     *            {@link #calculate(List, List, DistributionType, float)}, but I'll leave it here for now.
     */
    public EPTPrecipitationParameterCalculator(final double fixedZeroThreshold)
    {
        _fixedZeroThreshold = fixedZeroThreshold;
    }

    public String getQuestionableMessage()
    {
        return _questionableMessage;
    }

//    public void addRunTimes(final long[] times)
//    {
//        for(int i = 0; i < times.length; i++)
//        {
//            times[i] = times[i] + _runTimes[i];
//        }
//    }

    /**
     * @return An array specifying the number of values greater than the corresponding thresholds,
     *         {@link #_precipThresholdForFcst} and {@link #_precipThresholdForObs}; [3] = both exceed the thresholds,
     *         [2] obs exceeds threshold, [1] fcst exceeds threshold, [0] neither.
     */
    public int[] computeCounts(final List<Float> forecastValues, final List<Float> observedValues)
    {
        final int[] counts = new int[4];

        //Populate the rain/norain lists.
        final List<Float> x0_y1 = new ArrayList<Float>();
        final List<Float> x1_y0 = new ArrayList<Float>();
        final List<Float> r = new ArrayList<Float>();
        final List<Float> s = new ArrayList<Float>();
        final double thresh = _fixedZeroThreshold;
        counts[0] = MEFPModelTools.constructPrecipitationRainNoRainLists(forecastValues,
                                                                         observedValues,
                                                                         thresh,
                                                                         x0_y1,
                                                                         x1_y0,
                                                                         r,
                                                                         s);

        counts[1] = x1_y0.size();
        counts[2] = x0_y1.size();
        counts[3] = r.size();
        return counts;
    }

    /**
     * @param f_sample Sample of forecast data.
     * @param o_sample Sample of observed data.
     * @param d_type Distribution type to use.
     * @param pthresh_epp2 Threshold used for determining what 'zero precip' is.
     * @throws Exception If any errors are encountered.
     */
    public void calculate(final List<Float> f_sample,
                          final List<Float> o_sample,
                          final DistributionType distType,
                          final float pthresh_epp2,
                          final float alpha) throws Exception
    {
        //System.err.println("####>> SIZES -- " + f_sample.size() + " -- " + o_sample.size());
        //        Arrays.fill(this._runTimes, 0L);
        checkArgument(f_sample.size() == o_sample.size(),
                      "f_sample (%s) and o_sample (%s) have different lengths.",
                      f_sample.size(),
                      o_sample.size());

        int i;
        DistributionType workingDistType; // gam: 2; gev: 3; kap:8; pe3: 10; wei:12.

        //Populate the rain/norain lists.
        final List<Float> x0_y1 = new ArrayList<Float>();
        final List<Float> x1_y0 = new ArrayList<Float>();
        final List<Float> r = new ArrayList<Float>();
        final List<Float> s = new ArrayList<Float>();
        final double thresh = pthresh_epp2;
        final int count0 = MEFPModelTools.constructPrecipitationRainNoRainLists(f_sample,
                                                                                o_sample,
                                                                                thresh,
                                                                                x0_y1,
                                                                                x1_y0,
                                                                                r,
                                                                                s);

        //        HStopWatch timer = new HStopWatch();
        // Compute p00, p01, p10, and p11.

        //        _runTimes[0] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        //Compute probabilities of rain/norain
        final double p00 = (double)count0 / (double)f_sample.size();
        final double p01 = (double)x0_y1.size() / (double)f_sample.size();
        final double p10 = (double)x1_y0.size() / (double)f_sample.size();

        final DataSet x1y0FcstDataSet = createWorkingDataSet(x1_y0, true);
        DataSet rawFcstDataSet = createWorkingDataSet(r, true);
        DataSet obsDataSet = createWorkingDataSet(s, true);
        //        _runTimes[1] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        // Fit x0_y1 with a Pearson Type III distribution (Gamma with shift parameter).
        workingDistType = DistributionType.GAMMA;
        ContinuousDist fcst1Obs0FcstDistribution = null;
        if(x1_y0.size() >= MIN_SAMPLE_SIZE)
        {
            // Fit x1_y0 to the selected distribution for intermittency calculation.
            fcst1Obs0FcstDistribution = MEFPPrecipitationModelTools.fitDistribution(x1y0FcstDataSet,
                                                                                    workingDistType,
                                                                                    _fixedZeroThreshold);
        }

        //        _runTimes[2] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        //If r is too small, set the correlation to be invalid and throw an exception.
        if(r.size() < MIN_SAMPLE_SIZE)
        {
            throw new Exception("EPT: The sample size is too small to estimate the joint distribution; need "
                + MIN_SAMPLE_SIZE + " samples but found " + r.size() + ".");
        }

        // Fit R as a Pearson Type III for intermittency calculation.
        final ContinuousDist fcst1Obs1IntermittencyFcstMarginalDistribution = MEFPPrecipitationModelTools.fitDistribution(rawFcstDataSet,
                                                                                                                          workingDistType,
                                                                                                                          _fixedZeroThreshold);
        //        _runTimes[3] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        //Check dist type passed in.
        if((distType != DistributionType.GAMMA) && (distType != DistributionType.WEIBULL)
            && (distType != DistributionType.PEARSON_TYPE_3) && (distType != DistributionType.WEIBULL3))
        {
            throw new Exception("EPT: Invalid distribution model type specified, " + distType
                + " (expection GAMMA or WEIBULL).");
        }

        // Fit R to the selected distribution and map R to U using NQT.
        // R is for forecasts.
        final ContinuousDist fcst1Obs1FcstMarginalDistribution = MEFPPrecipitationModelTools.fitDistribution(rawFcstDataSet,
                                                                                                             distType,
                                                                                                             _fixedZeroThreshold);

        //        _runTimes[4] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        // Fit S to the selected distribution and map S to V using NQT.
        // S is for observations.
        final ContinuousDist fcst1Obs1ObsMarginalDistribution = MEFPPrecipitationModelTools.fitDistribution(obsDataSet,
                                                                                                            distType,
                                                                                                            _fixedZeroThreshold);

        //        _runTimes[5] = timer.getElapsedMillis();
        //        timer = new HStopWatch();

        // Compute sample correlation coefficient between U and V.
        //Reconstruct the original data set because I need to maintain the order of events.
        rawFcstDataSet = createWorkingDataSet(r, false);
        createNQTVariable(rawFcstDataSet, fcst1Obs1FcstMarginalDistribution);
        obsDataSet = createWorkingDataSet(s, false);
        createNQTVariable(obsDataSet, fcst1Obs1ObsMarginalDistribution);
        final double[] u = rawFcstDataSet.getVariable(2);
        final double[] v = obsDataSet.getVariable(2);

        double avg_u, sd_u;
        double avg_v, sd_v;
        double rho;

        avg_u = rawFcstDataSet.mean(2);
        sd_u = rawFcstDataSet.sampleStandardDeviation(2);
        avg_v = obsDataSet.mean(2);
        sd_v = obsDataSet.sampleStandardDeviation(2);

        rho = 0.0;
        for(i = 0; i < s.size(); i++)
        {
            rho += (u[i] - avg_u) * (v[i] - avg_v);
        }
        rho = rho / (s.size() - 1);
        rho = rho / (sd_u * sd_v);
        //        _runTimes[6] = timer.getElapsedMillis();

        if(rho < 0)
        {
            final DecimalFormat df = new DecimalFormat("0.######");
            _questionableMessage = "EPT: Negative correlation coefficient computed: " + df.format(rho);
        }

        _alpha = alpha;
        _pearsonCorrInNormalSpace = rho;
        _probabiliyFcst0Obs0 = p00;
        _probabiliyFcst0Obs1 = p01;
        _probabiliyFcst1Obs0 = p10;
        _fcst1Obs1IntermittencyFcstMarginalDistribution = fcst1Obs1IntermittencyFcstMarginalDistribution;
        _fcst1Obs1FcstMarginalDistribution = fcst1Obs1FcstMarginalDistribution;
        _fcst1Obs1ObsMarginalDistribution = fcst1Obs1ObsMarginalDistribution;
        _fcst1Obs0FcstDistribution = fcst1Obs0FcstDistribution;
    }

    /**
     * @param values The float values to use in the data set.
     * @return A DataSet to hold three variables: streamflow value, empirical CDF, and NQT transformed. The NQT
     *         transformed variable will NOT be based on the empirical CDF, but, rather, a fitted distribution. The
     *         empirical CDF variable is used to fit said distribution.
     */
    private DataSet createWorkingDataSet(final List<Float> values, final boolean createCDF)
    {
        final DataSet dataSet = new DataSet(values.size(), NUMBER_OF_DATA_SET_VARS);
        for(int i = 0; i < values.size(); i++)
        {
            final double[] asample = new double[NUMBER_OF_DATA_SET_VARS];
            Arrays.fill(asample, DataSet.MISSING);
            asample[0] = values.get(i);
            dataSet.addSample(asample);
        }

        if(createCDF)
        {
            dataSet.createCDFUsingWeibullPlottingPosition(0, 1);
            dataSet.setFitCDFVariable(1);
            dataSet.setFitSampleVariable(0);
        }
        return dataSet;
    }

    /**
     * Populates the third variable of the workingDataSet with the NQT transformed values assuning the second variable
     * are the CDFs.
     * 
     * @param dataSet
     * @param dist
     */
    private void createNQTVariable(final DataSet workingDataSet, final ContinuousDist dist)
    {
        workingDataSet.applyDistTransform(0, 2, NormalDist.STD_NORM_DIST, dist); //Create an NQT variable in variable 3.
    }

    public String generateTestOutputString()
    {
        final DecimalFormat nf = new DecimalFormat("0.00000");
        return "EPT Parameters: _probabiliyFcst0Obs0 = " + nf.format(_probabiliyFcst0Obs0)
            + ", _probabiliyFcst0Obs1 = " + nf.format(_probabiliyFcst0Obs1) + ", _probabiliyFcst1Obs0 = "
            + nf.format(_probabiliyFcst1Obs0) + ", _fcst1Obs0FcstDistribution = "
            + _fcst1Obs0FcstDistribution.toString(nf) + "; _fcst1Obs1IntermittencyFcstMarginalDistribution = "
            + _fcst1Obs1IntermittencyFcstMarginalDistribution.toString(nf) + "; _fcst1Obs1FcstMarginalDistribution = "
            + _fcst1Obs1FcstMarginalDistribution.toString(nf) + "; _fcst1Obs1ObsMarginalDistribution = "
            + _fcst1Obs1ObsMarginalDistribution.toString(nf) + "; _pearsonCorrInNormalSpace = "
            + nf.format(_pearsonCorrInNormalSpace) + "; _alpha = " + nf.format(_alpha) + ".";
    }

    /**
     * Sets values in the provided {@link PrecipitationOneSetParameterValues} class based on the computed parameters.
     * 
     * @param values
     */
    public void populateOneSetParameterValues(final PrecipitationOneSetParameterValues values)
    {
        values.setEPTBivarProbNoPrecip(_probabiliyFcst0Obs0);
        values.setEPTFcstZeroObsDist(null); //NOT USED!
        values.setEPTFcstBivarMarginalDist(_fcst1Obs1FcstMarginalDistribution);
        values.setEPTFcstIntermittencyDist(_fcst1Obs1IntermittencyFcstMarginalDistribution);
        values.setEPTObsBivarMarginalDist(_fcst1Obs1ObsMarginalDistribution);
        values.setEPTFcstPoPObsZero(_probabiliyFcst1Obs0);
        values.setEPTObsPopFcstZero(_probabiliyFcst0Obs1);
        values.setEPTObsZeroFcstDist(_fcst1Obs0FcstDistribution);
        values.setEPTRho(_pearsonCorrInNormalSpace);

        //XXX The following three parameters show up in the commented Fortran code below.  They are connected to an
        //old estimation option referred to as iadj_epp2 in the code.  At some point, we decided to not make this option
        //available to users, so that part of the parameter estimation algorithm is eliminated.  The values computed
        //for the parameters, then, are missing.
        values.setEPTMu(Double.NaN);
        values.setEPTSig(Double.NaN);
        values.setEPTXmin(_alpha); //The parameter used to store alpha, the conditional bias penalty weight
    }

// THE ORIGINAL CODE UPON WHICH THIS IS BASED
//    int compute_epp2_param(int *ilog,
//                           int *maxnparams_epp2,
//                           float f_sample[],    
//                           float o_sample[],
//                           int *sample_size, 
//                           int *d_type,
//                           int *adj,
//                           int *obj_fun_flag,
//                           int *err_std_flag,
//                           float *prob,
//                           float *weight1,
//                           float *weight2,
//                           float *weight3,
//                           float *err_multiplier,
//                           int *ithreshopt,
//                           float *pthresh_epp2,
//                           float *pthresh1,
//                           float *pthresh2,
//                           float parameters[],
//                           int *nparams_epp2)
//
//    {
//       int i, j, k;
//       int ifail;
//       int zone;
//       int count1 = 0;
//       int count2 = 0;
//       int count3 = 0;
//       int count4 = 0;
//       int x0_y1_size, x1_y0_size, x_pos_size,r_size, s_size, param_size;
//       int nor_dist = 9;
//       int dist_type; // gam: 2; gev: 3; kap:8; pe3: 10; wei:12.
//       int pe_method; // LM: 1; ML: 2; CM: 3. 
//       int log_line_len;
//       float p00 = 0.0;
//       float p01 = 0.0;
//       float p10 = 0.0;
//       float p11 = 0.0;
//       float param_snd[2]; 
//       float param[MAX_NUM_SAMPLE_POINTS];
//       float x0_y1[MAX_NUM_SAMPLE_POINTS];
//       float x1_y0[MAX_NUM_SAMPLE_POINTS];
//       float x_pos[MAX_NUM_SAMPLE_POINTS];
//       float cdf_r[MAX_NUM_SAMPLE_POINTS];
//       float cdf_s[MAX_NUM_SAMPLE_POINTS];
//       float y4x_pos[MAX_NUM_SAMPLE_POINTS]; 
//       float r[MAX_NUM_SAMPLE_POINTS], s[MAX_NUM_SAMPLE_POINTS];
//       float u[MAX_NUM_SAMPLE_POINTS], v[MAX_NUM_SAMPLE_POINTS];
//       char dump_file[100];
//       char log_line[200];
//
//       int log = *ilog;
//       int maxnparams = *maxnparams_epp2;
//       int n_sample_pts = *sample_size;
//       int adj_b = *adj; // Yes: 1; No: 0
//       int obj_fun = *obj_fun_flag;
//       int use_err_std = *err_std_flag;
//       float prob_value = *prob;
//       float lam1 = *weight1;
//       float lam2 = *weight2;
//       float lam3 = *weight3;
//       float sig_factor = *err_multiplier;
//       int threshopt = *ithreshopt;
//       float thresh = *pthresh_epp2;
//       float thresh1 = *pthresh1;
//       float thresh2 = *pthresh2;
//       int nparams = *nparams_epp2;
//
//    /*
//    printf("lllllllllllllllllllllllllllllllllllllllllllllll\n");
//    printf("log = %d \n",log);
//    printf("maxnparams = %d \n",maxnparams);
//    printf("n_sample_pts = %d \n",n_sample_pts);
//    printf("adj_b = %d \n",adj_b);
//    printf("obj_fun = %d \n",obj_fun);
//    printf("use_err_std = %d \n",use_err_std);
//    printf("prob_value = %f \n",prob_value);
//    printf("lam1 = %f \n",lam1);
//    printf("lam2 = %f \n",lam2);
//    printf("lam3 = %f \n",lam3);
//    printf("sig_factor = %f \n",sig_factor);
//    printf("threshopt = %d \n",threshopt);
//    printf("thresh = %f \n",thresh);
//    printf("thresh1 = %f \n",thresh1);
//    printf("thresh2 = %f \n",thresh2);
//    printf("nparams = %d \n",nparams);
//    printf("lllllllllllllllllllllllllllllllllllllllllllllll\n");
//    for (i = 0; i < n_sample_pts; i++)
//    {
//      printf("%f  ",f_sample[i]);
//    }
//    printf("\n ");
//    printf("LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL\n");
//    for (i = 0; i < n_sample_pts; i++)
//    {
//      printf("%f  ",o_sample[i]);
//    }
//    printf("\n ");
//    */
//
//    //   printf("Entering compute_param \n");
//    //   printf("n_sample_pts = %d \n",n_sample_pts);
//
//    //   strcpy(log_line,"Entering compute_param \n");
//    //   write2log_(&log,log_line);
//
//    printf("####>> HANK DATA  --------------------------------\n");
//    for (i = 0; i <  n_sample_pts; i ++)
//    {
//        printf("%f  %f\n", f_sample[i], o_sample[i]);
//    }
//    printf("####>> END HANK DATA -----------------------------\n");
//
//
//       param_size = 4;
//       param_snd[0] = 0.0;
//       param_snd[1] = 1.0;
//       // Compute p00, p01, p10, and p11.
//       for (i = 0; i < n_sample_pts; i++)
//       {
//          if (f_sample[i] < thresh) // X = 0
//          {
//             if (o_sample[i] < thresh)
//             {
//                p00 = p00+1.0;
//             }
//             else
//             {
//                p01 = p01+1.0;
//                x0_y1[count1] = o_sample[i]; // X = 0 & Y > 0
//                count1++;
//             } 
//          }
//          else // X > 0
//          {
//             if (o_sample[i] < thresh)
//             {
//                p10 = p10+1.0;
//                x1_y0[count2] = f_sample[i]; // X > 0 & Y = 0
//                count2++;
//             }
//             else
//             {
//                p11 = p11+1.0;
//                // X > 0 & Y > 0
//                r[count3] = f_sample[i];
//                s[count3] = o_sample[i];
//                count3++;
//             } 
//             x_pos[count4] = f_sample[i]; // X > 0
//             y4x_pos[count4] = o_sample[i];
//             count4++;
//          }
//       } // End of loop for i.
//
//       x0_y1_size = count1;
//       x1_y0_size = count2;
//       r_size = count3;
//       s_size = count3;
//       x_pos_size = count4;
//
//    //   printf("x0_y1_size = %d \n",x0_y1_size);
//    //   printf("x1_y0_size = %d \n",x1_y0_size);
//    //   printf("r_size = %d \n",r_size);
//
//       // Initialize array parameters to 0.
//       for (i = 0; i < nparams; i++)
//       {
//          parameters[i] = 0.0;
//       }
//
//       p00 = p00/n_sample_pts;
//       p01 = p01/n_sample_pts;
//       p10 = p10/n_sample_pts;
//       p11 = p11/n_sample_pts;
//
//       parameters[0] = p00;
//       parameters[1] = p01;
//       parameters[2] = p10;
//
//       // Fit x0_y1 to the selectd distribution.
//       // use_emp_quantile = 0: use simple empirical quantiles in the ensemble generation program; 
//       // use_emp_quantile = 1: use quantiles from Gaussain kernel smoothing fit.
//       int use_emp_quantile = 0;
//    //   if (use_emp_quantile == 1)  NEVER USED!!!
//    //   {
////          dist_type = *d_type;
////          if (dist_type == 51)
////          {
////             pe_method = 4;
////          }
////          else
////          {
//     //        dist_type = 12;
////             pe_method = 2;
////          }
////          par_(x0_y1,&x0_y1_size,&dist_type,&pe_method,param,&ifail);
////          for (i = 0; i < 4; i++)
////          {
////             parameters[3+i] = param[i];
////          }
//    //   }
//
//       dist_type = 10;
//       pe_method = 1;
//
//       if (x1_y0_size > MIN_SAMPLE_SIZE)
//       {
//          // Fit x1_y0 to the selected distribution for intermittency calculation.
//          par_(x1_y0,&x1_y0_size,&dist_type,&pe_method,param,&ifail);
//          for (i = 0; i < 4; i++)
//          {
//             parameters[7+i] = param[i];
//          }
//       }
//
//       if (r_size < MIN_SAMPLE_SIZE)
//       {
//          printf("WARNING: Sample size is too small for modeling the cont-cont part.\n");
//          parameters[23] = 1.5;
//          return;
//       }
//
//       // Fit R to the selected distribution for intermittency calculation.
//       par_(r,&r_size,&dist_type,&pe_method,param,&ifail);
//       for (i = 0; i < 4; i++)
//       {
//          parameters[11+i] = param[i];
//       }
//
//       // Fit R to the selected distribution and map R to U using NQT.
//       // R is for forecasts.
//       dist_type = *d_type;
//       if (dist_type == 51)
//       {
//          pe_method = 4;
//       }
//       else if (dist_type == 12)
//       {
//          pe_method = 2;
//       }
//       else if (dist_type == 10)
//       {
//          pe_method = 1;
//       }
//       else
//       {
//          printf("Wrong distribution model type. \n");
//          exit(EXIT_FAILURE);
//       }
//       par_(r,&r_size,&dist_type,&pe_method,param,&ifail);
//       for (i = 0; i < 4; i++)
//       {
//          parameters[15+i] = param[i];
//       }
//       if (dist_type == 51)
//       {
//          param[r_size] = param[0];
//          for (i = 0; i < r_size; i++)
//          {
//             param[i] = r[i];
//          }
//          param_size = r_size+1;
//       }
//       cdf_(r,&r_size,&dist_type,param,&param_size,cdf_r);
//       qua_(cdf_r,&r_size,&nor_dist,param_snd,&param_size,u);
//      
//       // Fit S to the selected distribution and map S to V using NQT.
//       // S is for observations.
//        dist_type = *d_type;
//       if (dist_type == 51)
//       {
//          pe_method = 4;
//       }
//       else if (dist_type == 12)
//       {
//          pe_method = 2;
//       }
//       else if (dist_type == 10)
//       {
//          pe_method = 1;
//       }
//       else
//       {
//          printf("Wrong distribution model type. \n");
//          exit(EXIT_FAILURE);
//       }
//       par_(s,&s_size,&dist_type,&pe_method,param,&ifail);
//       for (i = 0; i < 4; i++)
//       {
//          parameters[19+i] = param[i];
//       }
//       if (dist_type == 51)
//       {
//          param[s_size] = param[0];
//          for (i = 0; i < s_size; i++)
//          {
//             param[i] = s[i];
//          }
//          param_size = s_size+1;
//       }
//       cdf_(s,&s_size,&dist_type,param,&param_size,cdf_s);
//       qua_(cdf_s,&s_size,&nor_dist,param_snd,&param_size,v);
//
//       // Compute sample correlation coefficient between U and V.
//       float avg_u, sd_u;
//       float avg_v, sd_v;
//       float rho;
//       mean(u,r_size,&avg_u);
//       sd_u = standard_dev(u,r_size);
//       mean(v,s_size,&avg_v);
//       sd_v = standard_dev(v,s_size);
//       rho = 0.0;
//       for (i = 0; i < s_size; i++)
//       {
//          rho += (u[i]-avg_u)*(v[i]-avg_v);
//       }
//       rho = rho/(s_size-1);
//       rho = rho/(sd_u*sd_v);
//       parameters[23] = rho;
//
//
//       if (adj_b == 1)
//       {
//          // Minimize an objective function to get the optimal coefficient of 
//          // the linear model. Minimization is done over the (R>0, S) subspace 
//          // in the original space.
//
//          float tol = 0.01;
//          float ax, bx, cx;
//          float xmin, obj_min;
//          float mu, sig;
//          struct obj_func_params ofp;
//
//          // Don't use a value > 1.2 for cx for the extended linear model.
//          ax = 0.10;
//          bx=rho;   // used the sample cross-correlation coefficient as the initial guess //* djseo *//
//          cx = 1.18;
////        For optimization region 1 with cutoff = 0.23 inches for huntingdon.
////        cx = 1.50;
//
//          // Eliminate the outliers in the Z space if z_band_factor < 100.
//          float z_band_factor = 125;
////          if (z_band_factor < 100.00)  NOT USED -- IF CLAUSE NEVER TRUE!
//     //     {
////             for (i = 0; i < r_size; i++)
////             {
////                x0_y1[i] = v[i] - u[i]; // Use x0_y1 as a working array.
////             }
////             sig = standard_dev(x0_y1,r_size);
////             sig = z_band_factor*sig;
//    //
////             int cnt = 0;
////             for (i = 0; i < r_size; i++)
////             {
////                if (fabs(v[i] - u[i]) < sig)
////                {
////                   u[cnt] = u[i];
////                   v[cnt] = v[i];
////                   cnt++;
////                }
////             }
////             r_size = cnt;
////          }
//
//          // Variable ofp gets assigned here. 
//          ofp.u_size = r_size;
//          ofp.size = x_pos_size;
//          ofp.u = u;
//          ofp.v = v;
//          ofp.r = r;
//          ofp.s = s;
//          ofp.xr = x_pos;
//          ofp.ys = y4x_pos;
//          ofp.parms = parameters;
//
//          int n_op_zones = 1;
//          if (n_op_zones == 1)
//          {
//             // Compute a minimum coefficient value using the Brent method.
//             // For (R>0, S).
//             zone = 0;
//             dist_type = *d_type;
//             obj_min = brent(compute_obj_func, &ofp,
//                             ax, bx, cx, tol, &xmin, prob_value, lam1, lam2, lam3, 
//                             sig_factor, obj_fun, dist_type, use_err_std, zone);
//
//             printf("obj_min = %f\n",obj_min);
//
//             //* if use_err_std=1, use sample standard deviation of the residual error *//
//             //* if use_err_std=0, use sample median of the absolute residual error    *//
//
////             int use_err_std = 0; // Yes: 0; No: 1.   //* djseo Feb 06, 2009 *//
//
//             for (i = 0; i < r_size; i++)
//             {
//                x0_y1[i] = v[i] - xmin*u[i]; // Use x0_y1 as a working array.
//             }
//             mean(x0_y1,r_size,&mu);
//
//             if (use_err_std == 0)
//             {
//                sig = standard_dev(x0_y1,r_size);
//             }
//             else
//             {
//                // Use the median of the absolute value of the errors.
//                for (i = 0; i < r_size; i++)
//                {
//                   x0_y1[i] = fabs(v[i] - xmin*u[i]);
//                }
//                sort_float_arr(x0_y1,r_size);
//                k = r_size/2;
//                if (r_size%2 == 0)
//                   sig = sig_factor*(x0_y1[k-1]+x0_y1[k])/2.0;
//                else
//                   sig = sig_factor*x0_y1[k];
//             }
//
//             parameters[24] = xmin;
//             parameters[25] = mu;
//             parameters[26] = sig;
//          }
////          else NEVER USED -- SEE IF CLAUSE ABOVE!
//     //     {
//     //        dist_type = *d_type;
//
//             // Compute a minimum coefficient value using the Brent method 
//             // for zone 1.
//     //        zone = 1;
//     //        obj_min = brent(compute_obj_func, &ofp,
//     //                        ax, bx, cx, tol, &xmin, prob_value, lam1, lam2, 
//     //                        sig_factor, lam3, obj_fun, dist_type, use_err_std, zone);
//
////             for (i = 0; i < r_size; i++)
////             {
////                x0_y1[i] = v[i] - xmin*u[i]; // Use x0_y1 as a working array.
////             }
////             mean(x0_y1,r_size,&mu);
////             sig = standard_dev(x0_y1,r_size);
//
////             parameters[24] = xmin;
////             parameters[25] = mu;
////             parameters[26] = sig;
//
//             // Compute a minimum coefficient value using the Brent method 
//             // for zone 2.
////             zone = 2; 
////             obj_min = brent(compute_obj_func, &ofp,
////                             ax, bx, cx, tol, &xmin, prob_value, lam1, lam2, lam3, 
//     //                        sig_factor, obj_fun, dist_type, use_err_std, zone);
//    //
//     //        for (i = 0; i < r_size; i++)
////             {
////                x0_y1[i] = v[i] - xmin*u[i]; // Use x0_y1 as a working array.
////             }
////             mean(x0_y1,r_size,&mu);
////             sig = standard_dev(x0_y1,r_size);
//
////             parameters[27] = xmin;
////             parameters[28] = mu;
////             parameters[29] = sig;
//
////         }
//       }
//       
//    printf("####>> HANK -- Parameters ----------------------\n");
//    for (i = 0; i < 24; i ++)
//    {
//        printf("####>> parameters %d %f\n", i, parameters[i]);
//    }
//    printf("####>> HANK -- Parameters DONE -----------------\n");
//    printf("####>> EXITTING --------------------\n");
//    exit(1);
//
//    }

    public static void main(final String[] args)
    {
        System.out.println("####>> Commons inverse cdf vs. Abramowitz and Stegun...");
        double largestDiff = Double.MIN_VALUE;
        for(double p = 0.001; p < 1.0; p += 0.001)
        {
            final double diff = NormalDist.STD_NORM_DIST.functionInverseCDF(p)
                - NormalDist.STD_NORM_DIST.functionInverseCDF(p);
            largestDiff = Math.max(diff, largestDiff);
            System.out.println("####>> " + p + " -- " + NormalDist.STD_NORM_DIST.functionInverseCDF(p) + " -- "
                + NormalDist.STD_NORM_DIST.functionInverseCDF(p) + " -- " + diff);
        }
        System.out.println("####>> ------------------- largest diff " + largestDiff);
    }

}
