package ohd.hseb.hefs.mefp.models.precipitation;

import ohd.hseb.hefs.mefp.models.ForecastEnsembleCalculator;
import ohd.hseb.hefs.mefp.models.MEFPModelTools;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.dist.DataFittingDistributionException;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.dist.MomentsFittingDistribution;
import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.dist.types.NormalDist;
import ohd.hseb.hefs.utils.dist.types.WeibullDist;
import ohd.hseb.hefs.utils.plugins.DefaultFactoryException;

/**
 * Given a deterministic forecast of the canonical event value, this computes an ensemble of realization of that
 * canonical event value that can serve as a probabilistic forecast. This is a recode of epp3_fcst2ensem.f.
 * 
 * @author hankherr
 */
public class IPTPrecipitationForecastEnsembleCalculator implements ForecastEnsembleCalculator
{
    private final static double POP_THRESHOLD = 0.01;
    private static double NORMAL_VARIATE_LIMIT = 10d;
    private static int MAXIMUM_NUMBER_OF_NORMAL_DEVIATES = 1000;
    private static double WEIBULL_CCV_LIMIT = 0.75;

    private MEFPForecastSource _source = null;
    private MEFPFullModelParameters _fullParameters = null;
    private int _dayOfYear = -1;

    private DistributionType _fcstDist;
    private double _pthreshFcst;
    private double _fcstPoP;
    private double _fcstCAvg;
    private double _fcstCCV;
    private DistributionType _obsDist;
    private double _pthreshObs;
    private double _obsPoP;
    private double _obsCAvg;
    private double _obsCCV;
    private double _rho;

    //These numbers are computed in Fortran but never used.
//    private double _genAvg = 0d;
//    private double _genPoP = 0d;
//    private double _genCAvg = 0d;
//    private double _genCCV = 0d;
//    private double _vMax;
//    private double _exProbMin;
//    private double _pMin;
//    private double _pMax;

    private final double[] _fullRealizationsInNormalSpace = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //vval
    private final double[] _fullRealizationPDFValues = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //cpdfv
    private final double[] _fullRealizationCDFValues = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //ccdfv
    private final double[] _fullRealizationsInOriginalSpace = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //p

    /**
     * For testing only!!! If using this constructor, be sure to call {@link #calculateForecastEnsemble(double, int)},
     * not {@link #calculateForecastEnsemble(CanonicalEvent, double, int)}, because there is no {@link #_source} or
     * {@link #_fullParameters} after this constructor is called.
     */
    public IPTPrecipitationForecastEnsembleCalculator(final DistributionType fcstDist,
                                                      final double pthreshFcst,
                                                      final double fcstPoP,
                                                      final double fcstCAvg,
                                                      final double fcstCCV,
                                                      final DistributionType obsDist,
                                                      final double pthreshObs,
                                                      final double obsPoP,
                                                      final double obsCAvg,
                                                      final double obsCCV,
                                                      final double rho)
    {
        _fcstDist = fcstDist;
        _pthreshFcst = pthreshFcst;
        _fcstPoP = fcstPoP;
        _fcstCAvg = fcstCAvg;
        _fcstCCV = fcstCCV;
        _obsDist = obsDist;
        _pthreshObs = pthreshObs;
        _obsPoP = obsPoP;
        _obsCAvg = obsCAvg;
        _obsCCV = obsCCV;
        _rho = rho;
    }

    /**
     * @param srcParameters The parameters for the forecast source.
     * @param parameters The entire set of estimated parameters for the forecast day and canonical event to be
     *            processed.
     */
    public IPTPrecipitationForecastEnsembleCalculator(final MEFPForecastSource source,
                                                      final MEFPFullModelParameters fullParametrs,
                                                      final int dayOfYear) throws Exception
    {
        _source = source;
        _fullParameters = fullParametrs;
        _dayOfYear = dayOfYear;
    }

    /**
     * @param event The {@link CanonicalEvent} for which to calculate ensembles.
     * @param canonicalEventValue The forecast canonical event value conditional on which an ensemble will be generated.
     * @param numberOfEnsembleMembers The number of members to generate.
     * @return The forecast ensemble as a double[].
     * @throws Exception Typically if the forecast or observed distribution cannot be instantiated for any reason.
     */
    @Override
    public double[] calculateForecastEnsemble(final CanonicalEvent event,
                                              final double canonicalEventValue,
                                              final int numberOfEnsembleMembers) throws Exception
    {
        final PrecipitationOneSetParameterValues parametersForEvt = (PrecipitationOneSetParameterValues)_fullParameters.getSourceEstimatedModelParameterValues(_source,
                                                                                                                                                               _dayOfYear,
                                                                                                                                                               event);

        setParametersFromOneSetParameterValues(parametersForEvt);
        return calculateForecastEnsemble(canonicalEventValue, numberOfEnsembleMembers);

    }

    /**
     * This method should only be called directly during testing, as it does not require {@link #_fullParameters} or
     * {@link #_source} to execute directly.
     * 
     * @param canonicalEventValue The forecast canonical event value conditional on which an ensemble will be generated.
     * @param numberOfEnsembleMembers The number of members to generate.
     * @return The forecast ensemble as a double[].
     * @throws Exception Typically if the forecast or observed distribution cannot be instantiated for any reason.
     */
    protected double[] calculateForecastEnsemble(final double canonicalEventValue, final int numberOfEnsembleMembers) throws Exception
    {
        double xFcst;
        if(canonicalEventValue < _pthreshFcst)
        {
            xFcst = 0d;
        }
        else
        {
            xFcst = canonicalEventValue - _pthreshFcst;
        }

        //fpz stores the probability of no precipitation.
        double fpz = 1 - POP_THRESHOLD;
        if(_fcstPoP >= POP_THRESHOLD)
        {
            fpz = 1 - _fcstPoP;
        }
        final double obspz = 1 - _obsPoP;

        //u0 stores the standard normal value corresponding to the probability of no precip.
        //Its the line at which values below are associated with no rain and above associated with rain.
        final double u0 = NormalDist.STD_NORM_DIST.functionInverseCDF(fpz);

        //Apply the NQT to acquire the standard normal deviate corresponding to value xFcst.
        //This method fits a distribution and then applies the NQT.
        final double uFcst = computeStandardNormalDeviateCorrespondingToValue(xFcst, fpz);

        //Compute 1000 deviates of the bivariate normal condition on the forecast uFcst.
        computeNumericalEstimateConditionalNormalObservedDistribution(uFcst, u0);

        //Instantiate the observed distribution based on xvtransi.  
        final ContinuousDist obsDist = instantiateObservedDistribution();

        //Compute the 1000 corresponding precipitation values to the afore-computed normal deviates in the original space.
        //Account for the affect of _pthreshObs accordingly.  Also, accumulate computations of the conditional average
        //and conditional coefficient of variation for storage in the _gen variables.
        final double pz = convertConditionalNormalObservedToOriginalSpace(obsDist, obspz);

        //Note that pz is the obs probability of zero adjusted slightly based on how the 1000 values line up.
        return MEFPModelTools.computeEnsembleMembers(pz,
                                                     numberOfEnsembleMembers,
                                                     _fullRealizationsInOriginalSpace,
                                                     _fullRealizationCDFValues);
    }

    /**
     * Sets up the parameters based on the provided {@link PrecipitationOneSetParameterValues}.
     * 
     * @param parameters The parameters.
     * @throws Exception Exception if any needed parameters are {@link Double#NaN}.
     */
    private void setParametersFromOneSetParameterValues(final PrecipitationOneSetParameterValues parameters) throws Exception
    {
        if(_source.isClimatologySource())
        {
            _fcstDist = _fullParameters.getSourceModelParameters(_source).getDistributionOfObs();
            _pthreshFcst = parameters.getObsPrecipThresh();
            _fcstPoP = parameters.getObsPoP();
            _fcstCAvg = parameters.getObsCondAvg();
            _fcstCCV = parameters.getObsCondCoeffVar();
            _rho = 0d; //xrho in the Fortran code
        }
        else
        {
            _fcstDist = _fullParameters.getSourceModelParameters(_source).getDistributionOfFcsts();
            _pthreshFcst = parameters.getFcstPrecipThresh();
            _fcstPoP = parameters.getFcstPoP();
            _fcstCAvg = parameters.getFcstCondAvg();
            _fcstCCV = parameters.getFcstCondCoeffVar();
            _rho = Math.max(0d, parameters.getRho()); //xrho in the Fortran code
        }
        _obsDist = _fullParameters.getSourceModelParameters(_source).getDistributionOfObs();
        _pthreshObs = parameters.getObsPrecipThresh();
        _obsPoP = parameters.getObsPoP();
        _obsCAvg = parameters.getObsCondAvg();
        _obsCCV = parameters.getObsCondCoeffVar();

        if((Double.isNaN(_fcstCAvg)) || (Double.isNaN(_fcstCCV)) || (Double.isNaN(_obsCAvg)) || (Double.isNaN(_obsCCV))
            || (Double.isNaN(_fcstPoP)) || (Double.isNaN(_obsPoP)) || (Double.isNaN(_pthreshFcst))
            || (Double.isNaN(_pthreshObs)) || (Double.isNaN(_rho)))
        {
            throw new Exception("At least one required parameter is missing (parameters = " + _fcstCAvg + ", "
                + _fcstCCV + ", " + _obsCAvg + ", " + _obsCCV + ", " + _fcstPoP + ", " + _obsPoP + ", " + _pthreshFcst
                + ", " + _pthreshObs + ", " + _rho + ").");
        }
    }

    /**
     * This performs checks necessary to switch from Weibull to Gamma if Weibull is a problem. NOTE that no such check
     * is needed to mimick xvtransi.f (see
     * {@link #computeValueCorrespondingToStandardNormalDeviate(ContinuousDist, double, double, double, double)}).
     * 
     * @return Appropriate {@link ContinuousDist} to use for calling
     *         {@link #computeValueCorrespondingToStandardNormalDeviate(ContinuousDist, double, double, double, double)}
     *         .
     * @throws DefaultFactoryException
     */
    private ContinuousDist instantiateObservedDistribution() throws DefaultFactoryException
    {
        //Directly from xvtransi.f
        if((_obsCAvg <= 0) || (_obsCCV <= 0.02))
        {
            return null;
        }

        ContinuousDist obsDist = null;
        if((_obsDist == DistributionType.WEIBULL) && (_obsCCV <= WEIBULL_CCV_LIMIT))
        {
            obsDist = (ContinuousDist)DistributionType.instantiateDistribution(DistributionType.GAMMA);
        }
        else
        {
            obsDist = (ContinuousDist)DistributionType.instantiateDistribution(_obsDist);
        }
        try
        {
            ((MomentsFittingDistribution)obsDist).fitToMoments(_obsCAvg, _obsCCV);
        }
        catch(final DataFittingDistributionException e)
        {
            throw new DefaultFactoryException("Unable to fit the used obs distribution to the moments mean = "
                + _obsCAvg + ", and coeffvar = " + _obsCCV);
        }
        return obsDist;
    }

    /**
     * Translated from xvtrans.f. This method fits a distribution and returns the NQT transformed value.
     * 
     * @param xFcst
     * @param fpz
     * @return
     * @throws Exception
     */
    private double computeStandardNormalDeviateCorrespondingToValue(final double xFcst, final double fpz) throws Exception
    {
        double usedFcstCCV = _fcstCCV;
        if(usedFcstCCV < 0.10)
        {
            usedFcstCCV = 0.10;
        }
//        double usedFcstCAvg = _fcstCAvg;
//        if(usedFcstCAvg < 0.01)
//        {
//            usedFcstCAvg = 0.01;
//        }

        //XXX From Fortran: For Weibull no computation is done if the usedFcstCCV <= 0.75!  Just return the median.
        if((_fcstDist == DistributionType.WEIBULL) && (usedFcstCCV <= WEIBULL_CCV_LIMIT))
        {
            return NormalDist.STD_NORM_DIST.functionInverseCDF(0.5);
        }

        //Instantiate forecast distribution based on xvtrans.
        final ContinuousDist fcstDist = instantiateForecastDistribution();

        //Apply the NQT to acquire the standard normal deviate corresponding to value xFcst.
        return computeStandardNormalDeviateCorrespondingToValue(fcstDist, xFcst, fpz, _fcstCAvg);
    }

    /**
     * Works as xvtrans, including if checks to limit the avg and ccv values.
     * 
     * @return Distribution of forecast values to use in
     *         {@link #computeStandardNormalDeviateCorrespondingToValue(ContinuousDist, double, double, double)}.
     * @throws DefaultFactoryException
     */
    private ContinuousDist instantiateForecastDistribution() throws DefaultFactoryException
    {
        double usedFcstCAvg = _fcstCAvg;
        double usedFcstCCV = _fcstCCV;
        if(usedFcstCCV < 0.1d)
        {
            usedFcstCCV = 0.1d;
        }
        if(usedFcstCAvg < 0.01d)
        {
            usedFcstCAvg = 0.01d;
        }

        final ContinuousDist fcstDist = (ContinuousDist)DistributionType.instantiateDistribution(_fcstDist);
        try
        {
            ((MomentsFittingDistribution)fcstDist).fitToMoments(usedFcstCAvg, usedFcstCCV);
        }
        catch(final DataFittingDistributionException e)
        {
            throw new DefaultFactoryException("Unable to fit the used fcst distribution to the moments mean = "
                + usedFcstCAvg + ", and coeffvar = " + usedFcstCCV);
        }

        return fcstDist;
    }

    /**
     * This is based on xvtrans.f. Read the internal comments to understand its inner workings.
     * 
     * @param dist Distribution of the precipitation variate.
     * @param precipitationValue The realization of the precipitation variate.
     * @param probabilityOfNoRain The unconditional probability of zero for the precipitation variate.
     * @param condAvgPrecipitation The conditional average of the precipitation variate (i.e., mean of dist).
     * @return Corresponding realization of a standard normal distribution using the NQT.
     */
    private double computeStandardNormalDeviateCorrespondingToValue(final ContinuousDist dist,
                                                                    final double precipitationValue,
                                                                    final double probabilityOfNoRain,
                                                                    final double condAvgPrecipitation)
    {

        double result = Double.NaN;
        final double cdfValue = probabilityOfNoRain + (1 - probabilityOfNoRain) * dist.functionCDF(precipitationValue);
        //If the precip value is positive, return the NQT applied to dist evalauted at the value.
        if(precipitationValue > 0)
        {
            result = NormalDist.STD_NORM_DIST.functionInverseCDF(cdfValue);
        }
        //Otherwise...
        else
        {
            //If the probability of no rain is positive, then return an expected value given 0 precip.
            if(probabilityOfNoRain > 0)
            {
                final double z0 = NormalDist.STD_NORM_DIST.functionInverseCDF(probabilityOfNoRain);
                result = 0.25 * z0 + (1 - 0.25) * (-0.3984228 * Math.exp(-0.5 * z0 * z0) / probabilityOfNoRain);
            }
            //If the probability of rain is 0, then...
            else
            {
                //Return 0 (median of a standard normal) if avgPrecipitation is also 0.
                if(condAvgPrecipitation == 0)
                {
                    result = 0d;
                }
                //Otherwise, return the smallest allowable value.
                result = -1d * NORMAL_VARIATE_LIMIT;
            }
        }

        if(result < -1 * NORMAL_VARIATE_LIMIT)
        {
            result = -1 * NORMAL_VARIATE_LIMIT;
        }
        else if(result > NORMAL_VARIATE_LIMIT)
        {
            result = NORMAL_VARIATE_LIMIT;
        }

        return result;
    }

    /**
     * This is based on xvtransi_02.f. Read the internal comments to understand its inner workings.
     * 
     * @param dist Distribution of the precipitation variate.
     * @param precipitationValue The realization of a deviate in standard normal space.
     * @param probabilityOfNoRain The unconditional probability of zero for the precipitation variate.
     * @param condAvgPrecipitation The conditional average of the precipitation variate (i.e., mean of dist).
     * @param condCoeffVar The conditional coefficient of variation of the precipitation variate (i.e., cv of dist).
     * @return Corresponding realization of a standard normal distribution using the NQT.
     */
    private double computeValueCorrespondingToStandardNormalDeviate(final ContinuousDist dist,
                                                                    double normalDeviate,
                                                                    final double probabilityOfNoRain,
                                                                    final double condAvgPrecipitation,
                                                                    final double condCoeffVar)
    {
        //Handle various special cases.
        if(normalDeviate < -1 * NORMAL_VARIATE_LIMIT)
        {
            return -9999;
        }
        if(normalDeviate > NORMAL_VARIATE_LIMIT)
        {
            normalDeviate = NORMAL_VARIATE_LIMIT;
        }
        if(condAvgPrecipitation <= 0)
        {
            return 0;
        }
        if(condCoeffVar <= 0.02)
        {
            return condAvgPrecipitation;
        }

        //from xvtransi.f: Special case for Weibull if the ccv <= 0.75
        if((dist instanceof WeibullDist) && (condCoeffVar <= 0.75))
        {
            return -9999;
        }

        //Compute the cdf value and check it against the probability of no rain.
        double cdfValue = NormalDist.STD_NORM_DIST.functionCDF(normalDeviate);
        if((cdfValue < probabilityOfNoRain) || (cdfValue > 1.0))
        {
            return 0;
        }
        //Otherwise scale the cdfValue so that it spans only the portion associated with rainfall.
        else
        {
            cdfValue = (cdfValue - probabilityOfNoRain) / (1 - probabilityOfNoRain);
        }

        //Finally, return the inverse CDF applied to the cdf value.
        return dist.functionInverseCDF(cdfValue);
    }

    /**
     * Populate the {@link #_fullRealizationsInNormalSpace}, {@link #_fullRealizationCDFValues}, and
     * {@link #_fullRealizationPDFValues} attributes to contain a number of deviates equal to their array sizes.<br>
     * <br>
     * This is pulled verbatum from the xcgauss.f routine.
     * 
     * @param forecastInNormalSpace The forecast canonical event value in normal space.
     * @param noRainDeviateInNormalSpace The normal deviate corresponding to the probability of no rainfall.
     */
    private void computeNumericalEstimateConditionalNormalObservedDistribution(final double forecastInNormalSpace,
                                                                               final double forecastNoRainDeviateInNormalSpace)
    {
        final int nv = this._fullRealizationsInNormalSpace.length;

        //Direct translation of fortran. u == forecastInNormalSpace. u0 == noRainDeviateInNormalSpace
        final double rhothresh = 0.9999d;
        final double vstdthresh = Math.sqrt(1d - rhothresh * rhothresh);
//        final double theta = 0.25d;
        final double z1 = -4d;
        final double z2 = 4d;
        final double dz = (z2 - z1) / nv;
        double vavg, vstd;
        if(_rho < 0)
        {
            vavg = 0d;
            vstd = 1d;
        }
        else if(_rho < rhothresh)
        {
            vavg = _rho * forecastInNormalSpace; // conditional mean of v given u
            vstd = Math.sqrt(1. - _rho * _rho); // conditional std dev of v given u
        }
        else
        {
            vavg = rhothresh * forecastInNormalSpace;
            vstd = vstdthresh;
        }

//        final double fpz = NormalDist.STD_NORM_DIST.functionCDF(forecastNoRainDeviateInNormalSpace);
//        final double fpop = 1. - fpz;
        final int nv2 = (nv / 2);
        double dv;
        if(_rho < rhothresh)
        {
            dv = (vstd * dz);
        }
        else
        {
            dv = 0d;
        }

        //vval is centered at the median and includes values that are dz standard deviations apart.
        for(int i = 0; i < nv; i++)
        {
            _fullRealizationsInNormalSpace[i] = vavg + (i - nv2) * dv;
        }

        double sum = 0d;
        double zv;
        double ustd;
        double uavg;
        double z0;

        //This loop does not match exactly xcgauss.f.  But I've checked it against the loop there and it yields the same results.
        //It also uses fewer lines of code.
        for(int i = 0; i < nv; i++)
        {
            if(forecastInNormalSpace > forecastNoRainDeviateInNormalSpace)
            {
                if(_rho < rhothresh)
                {
                    zv = (_fullRealizationsInNormalSpace[i] - vavg) / vstd; // standardized conditional v
                    _fullRealizationPDFValues[i] = NormalDist.STD_NORM_DIST.functionPDF(zv) * dv;
                }
                else
                {
                    //Uniform distribution
                    _fullRealizationPDFValues[i] = 1. / nv;
                }
            }
            else
            //conditional distribution of v is integrated over u<=u0
            {
                if(_rho > rhothresh)
                {
                    //Uniform distribution
                    _fullRealizationPDFValues[i] = 1. / nv;
                }
                else
                {
                    ustd = Math.sqrt(1. - _rho * _rho);
                    uavg = _rho * _fullRealizationsInNormalSpace[i]; //  conditional mean of u, given v
                    z0 = (forecastNoRainDeviateInNormalSpace - uavg) / ustd; //  standardized value of u0 conditioned on v
                    _fullRealizationPDFValues[i] = NormalDist.STD_NORM_DIST.functionPDF(_fullRealizationsInNormalSpace[i])
                        * NormalDist.STD_NORM_DIST.functionCDF(z0) * dv;
                }
            }
            sum = sum + _fullRealizationPDFValues[i];
            _fullRealizationCDFValues[i] = sum;
        }

        sum = _fullRealizationCDFValues[nv - 1];
        for(int i = 0; i < nv; i++)
        {
            _fullRealizationCDFValues[i] = _fullRealizationCDFValues[i] / sum;
            _fullRealizationPDFValues[i] = _fullRealizationPDFValues[i] / sum;
        }
    }

    /**
     * Computes the attribute {@link #_fullRealizationsInOriginalSpace} based on the
     * {@link #_fullRealizationsInNormalSpace} values. The {@link #_fullRealizationsInOriginalSpace} attribute stores
     * the original space precipitation values corresponding to the normal space values in
     * {@link #_fullRealizationsInNormalSpace}.
     * 
     * @param obsDist The distribution of observed precipitation to use.
     * @param obsProbNoRain The observed probability of no rain to use.
     * @return the CDF value associated with the last zero value in the generated
     *         {@link #_fullRealizationsInOriginalSpace} attribute.
     */
    private double convertConditionalNormalObservedToOriginalSpace(final ContinuousDist obsDist,
                                                                   final double obsProbNoRain)
    {
//These numbers are computed but not used in Fortran!
//        double avg = 0;
//        double cavg = 0;
//        double ccv = 0;
        double pz = 0;
        boolean foundFirstPositiveValueAboveThreshold = false;
        for(int i = 0; i < _fullRealizationsInOriginalSpace.length; i++)
        {
            _fullRealizationsInOriginalSpace[i] = computeValueCorrespondingToStandardNormalDeviate(obsDist,
                                                                                                   _fullRealizationsInNormalSpace[i],
                                                                                                   obsProbNoRain,
                                                                                                   _obsCAvg,
                                                                                                   _obsCCV);

            //For missing/bad data, just assume 0 precip.
            if(_fullRealizationsInOriginalSpace[i] < 0)
            {
                _fullRealizationsInOriginalSpace[i] = 0;
            }

            //For positive data, make sure it exceeds the _pthreshObs.
            if(_fullRealizationsInOriginalSpace[i] > 0)
            {
                _fullRealizationsInOriginalSpace[i] += _pthreshObs;
            }
//            avg += _p[i] * _cpdfv[i];

            if((_fullRealizationsInOriginalSpace[i] == 0) && (!foundFirstPositiveValueAboveThreshold))
            {
                pz = _fullRealizationCDFValues[i];
            }
            if(_fullRealizationsInOriginalSpace[i] > _pthreshObs)
            {
//                cavg += _p[i] * _cpdfv[i];
//                ccv += _p[i] * _p[i] * _cpdfv[i];
                foundFirstPositiveValueAboveThreshold = true;
            }
        }

//The numbers computed here are not used in fortran
//        _genPoP = 1d - pz;
//        if(_genPoP > 0)
//        {
//            cavg = cavg / _genPoP;
//            ccv = Math.sqrt(Math.abs(ccv - cavg * cavg));
//        }
//        else
//        {
//            cavg = 0;
//            ccv = 0;
//        }
//
//        _genAvg = avg;
//        _genCAvg = cavg;
//        _genCCV = ccv;
//        _pMin = _p[0];
//        _pMax = _p[_p.length - 1];

        return pz;
    }

}
