package ohd.hseb.hefs.mefp.models.precipitation;

import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.math.stat.descriptive.summary.Sum;

public class IPTPrecipitationParameterCalculator
{
    public static final int MINIMUM_REQUIRED_OBSERVED = 20;

    //Required inputs (see constructor)
    private final double _precipThresholdForFcst;
    private final double _precipThresholdForObs;
    private final double _maximumAllowedAvgWhenPositive; //cavgobsx, cavgfcstx in the fcstparms, specified by one value, cavgmax, via epp3_pparest.

    //Results/output
    private double _avgObservation = Double.NaN; //=avgobs in epp3_fcstparms_v2.f Fortran subroutine
    private double _obsPoP = Double.NaN; //=popobs in epp3_fcstparms_v2.f Fortran subroutine
    private double _obsAverageWhenPositive = Double.NaN; //=cavgobs in epp3_fcstparms_v2.f Fortran subroutine
    private double _obsCoeffVarWhenPositive = Double.NaN; //=ccvobs in epp3_fcstparms_v2.f Fortran subroutine
    private double _avgForecast = Double.NaN; //=avgfcst in epp3_fcstparms_v2.f Fortran subroutine
    private double _fcstPoP = Double.NaN; //=popfcst in epp3_fcstparms_v2.f Fortran subroutine   
    private double _fcstAverageWhenPositive = Double.NaN; //=cavgfcst in epp3_fcstparms_v2.f Fortran subroutine  
    private double _fcstCoeffVarWhenPositive = Double.NaN; //=ccvfcst in epp3_fcstparms_v2.f Fortran subroutine  
    private double _recCoefOfCorrBtwnZFcstAndZObs = Double.NaN; //=rho in epp3_fcstparms_v2.f Fortran subroutine
    private int _numberOfObservations = -1;
    private int _numberOfBothPositive = -1;
    private String _questionableMessage = null;

//    private double _rMSDiffBtwnFcstAndObs; //=rmsfcst in epp3_fcstparms_v2.f Fortran subroutine
//    private double _NashSutcliffeEfficiencyOfFcst; //=effnfcst in epp3_fcstparms_v2.f Fortran subroutine
//    private double _RatioOfAvgObsToAvgFcst; //=ratio in epp3_fcstparms_v2.f Fortran subroutine
//    private double _equitableThreatScoreForWetDayFcst; //=eqts_fcst in epp3_fcstparms_v2.f Fortran subroutine
//    private double _corrCoefBtwnObsAndFcstInclZeros; //=cor in epp3_fcstparms_v2.f Fortran subroutine
//    private double _corrCoefBtwnWetObsAndFcst; //=ccor in epp3_fcstparms_v2.f Fortran subroutine
//    private double _corrCoefBtwnTransformedWetObsAndFcst; //=trccor in epp3_fcstparms_v2.f Fortran subroutine
//    private double _corrCoefFromTransformedFcstAndObs; //=rhoopt in epp3_fcstparms_v2.f Fortran subroutine

    public IPTPrecipitationParameterCalculator(final double precipThresholdForFcst,
                                               final double precipThresholdForObs,
                                               final double maximumAllowedAvgWhenPositive)
    {
        _precipThresholdForObs = precipThresholdForObs;
        _precipThresholdForFcst = precipThresholdForFcst;
        _maximumAllowedAvgWhenPositive = maximumAllowedAvgWhenPositive;
    }

    public String getQuestionableMessage()
    {
        return _questionableMessage;
    }

    /**
     * @return An array specifying the number of values greater than the corresponding thresholds,
     *         {@link #_precipThresholdForFcst} and {@link #_precipThresholdForObs}; [3] = both exceed the thresholds,
     *         [2] obs exceeds threshold, [1] fcst exceeds threshold, [0] neither.
     */
    public int[] computeCounts(final List<Float> forecastValues, final List<Float> observedValues)
    {
        final int[] counts = new int[4];
        Arrays.fill(counts, 0);
        for(int i = 0; i < observedValues.size(); i++)
        {
            if(observedValues.get(i) > _precipThresholdForObs)
            {
                if(forecastValues.get(i) > _precipThresholdForFcst)
                {
                    counts[3]++;
                }
                else
                {
                    counts[2]++;
                }
            }
            else
            {
                if(forecastValues.get(i) > _precipThresholdForFcst)
                {
                    counts[1]++;
                }
                else
                {
                    counts[0]++;
                }
            }
        }
        return counts;
    }

    public void calculate(final List<Float> forecastValues,
                          final List<Float> observedValues,
                          final int minimumRequiredNumberOfObservations)
    {
        if(forecastValues.size() != observedValues.size())
        {
            throw new IllegalArgumentException("IPT: The number of forecast event values does not equal the number of observed values.");
        }

        //I'm going to count the number of times when both obs and fcst are positive in the loop below.  In Fortran, this count is output
        //by the data gathering algorithm.  
        int numBothPos = 0;

        final int nobs = observedValues.size();
        final float[] adjForecastValues = new float[forecastValues.size()];
        final float[] adjObservedValues = new float[forecastValues.size()];
        for(int i = 0; i < nobs; i++)
        {
            if(observedValues.get(i) > _precipThresholdForObs)
            {
                adjObservedValues[i] = observedValues.get(i) - (float)_precipThresholdForObs;
            }
            else
            {
                adjObservedValues[i] = 0;
            }
            if(forecastValues.get(i) > _precipThresholdForFcst)
            {
                adjForecastValues[i] = forecastValues.get(i) - (float)_precipThresholdForFcst;
            }
            else
            {
                adjForecastValues[i] = 0;
            }

            if((observedValues.get(i) > _precipThresholdForObs) && (forecastValues.get(i) > _precipThresholdForFcst))
            {
                numBothPos++;
            }
        }

        //All data
        double avgobs = 0.0d;
        double stdobs = 0.0d;
        double avgfcst = 0.0d;
        double stdfcst = 0.0d;
        double correlationCoefficientAllData = 0.0d;
        double combinedCorrelation = 0.0d;
//      double rmsfcst = 0.0d;

        //Observed positive
        int nposobs = 0;
        double cavgobs = 0.0d;
        double cstdobs = 0.0d;

        //Forecast positive
        int nposfcst = 0;
        double cavgfcst = 0.0d;
        double cstdfcst = 0.0d;

        //Both positive
        int numberBothPositive = 0;
        double ccavgobs = 0.0d;
        double ccstdobs = 0.0d;
        double ccavgfcst = 0.0d;
        double ccstdfcst = 0.0d;
        double correlationBothPositive = 0.0d;

        double popobs = 0;
        double ccvobs = 1;
        double popfcst = 0;
        double ccvfcst = 1;
//        double cvobs = 1.0D;
//        double cvfcst = 1.0D;

        //If we have enough observations...
        if(nobs >= minimumRequiredNumberOfObservations)
        {
            //Loop over all data accumulating values that will later be turned into averages, cvs, etc.
            for(int i = 0; i < nobs; i++)
            {
                avgobs += observedValues.get(i);
                stdobs += observedValues.get(i) * observedValues.get(i);
                avgfcst += forecastValues.get(i);
                stdfcst += forecastValues.get(i) * forecastValues.get(i);
//                rmsfcst += (observedValues.get(i) - forecastValues.get(i))
//                    * (observedValues.get(i) - forecastValues.get(i));
                correlationCoefficientAllData += observedValues.get(i) * forecastValues.get(i);
                if(adjObservedValues[i] > 0)
                {
                    nposobs++;
                    cavgobs += adjObservedValues[i];
                    cstdobs += adjObservedValues[i] * adjObservedValues[i];
                }
                if(adjForecastValues[i] > 0)
                {
                    nposfcst++;
                    cavgfcst += adjForecastValues[i];
                    cstdfcst += adjForecastValues[i] * adjForecastValues[i];
                }
                if(adjObservedValues[i] > 0 && adjForecastValues[i] > 0)
                {
                    numberBothPositive++;
                    ccavgobs += adjObservedValues[i];
                    ccstdobs += adjObservedValues[i] * adjObservedValues[i];
                    ccavgfcst += adjForecastValues[i];
                    ccstdfcst += adjForecastValues[i] * adjForecastValues[i];
                    correlationBothPositive += adjObservedValues[i] * adjForecastValues[i];
                }
            }

            //Calculations over all sample =============================================
            //Compute needed values based on the accumulations.  The averages and standard deviations accumulations
            //are adjusted by sample size and means in order to get a final value. 
            //This is for the variables involving all samples:
            avgobs = avgobs / nobs;
            stdobs = Math.sqrt(Math.abs(stdobs / nobs - avgobs * avgobs));
//            cvobs = stdobs / avgobs;
            avgfcst = avgfcst / nobs;
            stdfcst = Math.sqrt(Math.abs(stdfcst / nobs - avgfcst * avgfcst));
//            cvfcst = stdfcst / avgfcst;
//            rmsfcst = Math.sqrt(rmsfcst / nobs);

            //Calculations over samples where both are positive =========================
            if(numberBothPositive > 0)
            {
                ccavgobs = ccavgobs / numberBothPositive;
                ccstdobs = Math.sqrt(Math.abs(ccstdobs / numberBothPositive - ccavgobs * ccavgobs));
                ccavgfcst = ccavgfcst / numberBothPositive;
                ccstdfcst = Math.sqrt(Math.abs(ccstdfcst / numberBothPositive - ccavgfcst * ccavgfcst));
            }

            //Correlation coefficients ==================================================
            //Adjust both correlation coefficients (over all data and only positive data) and compute the combined value
            //as the average.  The combined value will be returned as the correlation.  This was decided after conversations
            //with Limin in August/September 2012.
            int numberOfSummands = 0;
            if(stdfcst > 0 && stdobs > 0)
            {
                correlationCoefficientAllData = (correlationCoefficientAllData / nobs - avgobs * avgfcst)
                    / (stdfcst * stdobs);
                combinedCorrelation += correlationCoefficientAllData;
                numberOfSummands++;
            }
            else
            {
                correlationCoefficientAllData = 0.0d;
            }
            if((numberBothPositive > 0) && (ccstdfcst > 0 && ccstdobs > 0))
            {
                correlationBothPositive = (correlationBothPositive / numberBothPositive - ccavgobs * ccavgfcst)
                    / (ccstdfcst * ccstdobs);
                combinedCorrelation += correlationBothPositive;
                numberOfSummands++;
            }
            combinedCorrelation = combinedCorrelation / numberOfSummands;

            //Calculations when observed are positive =====================================
            if(nposobs > 0)
            {
                popobs = (double)nposobs / (double)nobs;
                if(nposobs > MINIMUM_REQUIRED_OBSERVED)
                {
                    cavgobs = cavgobs / nposobs;
                    cstdobs = Math.sqrt(cstdobs / nposobs - cavgobs * cavgobs);
                    ccvobs = cstdobs / cavgobs;
                }
                else
                {
                    _questionableMessage = "IPT: Not enough positive observations found; requires "
                        + MINIMUM_REQUIRED_OBSERVED + " but found" + nposobs + ".";
                    cavgobs = this._maximumAllowedAvgWhenPositive;
                    cstdobs = this._maximumAllowedAvgWhenPositive;
                    ccvobs = 1.0;
                }
//commented in fortran, too                if(cavgobs > 0 && cavgobs >= avgobs)
//                {
//                    popobs = avgobs / cavgobs; 
//                }
            }
            else
            {
                _questionableMessage = "IPT: No positive observations found.";
                cavgobs = _maximumAllowedAvgWhenPositive;
                cstdobs = _maximumAllowedAvgWhenPositive;
                popobs = 0.0D;
                ccvobs = 1.0D;
            }

            //Calculations when forecasts are positive =====================================
            if(nposfcst > 0)
            {
                popfcst = (double)nposfcst / (double)nobs;
                if(nposfcst > MINIMUM_REQUIRED_OBSERVED)
                {
                    cavgfcst = cavgfcst / nposfcst;
                    cstdfcst = Math.sqrt(cstdfcst / nposfcst - cavgfcst * cavgfcst);
                    ccvfcst = cstdfcst / cavgfcst;
                }
                else
                {
                    _questionableMessage = "IPT: Not enough positive forecasts found; requires "
                        + MINIMUM_REQUIRED_OBSERVED + " but found " + nposfcst + ".";
                    cavgfcst = this._maximumAllowedAvgWhenPositive;
                    cstdfcst = this._maximumAllowedAvgWhenPositive;
                    ccvfcst = 1.0;
                }
            }
            else
            {
                _questionableMessage = "IPT: No positive forecasts found.";
                cavgfcst = _maximumAllowedAvgWhenPositive;
                cstdfcst = _maximumAllowedAvgWhenPositive;
                popfcst = 0.0D;
                ccvfcst = 1.0D;
            }

//The conditional correlation -- this is already included above in a slightly different manner.
//            if(nccor > 0)
//            {
//                ccavgfcst = ccavgfcst / nccor;
//                ccstdfcst = Math.sqrt(Math.abs(ccstdfcst / nccor - ccavgfcst * ccavgfcst));
//                ccavgobs = ccavgobs / nccor;
//                ccstdobs = Math.sqrt(Math.abs(ccstdobs / nccor - ccavgobs * ccavgobs));
//                ccor = (ccor / nccor - ccavgobs * ccavgfcst) / (ccstdobs * ccstdfcst);
//            }
        }
        else
        {
            _questionableMessage = "Not enough observations for IPT parameter calculation.";
            cavgobs = _maximumAllowedAvgWhenPositive;
            cavgfcst = _maximumAllowedAvgWhenPositive;
        }

        this._avgForecast = avgfcst;
        this._avgObservation = avgobs;
        this._fcstAverageWhenPositive = cavgfcst;
        this._fcstCoeffVarWhenPositive = ccvfcst;
        this._fcstPoP = popfcst;
        this._obsAverageWhenPositive = cavgobs;
        this._obsCoeffVarWhenPositive = ccvobs;
        this._obsPoP = popobs;
        this._recCoefOfCorrBtwnZFcstAndZObs = combinedCorrelation;
        this._numberOfObservations = nobs;
        this._numberOfBothPositive = numBothPos;

        if(_recCoefOfCorrBtwnZFcstAndZObs < 0)
        {
            final DecimalFormat df = new DecimalFormat("0.######");
            _questionableMessage = "IPT: Negative combined correlation value computed: "
                + df.format(_recCoefOfCorrBtwnZFcstAndZObs) + ".";
        }
    }

    public String generateTestOutputString()
    {
        return "IPT Parameters: _avgObservation = " + _avgObservation + "; _obsPoP = " + _obsPoP
            + "; _obsAverageWhenPositive = " + _obsAverageWhenPositive + "; _obsCoeffVarWhenPositive = "
            + _obsCoeffVarWhenPositive + "; _avgForecast = " + _avgForecast + "; _fcstPoP = " + _fcstPoP
            + "; _fcstAverageWhenPositive = " + _fcstAverageWhenPositive + "; _fcstCoeffVarWhenPositive = "
            + _fcstCoeffVarWhenPositive + "; _recCoefOfCorrBtwnZFcstAndZObs = " + _recCoefOfCorrBtwnZFcstAndZObs
            + "; _numberOfObservations = " + _numberOfObservations + "; _numberOfPositiveObservations = "
            + _numberOfBothPositive + ".";
    }

    public void populateOneSetParameterValues(final PrecipitationOneSetParameterValues values)
    {
        values.setFcstAvg(_avgForecast);
        values.setFcstCondAvg(_fcstAverageWhenPositive);
        values.setFcstCondCoeffVar(_fcstCoeffVarWhenPositive);
        values.setFcstPoP(_fcstPoP);
        values.setFcstPrecipThresh(this._precipThresholdForFcst);
        values.setNumObs(_numberOfObservations);
        values.setNumBothPos(_numberOfBothPositive);
        values.setObsAvg(_avgObservation);
        values.setObsCondAvg(_obsAverageWhenPositive);
        values.setObsCondCoeffVar(_obsCoeffVarWhenPositive);
        values.setObsPoP(_obsPoP);
        values.setObsPrecipThresh(this._precipThresholdForObs);
        values.setRho(_recCoefOfCorrBtwnZFcstAndZObs);
    }

    /*
     * This is a rewrite of the original threshold() Fortran subroutine written by John Schaake in Aug 2008
     */
    public static double computePrecipitationThreshold(final double[] precipObserved,
                                                       final double fractionOfTotalPrecipGTZero)
    {
        //TODO: Advertised default for precipThresh (popfraction) is .98, but code says .97!

        //Copy the original values (before sorting), to avoid affecting the original
        final double[] sortedObservations = Arrays.copyOf(precipObserved, precipObserved.length);

        //The original Fortran sorted the numbers in ascending order (low to high) 
        Arrays.sort(sortedObservations); //sorts low to high (ascending)

        //compute sum total of all observations
        final double totalOfAllObsPrecip = new Sum().evaluate(sortedObservations);

        double precipitationThreshold = 0.0;
        double sumOfLargestToSmallestPrecipSoFar = 0.0;
        double fraction = 0.0;

        //Begin with largest precip observed and work down to smallest precip observed. 
        for(int i = sortedObservations.length - 1; i >= 0; i--)
        {
            sumOfLargestToSmallestPrecipSoFar = sumOfLargestToSmallestPrecipSoFar + sortedObservations[i];
            fraction = sumOfLargestToSmallestPrecipSoFar / totalOfAllObsPrecip; //Approaches 1
            //fractionOfTotalPrecipGTZero can be set in control file.  Otherwise it defaults to .97.
            //The actual threshold is the largest observed precip where 
            //   the (sumOfLargestToSmallestPrecipSoFar / totalOfAllObsPrecip) >= fractionOfTotalPrecipGTZero (from control file)
            if((fraction >= fractionOfTotalPrecipGTZero) && (precipitationThreshold == 0.0))
            {
                precipitationThreshold = sortedObservations[i];
            }
        }
        return precipitationThreshold;
    }

//Original Fortran code for the above:
//
//subroutine threshold (nobs,obs,pop_fraction,pthresh)
//c Calling Arguments:
//c  Name           Input/Output        Type       Description
//c  nobs             Input           integer*4   number of observations
//c  obs              Input           real*4      observations array
//c  pop_fraction     Input           real*4      fraction of total precip > 0
//c  pthresh          Output          real*4      precipitation threshold
//c      
//      parameter (maxobs=100000)
//      real*4 obs(nobs),sortobs(maxobs)
//c
//      pthresh = 0.
//      n = nobs
//      if (n.gt.maxobs) n = maxobs
//      do i=1,n
//        sortobs(i) = obs(i)
//      enddo
//     call sort (n,sortobs)
//      sum = 0.
//      do i=1,nobs
//       sum = sum + sortobs(i)
//      enddo
//      amt = 0.
//      do i=nobs,1,-1
//        amt = amt + sortobs(i)
//        fraction = amt/sum
//        if (fraction.ge.pop_fraction.and.
//     x      pthresh.eq.0.) then
//          pthresh = sortobs(i)
//        endif
//      enddo
//      return
//      end
//

}
//
// The original (Fortran) code upon which the above is based:
//  
//        subroutine epp3_fcstparms_v2 (ilog,nobs,pthresh1,pthresh2,
//     x                         obstran,fcsttran,
//     x                         obsx,fcstx,
//     x                         verstats,cor_weight,iopt_rho,
//     x                         cavgobsx,cavgfcstx,
//     x                         avgobs,popobs,cavgobs,ccvobs,
//     x                         avgfcst,popfcst,cavgfcst,ccvfcst,
//     x                         rho_est,rmsfcst,effnsfcst,
//     x                         ratio,eqts_fcst,
//     x                         cor,ccor,trccor,rhoopt,
//     x                         nvalcal,avgobscal,stdobscal,
//     x                         avgfcstcal,stdfcstcal)
//c
//c  analyze joint distribution to estimate parameters
//c
//c   INPUT ARGUMENTS:
//c     nobs     = number of observations = number of forecasts
//c     pthresh1 = precipitation threshold
//c     pthresh2 = forecast threshold
//c     obstran  = observations probability distribution
//c                  GAMM = Gamma
//c                  EXPO = Exponetial
//c                  LOGN = Log Normal
//c                  WEIB = Weibull
//c     fcsttran = forecast probability distribution
//c                  GAMM = Gamma
//c                  EXPO = Exponetial
//c                  LOGN = Log Normal
//c                  WEIB = Weibull
//c     obsx     = observed values
//c     fcstx    = deterministic forecasts
//c     verstats = verification statistics switch (compute if 'yes')
//c
//c   OUTPUT ARGUMENTS:
//c     avgobs     = average obervation
//c     popobs     = climatological probability of obs > pthreshold
//c     cavgobs    = climatological mean of obs when obs > pthreshold
//c     ccvobs     = climatological cv of obs when obs > pthreshold
//c     avgfcst    = average forecast
//c     popfcst    = climatological probability of fcst > pthreshold
//c     cavgfcst   = climatological mean of fcst when fcst > pthreshold
//c     ccvfcst    = climatological cv of fcst when fcst > pthreshold
//c     rho        = recommended coefficient of correlation between z(fcst) and z(obs)
//c     rmsfcst    = rms difference between fcst and obs
//c     effnfcst   = Nash-Sutcliffe efficiency of forecast
//c     ratio      = avgobs/avgfcst
//c     eqts_fcst  = equitable threat score for wet day forecast
//c     cor        = correlation coefficient between obs and fcst (includes zeros)
//c     ccor       = correlation coefficient between wet obs and fcsts
//c     trccor     = correlation coefficient between transformed wet obs and fcsts
//c     rhoopt     = correlation coefficient from transformed forecasts and obs
//c
//c     20110623 Ward Using the ddd debugger, the CFSv1 correlation coefficient,
//c     cor, changed when we stepped over an 'else' statement
//c     -> no value was modified. The .gdbinit used was:
//c
//c        b epp3_fcstparms.f:351
//c        condition 1 cor=-0.15
//c        b epp3_fcstparms.f:363
//c        condition 2 rho_est=-0.15
//c        b epp3_fcstparms.f:314
//c        condition 3 effnsfcst=0.704644322
//c
//c     However, cor was stable the entire run when Varalakshmi and I put
//c     in print statements to track it. So either the debugger is buggy,
//c     or some odd memory thing is going on in the code. Varalakshmi ran
//c     valgrind on the code, and detected no memory leaks. I tried moving
//c     the correlation coefficient into a common block, but the value still
//c     changed mysteriously in the common block when I stepped over the else
//c     statement. The value didn't change when I compared it right after
//c     it was calculated and after the return in the calling subroutine.
//c
//c     Since we are recoding to Java, a leak (if it exists) should go away,
//c     but it bears keeping an eye on.
//c
//      parameter (maxobs=50000)
//      character*8 obstran,fcsttran
//      character*3 verstats
//      real*4 obs(maxobs),fcst(maxobs),pval(2,2)
//      real*4 obsx(maxobs),fcstx(maxobs)
//      integer*4 nvalcal(2,2),nvalval(2,2)
//      real*4 avgobscal(2,2),stdobscal(2,2)
//      real*4 avgfcstcal(2,2),stdfcstcal(2,2)
//c     20110623 Ward Deleted unused variables
//c     real*4 stdobsval(2,2),avgobsval(2,2)
//c     real*4 stdensval(2,2),avgensval(2,2)
//c     real*4 obsxx(maxobs),fcstxx(maxobs),ensfcstx(maxobs)
//c
//      minobs = 20
//c
//      avgobs = 0.
//      stdobs = 0.
//      avgfcst = 0.
//      stdfcst = 0.
//      cor = 0.
//      nobspx = 0
//      cavgobs = 0.
//      cstdobs = 0.
//      nfcstpx = 0.
//      cavgfcst = 0.
//      cstdfcst = 0.
//      ccor = 0.
//      ccavgobs = 0.
//      ccstdobs = 0.
//      ccavgfcst = 0.
//      ccstdfcst = 0.
//      nccor = 0
//      rmsfcst = 0.
//      qscore = 0.
//      rho = 0.
//      eff = 0.
//      ratio = 0.
//      if (obstran(1:4).eq.'WEIB'.or.
//     x    obstran(1:4).eq.'weib') then
//        iobstran = 5
//      elseif (obstran(1:4).eq.'GAMM'.or.
//     x        obstran(1:4).eq.'gamm') then
//        iobstran = 1
//      elseif (obstran(1:4).eq.'LOGN'.or.
//     x        obstran(1:4).eq.'logn') then
//        iobstran = 2
//      elseif (obstran(1:4).eq.'EXPO'.or.
//     x        obstran(1:4).eq.'expo') then
//        iobstran = 3
//      else
//        iobstran = 0
//      endif
//      if (fcsttran(1:4).eq.'WEIB'.or.
//     x    fcsttran(1:4).eq.'weib') then
//        ifcsttran = 5
//      elseif (fcsttran(1:4).eq.'GAMM'.or.
//     x        fcsttran(1:4).eq.'gamm') then
//        ifcsttran = 1
//      elseif (fcsttran(1:4).eq.'LOGN'.or.
//     x        fcsttran(1:4).eq.'logn') then
//        ifcsttran = 2
//      elseif (fcsttran(1:4).eq.'EXPO'.or.
//     x        fcsttran(1:4).eq.'expo') then
//        ifcsttran = 3
//      else
//        ifcsttran = 0
//      endif
//c
//      do i=1,nobs
//        if (obsx(i).gt.pthresh1) then
//          obs(i) = obsx(i) - pthresh1
//        else
//          obs(i) = 0.
//        endif
//        if (fcstx(i).gt.pthresh2) then
//          fcst(i) = fcstx(i) - pthresh2
//        else
//          fcst(i) = 0.
//        endif
//      enddo
//c
//      if (nobs.ge.minobs) then
//        do i=1,nobs
//          avgobs = avgobs + obsx(i)
//          stdobs = stdobs + obsx(i)**2
//          avgfcst = avgfcst + fcstx(i)
//          stdfcst = stdfcst + fcstx(i)**2
//          rmsfcst = rmsfcst + (obsx(i)-fcstx(i))**2
//          cor = cor + obsx(i)*fcstx(i)
//          if (obs(i).gt.0.) then
//            nobspx = nobspx + 1
//            cavgobs = cavgobs + obs(i)
//            cstdobs = cstdobs + obs(i)**2
//          endif
//          if (fcst(i).gt.0.) then
//            nfcstpx = nfcstpx + 1
//            cavgfcst = cavgfcst + fcst(i)
//            cstdfcst = cstdfcst + fcst(i)**2
//          endif
//          if (obs(i).gt.0..and.fcst(i).gt.0.) then
//            nccor = nccor + 1
//            ccavgobs = ccavgobs + obs(i)
//            ccstdobs = ccstdobs + obs(i)**2
//            ccavgfcst = ccavgfcst + fcst(i)
//            ccstdfcst = ccstdfcst + fcst(i)**2
//            ccor = ccor + obs(i)*fcst(i)
//          endif
//        enddo
//        avgobs = avgobs/nobs
//        stdobs = sqrt(abs(stdobs/nobs - avgobs**2))
//        cvobs = stdobs/avgobs
//        avgfcst = avgfcst/nobs
//        stdfcst = sqrt(abs(stdfcst/nobs-avgfcst**2))
//        cvfcst = stdfcst/avgfcst
//        rmsfcst = sqrt(rmsfcst/nobs)
//        bias = (avgfcst - avgobs)/((avgfcst+avgobs)/2.)
//        effnsfcst = 1. - rmsfcst**2/stdobs**2
//        if (stdfcst.gt.0..and.stdobs.gt.0.) then
//          cor = (cor/nobs-avgobs*avgfcst)/(stdfcst*stdobs)
//          ratio = avgobs/avgfcst
//        else
//          cor = 0.
//          ratio = 0.
//        endif
//c
//        if (nobspx.gt.0) then
//          popobs = float(nobspx)/float(nobs)
//          if (nobspx.gt.20) then
//            cavgobs = cavgobs/nobspx
//            cstdobs = sqrt(abs(cstdobs/nobspx-cavgobs**2))
//            ccvobs = cstdobs/cavgobs
//            cavgobsx = cavgobs
//          else
//            cavgobs = cavgobsx
//            cstdobs = cavgobsx
//            ccvobs = 1.
//          endif
//c          if (cavgobs.gt.0..and.cavgobs.ge.avgobs) then
//c            popobs = avgobs/cavgobs
//c          endif
//        else
//          popobs = 0.
//          cavgobs = cavgobsx
//          cstdobs = cavgobsx
//          ccvobs = 1.
//        endif
//c
//        if (nfcstpx.gt.0) then
//          popfcst = float(nfcstpx)/float(nobs)
//          if (nfcstpx.gt.20) then
//            cavgfcst = cavgfcst/nfcstpx
//            cstdfcst = sqrt(abs(cstdfcst/nfcstpx-cavgfcst**2))
//            ccvfcst = cstdfcst/cavgfcst
//            cavgfcstx = cavgfcst
//          else
//            cavgfcst = cavgfcstx
//            cstdfcst = cavgfcstx
//            ccvfcst = 1.
//          endif
//        else
//          cavgfcst = cavgfcstx
//          cstdfcst = cavgfcstx
//          ccvfcst = 1.
//          popfcst = 0.
//        endif
//c
//        if (nccor.gt.0) then
//          ccavgfcst = ccavgfcst/nccor
//          ccstdfcst = sqrt(abs(ccstdfcst/nccor-ccavgfcst**2))
//          ccavgobs = ccavgobs/nccor
//          ccstdobs = sqrt(abs(ccstdobs/nccor-ccavgobs**2))
//          ccor = (ccor/nccor - ccavgobs*ccavgfcst)/
//     x           (ccstdobs*ccstdfcst)
//        endif
//c
//c       ...compute contingency table...
//c
//        do i=1,2
//          do j=1,2
//            nvalcal(i,j) = 0
//            avgobscal(i,j) = 0.
//            stdobscal(i,j) = 0.
//            avgfcstcal(i,j) = 0.
//            stdfcstcal(i,j) = 0.
//          enddo
//        enddo
//        do k=1,nobs
//          if (fcstx(k).gt.pthresh2) then
//            i = 2
//          else
//            i = 1
//          endif
//          if (obsx(k).gt.pthresh1) then
//            j = 2
//          else
//            j = 1
//          endif
//          nvalcal(i,j) = nvalcal(i,j) + 1
//          avgobscal(i,j) = avgobscal(i,j) + obsx(k)
//          stdobscal(i,j) = stdobscal(i,j) + obsx(k)**2
//          avgfcstcal(i,j) = avgfcstcal(i,j) + fcstx(k)
//          stdfcstcal(i,j) = stdfcstcal(i,j) + fcstx(k)**2
//        enddo
//        do i=1,2
//          do j=1,2
//            if (nvalcal(i,j).gt.0.) then
//              avgobscal(i,j) = avgobscal(i,j)/nvalcal(i,j)
//              stdobscal(i,j) = sqrt(abs(stdobscal(i,j)/nvalcal(i,j) -
//     x                                  avgobscal(i,j)**2))
//              avgfcstcal(i,j) = avgfcstcal(i,j)/nvalcal(i,j)
//              stdfcstcal(i,j) = sqrt(abs(stdfcstcal(i,j)/nvalcal(i,j) -
//     x                                  avgfcstcal(i,j)**2))
//            endif
//          enddo
//        enddo
//        ndp = 0
//        do i=1,2
//          do j=1,2
//            ndp = ndp + nvalcal(i,j)
//          enddo
//        enddo
//        do i=1,2
//          do j=1,2
//            pval(i,j) = float(nvalcal(i,j))/float(ndp)
//          enddo
//        enddo
//        a = nvalcal(2,2)
//        b = nvalcal(2,1)
//        c = nvalcal(1,2)
//        d = nvalcal(1,1)
//        exp_h = ((a+b)*(a+c))/(a+b+c+d)
//        eqts_fcst = (a - exp_h)/(a + b + c - exp_h)
//c
//c       ...transform obs (Weibull) and ens mean (gamma distr) to snd
//c
//        trccor = 0.
//        trcavg = 0.
//        trcstd = 0.
//        trcavgobs = 0.
//        trcstdobs = 0.
//        travg = 0.
//        travgobs = 0.
//        np = 0
//c
//        if (cor.lt.0.9995) then
//          pzobs = 1. - popobs
//          pzfcst = 1. - popfcst
//          do i=1,nobs
//            if (obs(i).gt.0.) then
//              if (fcst(i).gt.0.) then
//                trobs = vtrans(cavgobs,ccvobs,pzobs,obs(i),iobstran)
//                trfcst = vtrans(cavgfcst,ccvfcst,pzfcst,fcst(i),
//     x                          ifcsttran)
//                trcavg = trcavg + trfcst
//                trcstd = trcstd + trfcst**2
//                trcavgobs = trcavgobs + trobs
//                trcstdobs = trcstdobs + trobs**2
//                trccor = trccor + trfcst*trobs
//                np = np + 1
//              endif
//            endif
//          enddo
//          if (np.gt.minobs) then
//            trcavg = trcavg/np
//            trcstd = sqrt(abs(trcstd/np - trcavg**2))
//            trcavgobs = trcavgobs/np
//            trcstdobs = sqrt(abs(trcstdobs/np - trcavgobs**2))
//            trccor = (trccor/np - trcavg*trcavgobs)/
//     x               (trcstd*trcstdobs)
//          else
//            trccor = cor
//          endif
//          rho22=trccor
//          if (pzobs.lt.0.10.or.
//     x        pzobs.gt.0.90.or.
//     x        pzfcst.lt.0.10.or.
//     x        pzfcst.gt.0.90) then
//            rhoopt = cor
//          else
//            if (pval(1,1).lt.0.05.or.
//     x          pval(1,1).gt.0.95.or.
//     x          pval(2,2).lt.0.05.or.
//     x          pval(2,2).gt.0.95) then
//              rhoopt = cor
//            else
//              call estrho3(iout,pval(1,1),pval(1,2),pval(2,1),
//     x                     pval(2,2),rho22,rhoopt)
//            endif
//          endif
//        else
//          ccor = cor
//          trccor = cor
//          rhoopt = cor
//        endif
//      else
//          cavgobs = cavgobsx
//          cavgfcst = cavgfcstx
//      endif
//      if (iopt_rho.eq.1) then
//        rho_est = cor
//      elseif (iopt_rho.eq.2) then
//        rho_est = rhoopt
//      elseif (iopt_rho.eq.3) then
//        rho_est = cor_weight*cor + (1. - cor_weight)*rhoopt
//      elseif (iopt_rho.eq.4) then
//        rho_est = ccor
//      elseif (iopt_rho.eq.5) then
//        rho_est = trccor
//      else
//        rho_est = cor
//      endif
//      return
//      end