package ohd.hseb.hefs.mefp.models.precipitation;

import java.util.EnumSet;

import ohd.hseb.hefs.utils.dist.DataFittingDistribution;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.dist.types.GammaDist;
import ohd.hseb.hefs.utils.dist.types.WeibullDist;
import ohd.hseb.hefs.utils.plugins.DefaultFactoryException;
import ohd.hseb.util.data.DataSet;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
/**
 * Tools that may be used by either the parameter estimator or ensemble generator.
 * 
 * @author Hank.Herr
 */
public abstract class MEFPPrecipitationModelTools
{
    private static final Logger LOG = LogManager.getLogger(MEFPPrecipitationModelTools.class);

    /**
     * Used to determine the fixed shift parameter to use such that precip values equal to {@link #_fixedZeroThreshold}
     * which are treated as positive are larger than the distribution shift (if not, the commons normal dist returns
     * -INFINITY, which is bad, when the original space values are equal to the shift thereby having 0 probability).
     * Specifically, the shift will be equal to the fixed zero - this value.
     */
    protected static final double DISTRIBUTION_SHIFT_ADJUSTMENT_FACTOR = 0.001D;

    /**
     * @return An array of the {@link DistributionType} choices.
     */
    public static DistributionType[] getSelectableDistributionTypes()
    {
        return getSelectableDistributionTypesEnumSet().toArray(new DistributionType[4]);
//        return new DistributionType[]{DistributionType.WEIBULL, DistributionType.GAMMA, DistributionType.WEIBULL3,
//            DistributionType.PEARSON_TYPE_3, DistributionType.LOGLOGISTIC, DistributionType.LOGNORMAL,
//            DistributionType.LOGWEIBULL, DistributionType.PARETO, DistributionType.WAKEBY};
    }

    /**
     * @return An {@link EnumSet} of the {@link DistributionType} choices.
     */
    public static EnumSet<DistributionType> getSelectableDistributionTypesEnumSet()
    {
        return EnumSet.of(DistributionType.WEIBULL,
                          DistributionType.GAMMA,
                          DistributionType.PEARSON_TYPE_3,
                          DistributionType.WEIBULL3);
    }

    /**
     * @param fitData Data to fit with the fit attributes set. Make sure that the fit variables are specified; those
     *            numbers returned by {@link DataSet#getFitCDFVariable()} and {@link DataSet#getFitSampleVariable()}
     *            must not be negative. number returned by {@link DataSet#getFitSampleVariable()}.
     * @param distType Int: 10 for Pearson Type III (Gamma with shift) and 12 for Weibull.
     * @return An AbstractProbabilityDistribution fit to the fitData for the distribution specified by type.
     * @throws Exception If fitting fails for whatever reason.
     */
    public static ContinuousDist fitDistribution(final DataSet fitData,
                                                 final DistributionType distType,
                                                 final Double fixedZeroThreshold) throws Exception
    {
        ContinuousDist dist = null;

        //Weibull ================================================================================================
        if((distType == DistributionType.WEIBULL) || (distType == DistributionType.WEIBULL3))
        {
            final WeibullDist fitDist = new WeibullDist();
            fitDist.setFitShift(distType == DistributionType.WEIBULL3);
            fitDist.setShift(fixedZeroThreshold - DISTRIBUTION_SHIFT_ADJUSTMENT_FACTOR);
            try
            {
                fitDist.fitToData(fitData);
            }
            catch(final Throwable t)
            {
                throw new Exception("Error fitting Weibull three-parameter distribution: ");
            }
            dist = fitDist;
        }

        //Gamma ===================================================================================================
        else if((distType == DistributionType.GAMMA) || (distType == DistributionType.PEARSON_TYPE_3))
        {
            final GammaDist fitDist = new GammaDist();
            fitDist.setFitShift(distType == DistributionType.PEARSON_TYPE_3);
            fitDist.setShift(fixedZeroThreshold - DISTRIBUTION_SHIFT_ADJUSTMENT_FACTOR);

            //This uses method of L-moments, which matches the scientific prototype.  If a failure occurs, it attempts to use moments, instead.
            //If that fails, and exception is thrown.
            try
            {
                fitDist.fitToData(fitData);
            }
            catch(final Throwable t)
            {
                LOG.debug("L-moments fitting for Gamma failure encountered; fitting using moments.");

                //Moments based Gamma.
                try
                {
                    fitDist.estimateMethodOfMoments(fitData);
                }
                catch(final Throwable tt)
                {
                    throw new Exception("Error fitting Pearson Type III (Gamma three-parameter) distribution using both L-moments and moments: "
                        + tt.getMessage());
                }
            }
            dist = fitDist;
        }

        //All others ===============================================================================================
        else
        {
            try
            {
                //Instantiate the distribution, record it, and fit it to the cdfDataSet.
                dist = (ContinuousDist)DistributionType.instantiateDistribution(distType);
                if(fixedZeroThreshold != null)
                {
                    ((DataFittingDistribution)dist).fitToData(fitData, new double[]{0.0D, fixedZeroThreshold});
                }
                else
                {
                    ((DataFittingDistribution)dist).fitToData(fitData, new double[]{0.0D});
                }
            }
            catch(final DefaultFactoryException e)
            {
                throw new IllegalStateException("Instantiating a distributin with known type should never cause a problem, but it did!",
                                                e);
            }
        }

        return dist;
    }

    /**
     * Used internally, it maps distributions to the types actually used, operationally. Specifically, the Weibull is
     * always a three-parameter version as is the Gamma so that the fixed shift is recorded in the parameter file and
     * pulled from it.
     * 
     * @param type The type of distribution to instantiate.
     * @param parameters The parameters.
     * @return A {@link ContinuousDist} instantiated via
     *         {@link DistributionType#instantiateDistribution(DistributionType, Number[])}.
     * @throws DefaultFactoryException
     */
    protected static ContinuousDist instantiateOperationalDistribution(final DistributionType type,
                                                                       final Double[] parameters) throws DefaultFactoryException
    {
        if(type == DistributionType.WEIBULL3)
        {
            final WeibullDist dist = new WeibullDist();
            dist.setParameters(parameters);
            dist.setFitShift(true);
            return dist;
        }
        else if(type == DistributionType.PEARSON_TYPE_3)
        {
            final GammaDist dist = new GammaDist();
            dist.setParameters(parameters);
            dist.setFitShift(true);
            return dist;
        }

        return (ContinuousDist)DistributionType.instantiateDistribution(type, parameters);
    }

}
