package ohd.hseb.hefs.mefp.models.precipitation;

import java.util.HashMap;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.ForecastEnsembleCalculator;
import ohd.hseb.hefs.mefp.models.MEFPEnsembleGeneratorModel;
import ohd.hseb.hefs.mefp.models.SchaakeShuffleApplier;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;

/**
 * Basic precipitation ensemble generation model using the {@link MEFPEnsembleGeneratorModel} superclass.
 * 
 * @author hankherr
 */
public class PrecipitationEnsembleGenerationModel extends MEFPEnsembleGeneratorModel
{
    private static final Logger LOG = LogManager.getLogger(PrecipitationEnsembleGenerationModel.class);
    public final static ParameterId[] PROCESSED_DATA_TYPES = {ParameterId.FMAP};
    public final static double COND_COEFF_VAR_LIMIT = 2.5d;

    /**
     * This serves as a maximum on the conditional coefficient of variation in the parameters. Can this be put in the
     * super class? --- no... I don't see it used in temperature model.
     */
    private double _condCoeffVarMax = 2.0;

    /**
     * This flag indicates if the EPT ensemble generation should be called.
     */
    private final HashMap<MEFPForecastSource, Boolean> _sourceToUseEPTMap = new HashMap<MEFPForecastSource, Boolean>();

    /**
     * Indicates if log messages related to sampling should be included for the ensemble generator
     * calculators.  This flag will be passed through.
     */
//    private boolean _includeSamplingLogMessages = false;
    private Double _eptThresholdToIncludeSamplingLogMessages = null;
    

    @Override
    protected void initializeDataTypeSpecificParameters()
    {
        //Checking the conditional coeff of var max setting.
        if(_condCoeffVarMax == 0)//TODO Should an upper check be done as well: || (_condCoeffVarMax > COND_COEFF_VAR_LIMIT))
        {
            LOG.warn("The maximum allowed conditional coefficient of variation was set to 0, which is not allowed.  Setting it to its max allowed, "
                + COND_COEFF_VAR_LIMIT);
            _condCoeffVarMax = COND_COEFF_VAR_LIMIT;
        }
    }

    @Override
    protected int getCanonicalEventPeriodStepSizeInHours()
    {
        return CanonicalEvent.determineCanonicalEventPeriodUnitInHours(true);
    }

    @Override
    protected ForecastEnsembleCalculator instantiateCalculator(final MEFPForecastSource source, final int dayOfYear) throws Exception
    {
        //Prep the calculator. Note that the EPT needs to gather canonical events and its easier to gather all of the
        //events one time, as is done in the constructor for all events for which source parameters were estimated.  Note that this will more
        //setup work than necessary, since it will be prepped to gather any event for which parameters were estimated, not just those for which
        //ensemble generation will be run (which may be fewer depending on the number of forecast days).  However, that work is just setup and
        //should not add signficant computation time.
        ForecastEnsembleCalculator ensCalculator = null;
        if(useEPT(source))
        {
            ensCalculator = new EPTPrecipitationForecastEnsembleCalculator(source,
                                                                           getModelParameters(),
                                                                           dayOfYear,
                                                                           getModelParameters().getSourceModelParameters(source)
                                                                                               .getPrecipitationSourceEventValues(),
                                                                           getStratifiedSampling(),
                                                                           getEPTThresholdToIncludeSamplingLogMessages());
        }
        else
        {
            ensCalculator = new IPTPrecipitationForecastEnsembleCalculator(source, getModelParameters(), dayOfYear);
        }
        return ensCalculator;
    }

    @Override
    protected void checkAndModifyFullModelParameters(final MEFPForecastSource source,
                                                     final CanonicalEvent event,
                                                     final int dayOfYear)
    {
        //Acquire the one-day parameters for the given forecastTime, source, and event.  Modify the conditional coefficients
        //of variation if necessary due to exceeding the max allowed value, and record the parameters back into the full model parameters.
        final PrecipitationOneSetParameterValues parameters = (PrecipitationOneSetParameterValues)getModelParameters().getSourceEstimatedModelParameterValues(source,
                                                                                                                                                              dayOfYear,
                                                                                                                                                              event);
        if(parameters.getObsCondCoeffVar() > _condCoeffVarMax)
        {
            parameters.setObsCondCoeffVar(_condCoeffVarMax);
        }
        if(parameters.getFcstCondCoeffVar() > _condCoeffVarMax)
        {
            parameters.setFcstCondCoeffVar(_condCoeffVarMax);
        }
        getModelParameters().recordValuesFrom(parameters);
    }

    @Override
    protected double getRhoValue(final MEFPForecastSource source, final int dayOfYear, final CanonicalEvent evt0)
    {
        double rho = getModelParameters().getSourceModelParameters(source).getPrecipitationRhoParameterValue(dayOfYear,
                                                                                                             evt0);
        if(useEPT(source))
        {
            rho = getModelParameters().getSourceModelParameters(source).getEPTPrecipitationRhoParameterValue(dayOfYear,
                                                                                                             evt0);
        }
        return rho;
    }

    @Override
    protected TimeSeriesArray getSingleHistoricalTimeSeries()
    {
        return getModelParameters().getSingleHistoricalTimeSeries();
    }
    
    public Double getEPTThresholdToIncludeSamplingLogMessages()
    {
        return _eptThresholdToIncludeSamplingLogMessages;
    }
    
    /**
     * @param includingSamplingLogMessages Null is allowed and will result in no messages.  Otherwise, the number must be between 0 and 1 and
     * is the threshold above which messages will be presented.
     */
    public void setEPTThresholdToIncludeSamplingLogMessages(final Double eptThresholdToIncludeSamplingLogMessages)
    {
            _eptThresholdToIncludeSamplingLogMessages = eptThresholdToIncludeSamplingLogMessages;
    }
    
    public double getCondCoeffVarMax()
    {
        return _condCoeffVarMax;
    }

    public void setCondCoeffVarMax(final double condCoeffVarMax)
    {
        _condCoeffVarMax = condCoeffVarMax;
    }

    /**
     * @param source The source for which to check if EPT is to be used.
     * @return True if {@link #setUseEPT(MEFPForecastSource, boolean)} was called for the source and given true AND if
     *         it is valid for the EPT to be applied to the source based on
     *         {@link MEFPForecastSource#canEPTModelBeUsedForSource()}.
     */
    public boolean useEPT(final MEFPForecastSource source)
    {
        final Boolean b = _sourceToUseEPTMap.get(source);
        if(b == null)
        {
            return false;
        }
        return b && source.canEPTModelBeUsedForSource();
    }

    public void setUseEPT(final MEFPForecastSource source, final boolean useEPT)
    {
        _sourceToUseEPTMap.put(source, useEPT);
    }

    @Override
    public ParameterId[] getProcessedDataTypes()
    {
        return PROCESSED_DATA_TYPES;
    }

    @Override
    protected ParameterId determineForecastEnsembleParameterId(final String baseEnsembleHistoricalParameterId)
    {
        if(!ParameterId.valueOf(baseEnsembleHistoricalParameterId).isPrecipitation())
        {
            throw new IllegalArgumentException("Base ensemble parameter id '" + baseEnsembleHistoricalParameterId
                + "' is not a valid precipitation parameter id.");
        }
        return ParameterId.FMAP;
    }

    @Override
    public ParameterId checkForValidityAndConvertToProcessedType(final String inputParameterStr) throws Exception
    {
        //Check...
        if(!ParameterId.valueOf(inputParameterStr).isPrecipitation())
        {
            throw new IllegalArgumentException("Parameter id of '" + inputParameterStr
                + "' is not a valid precipitation parameter.");
        }

        //Convert...
        if(!ParameterId.valueOf(inputParameterStr).equals(ParameterId.FMAP))
        {
            LOG.warn("Provided parameter id " + inputParameterStr + " is being treated as " + ParameterId.FMAP
                + " the purposes of running this model.");
        }
        return ParameterId.FMAP;
    }

    @Override
    public void checkForValidityAndConvertUnits(final TimeSeriesArray inputTS) throws Exception
    {
        //Null unit is treated as MM.  So if its null, it better be in metric!
        if(inputTS.getHeader().getUnit() == null)
        {
            ((DefaultTimeSeriesHeader)inputTS.getHeader()).setUnit("MM");
        }
        //MeasuringUnit does not recognize the unit:
        else if(MeasuringUnit.getMeasuringUnit(inputTS.getHeader().getUnit()) == null)
        {
            throw new Exception("Unit string, '" + inputTS.getHeader().getUnit() + "', is not recognized internally.");
        }
        //Unit is invalid:
        else if(!MeasuringUnit.getMeasuringUnit(inputTS.getHeader().getUnit())
                              .getType()
                              .equals(MeasuringUnitType.length))
        {
            throw new Exception("Units of time series must be length (mm, in, etc), but was "
                + inputTS.getHeader().getUnit() + ".");
        }
        TimeSeriesArrayTools.convertUnits(inputTS, "mm");
    }

    /**
     * @return The precipitation estimation control options.
     */
    private MEFPPrecipitationModelControlOptions getEstimationControlOptions()
    {
        return (MEFPPrecipitationModelControlOptions)getModelParameters().getEstimationControlOptions()
                                                                         .getModelParameters();
    }

    @Override
    protected SchaakeShuffleApplier constructSchaakeShuffleApplier(final TimeSeriesEnsemble baseEnsemble)
    {
        final double noRainThresh = getEstimationControlOptions().getEPT().getEPTPrecipThreshold();
        return new SchaakeShuffleApplier(baseEnsemble, true, noRainThresh);
    }
}
