package ohd.hseb.hefs.mefp.models.precipitation;

import org.apache.commons.math.distribution.ContinuousDistribution;

import ohd.hseb.hefs.mefp.models.parameters.types.RhoParameterType;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneSetParameterValues;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.dist.types.ContinuousDist;
import ohd.hseb.hefs.utils.plugins.DefaultFactoryException;

/**
 * This class MUST be kept in sync with the list of parameters in the {@link PrecipitationParameterEstimationModel}. I
 * have added sets and gets to allow for directly gathering parameters instead of having to know which
 * {@link ModelParameterType} to pass in. The sets/gets are based on the order of the parameters!
 * 
 * @author hank.herr
 */
public class PrecipitationOneSetParameterValues extends OneSetParameterValues
{
    /**
     * @param dayOfYear The day of the year to associate with the parameters.
     * @param canonicalEventIndex The index in a list of canonical events maintained outside of this class.
     * @param model The {@link ParameterEstimationModel} for which these parameters apply.
     * @param source The {@link MEFPForecastSource} for which these parameters apply.
     */
    public PrecipitationOneSetParameterValues(final int dayOfYear,
                                              final int canonicalEventIndex,
                                              final ParameterEstimationModel model,
                                              final MEFPForecastSource source)
    {
        super(dayOfYear, canonicalEventIndex, model, source);
    }

    public boolean areaIPTParametersMissing()
    {
        return getNumObs() < 0;
    }

    public boolean areEPTParametersMissing()
    {
        return Double.isNaN(getEPTRho());
    }

    public void copyIPTParameters(final PrecipitationOneSetParameterValues base)
    {
        this.setFcstAvg(base.getFcstAvg());
        this.setFcstCondAvg(base.getFcstCondAvg());
        this.setFcstCondCoeffVar(base.getFcstCondCoeffVar());
        this.setFcstPoP(base.getFcstPoP());
        this.setFcstPrecipThresh(base.getFcstPrecipThresh());

        this.setNumObs(base.getNumObs());
        this.setNumBothPos(base.getNumBothPos());

        this.setObsAvg(base.getObsAvg());
        this.setObsCondAvg(base.getObsCondAvg());
        this.setObsCondCoeffVar(base.getObsCondCoeffVar());
        this.setObsPoP(base.getObsPoP());
        this.setObsPrecipThresh(base.getObsPrecipThresh());

        this.setRho(base.getRho());
    }

    public void copyEPTParameters(final PrecipitationOneSetParameterValues base)
    {

        this.setEPTBivarProbNoPrecip(base.getEPTBivarProbNoPrecip());
        for(int i = 0; i < 4; i++)
        {
            this.setEPTFcstBivarMarginalDistParm(i, base.getEPTFcstBivarMarginalDistParm(i));
        }
        for(int i = 0; i < 4; i++)
        {
            this.setEPTFcstIntermittencyDistParm(i, base.getEPTFcstIntermittencyDistParm(i));
        }
        this.setEPTFcstPoPObsZero(base.getEPTFcstPoPObsZero());
        for(int i = 0; i < 4; i++)
        {
            this.setEPTFcstZeroObsDistParm(i, base.getEPTFcstZeroObsDistParm(i));
        }
        this.setEPTMu(base.getEPTMu());
        for(int i = 0; i < 4; i++)
        {
            this.setEPTObsBivarMarginalDistParm(i, base.getEPTObsBivarMarginalDistParm(i));
        }
        this.setEPTObsPopFcstZero(base.getEPTObsPopFcstZero());
        for(int i = 0; i < 4; i++)
        {
            this.setEPTObsZeroFcstDistParm(i, base.getEPTObsZeroFcstDistParm(i));
        }
        this.setEPTRho(base.getEPTRho());
        this.setEPTSig(base.getEPTSig());
        this.setEPTXmin(base.getEPTXmin());
    }

    //ORDER FROM THE MODEL
    //0    parameters.add(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
    //1    parameters.add(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
    //2    parameters.add(new PThreshParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
    //3    parameters.add(new POPParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
    //4    parameters.add(new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
    //5    parameters.add(new CCVParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
    //6    parameters.add(new PThreshParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
    //7    parameters.add(new POPParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
    //8    parameters.add(new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
    //9    parameters.add(new CCVParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
    //10    parameters.add(new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
    //11    parameters.add(new NObsParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
    //12    parameters.add(new NPosParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

    private void setParametersFromDistribution(final int firstIndex, final ContinuousDist dist)
    {
        for(int i = 0; i < 4; i++)
        {
            if(dist == null)
            {
                setValue(firstIndex + i, Double.NaN);
            }
            else if(i < dist.getParameterCount())
            {
                setValue(firstIndex + i, dist.getParameter(i).doubleValue());
            }
            else
            {
                setValue(firstIndex + i, Double.NaN);
            }
        }
    }

    public double getObsAvg()
    {
        return getValue(0);
    }

    public void setObsAvg(final double value)
    {
        setValue(0, value);
    }

    public double getFcstAvg()
    {
        return getValue(1);
    }

    public void setFcstAvg(final double value)
    {
        setValue(1, value);
    }

    public double getObsPrecipThresh()
    {
        return getValue(2);
    }

    public void setObsPrecipThresh(final double value)
    {
        setValue(2, value);
    }

    public double getObsPoP()
    {
        return getValue(3);
    }

    public void setObsPoP(final double value)
    {
        setValue(3, value);
    }

    public double getObsCondAvg()
    {
        return getValue(4);
    }

    public void setObsCondAvg(final double value)
    {
        setValue(4, value);
    }

    public double getObsCondCoeffVar()
    {
        return getValue(5);
    }

    public void setObsCondCoeffVar(final double value)
    {
        setValue(5, value);
    }

    public double getFcstPrecipThresh()
    {
        return getValue(6);
    }

    public void setFcstPrecipThresh(final double value)
    {
        setValue(6, value);
    }

    public double getFcstPoP()
    {
        return getValue(7);
    }

    public void setFcstPoP(final double value)
    {
        setValue(7, value);
    }

    public double getFcstCondAvg()
    {
        return getValue(8);
    }

    public void setFcstCondAvg(final double value)
    {
        setValue(8, value);
    }

    public double getFcstCondCoeffVar()
    {
        return getValue(9);
    }

    public void setFcstCondCoeffVar(final double value)
    {
        setValue(9, value);
    }

    public double getRho()
    {
        if(!getParameterTypes().get(10)
                               .equals(new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)))
        {
            throw new IllegalStateException("Based on rho parameter, the model parameter types list is out of sync with the PrecipitationOneSetParameterValues class; found "
                + getParameterTypes().get(10).toString());
        }
        return getValue(10);
    }

    public void setRho(final double value)
    {
        setValue(10, value);
    }

    public int getNumObs()
    {
        return (int)getValue(11);
    }

    public void setNumObs(final int value)
    {
        setValue(11, value);
    }

    public int getNumBothPos()
    {
        return (int)getValue(12);
    }

    public void setNumBothPos(final int value)
    {
        setValue(12, value);
    }

//13    parameters.add(new EPTBivariateProbOfNoPrecipParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//14    parameters.add(new EPTPOPObsParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//15    parameters.add(new EPTPOPFcstParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//16    parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 0));
//17    parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 1));
//18    parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 2));
//19    parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 3));
//20    parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 0));
//21    parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 1));
//22    parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 2));
//23    parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 3));
//24    parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 0));
//25    parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,  1));
//26    parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 2));
//27    parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 3));
//28    parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 0));
//29    parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 1));
//30    parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 2));
//31    parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 3));
//32    parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 0));
//33    parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 1));
//34    parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 2));
//35    parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, 3));
//36    parameters.add(new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//37    parameters.add(new EPTXminParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//38    parameters.add(new EPTMuParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
//39    parameters.add(new EPTSigParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

    public double getEPTBivarProbNoPrecip()
    {
        return getValue(13);
    }

    public void setEPTBivarProbNoPrecip(final double value)
    {
        setValue(13, value);
    }

    public double getEPTObsPopFcstZero()
    {
        return getValue(14);
    }

    public void setEPTObsPopFcstZero(final double value)
    {
        setValue(14, value);
    }

    public double getEPTFcstPoPObsZero()
    {
        return getValue(15);
    }

    public void setEPTFcstPoPObsZero(final double value)
    {
        setValue(15, value);
    }

    public double getEPTFcstZeroObsDistParm(final int distParameterIndex)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        return getValue(16 + distParameterIndex);
    }

    /**
     * Instantiates the distribution of the given type and specifies its parameters based on the return of
     * {@link #getEPTFcstZeroObsDistParm(int)}.
     * 
     * @param type The type of distribution to instantiate.
     * @return A instance of the needed {@link ContinuousDistribution}.
     * @throws DefaultFactoryException DistributionType cannot be instantiated.
     */
    public ContinuousDist getEPTFcstZeroObsDist(final DistributionType type) throws DefaultFactoryException
    {
        final Double[] parameters = new Double[4];
        for(int i = 0; i < parameters.length; i++)
        {
            parameters[i] = getEPTFcstZeroObsDistParm(i);
        }
        return MEFPPrecipitationModelTools.instantiateOperationalDistribution(type, parameters);
    }

    //TODO I don't think the EPT fcst zero obs dist is ever used.
    public void setEPTFcstZeroObsDist(final ContinuousDist dist)
    {
        this.setParametersFromDistribution(16, dist);
    }

    public void setEPTFcstZeroObsDistParm(final int distParameterIndex, final double value)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        setValue(16 + distParameterIndex, value);
    }

    /**
     * This distribution is that of the obs values when the fcst values are 0 (x0_y1 in the prototype code). It is fit
     * using dist type 10 and pe_method 1, according to the prototype code.
     * 
     * @param distParameterIndex
     * @return
     */
    public double getEPTObsZeroFcstDistParm(final int distParameterIndex)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        return getValue(20 + distParameterIndex);
    }

    /**
     * Instantiates the distribution of the given type and specifies its parameters based on the return of
     * {@link #getEPTObsZeroFcstDistParm(int)}.
     * 
     * @param type The type of distribution to instantiate.
     * @return A instance of the needed {@link ContinuousDistribution}.
     * @throws DefaultFactoryException DistributionType cannot be instantiated.
     */
    public ContinuousDist getEPTObsZeroFcstDist(final DistributionType type) throws DefaultFactoryException
    {
        final Double[] parameters = new Double[4];
        for(int i = 0; i < parameters.length; i++)
        {
            parameters[i] = getEPTObsZeroFcstDistParm(i);
        }
        return MEFPPrecipitationModelTools.instantiateOperationalDistribution(type, parameters);
    }

    public void setEPTObsZeroFcstDist(final ContinuousDist dist)
    {
        setParametersFromDistribution(20, dist);
    }

    public void setEPTObsZeroFcstDistParm(final int distParameterIndex, final double value)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        setValue(20 + distParameterIndex, value);
    }

    /**
     * Intermittency distribution for forecast values is fit to the fcst values such that both fcst and obs are positive
     * (r in the prototype code), but is forced to be fit for distribution 10 and pe_method 1, whatever they are.
     * 
     * @param distParameterIndex 0..3.
     * @return Distribution parameter.
     */
    public double getEPTFcstIntermittencyDistParm(final int distParameterIndex)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        return getValue(24 + distParameterIndex);
    }

    /**
     * Instantiates the distribution of the given type and specifies its parameters based on the return of
     * {@link #getEPTFcstIntermittencyDistParm(int)}.
     * 
     * @param type The type of distribution to instantiate.
     * @return A instance of the needed {@link ContinuousDistribution}.
     * @throws DefaultFactoryException DistributionType cannot be instantiated.
     */
    public ContinuousDist getEPTFcstIntermittencyDist(final DistributionType type) throws DefaultFactoryException
    {
        final Double[] parameters = new Double[4];
        for(int i = 0; i < parameters.length; i++)
        {
            parameters[i] = getEPTFcstIntermittencyDistParm(i);
        }
        return MEFPPrecipitationModelTools.instantiateOperationalDistribution(type, parameters);
    }

    public void setEPTFcstIntermittencyDistParm(final int distParameterIndex, final double value)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        setValue(24 + distParameterIndex, value);
    }

    public void setEPTFcstIntermittencyDist(final ContinuousDist dist)
    {
        setParametersFromDistribution(24, dist);
    }

    /**
     * This distribution is that of the fcst values when both fcst and obs are positive (r in the prototype code; same
     * as intermittency above). It is fit using whatever distribution and method is specified by the distribution type.
     * 
     * @param distParameterIndex
     * @return
     */
    public double getEPTFcstBivarMarginalDistParm(final int distParameterIndex)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        return getValue(28 + distParameterIndex);
    }

    /**
     * Instantiates the distribution of the given type and specifies its parameters based on the return of
     * {@link #getEPTFcstBivarMarginalDistParm(int)}.
     * 
     * @param type The type of distribution to instantiate.
     * @return A instance of the needed {@link ContinuousDistribution}.
     * @throws DefaultFactoryException DistributionType cannot be instantiated.
     */
    public ContinuousDist getEPTFcstBivarMarginalDist(final DistributionType type) throws DefaultFactoryException
    {
        final Double[] parameters = new Double[4];
        for(int i = 0; i < parameters.length; i++)
        {
            parameters[i] = getEPTFcstBivarMarginalDistParm(i);
        }
        return MEFPPrecipitationModelTools.instantiateOperationalDistribution(type, parameters);
    }

    public void setEPTFcstBivarMarginalDist(final ContinuousDist dist)
    {
        setParametersFromDistribution(28, dist);
    }

    public void setEPTFcstBivarMarginalDistParm(final int distParameterIndex, final double value)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        setValue(28 + distParameterIndex, value);
    }

    /**
     * This distribution is that of the obs values when both fcst and obs are positive (r in the prototype code; same as
     * intermittency above). It is fit using whatever distribution and method is specified by the distribution type.
     * 
     * @param distParameterIndex
     * @return
     */
    public double getEPTObsBivarMarginalDistParm(final int distParameterIndex)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        return getValue(32 + distParameterIndex);
    }

    /**
     * Instantiates the distribution of the given type and specifies its parameters based on the return of
     * {@link #getEPTObsBivarMarginalDistParm(int)}.
     * 
     * @param type The type of distribution to instantiate.
     * @return A instance of the needed {@link ContinuousDistribution}.
     * @throws DefaultFactoryException DistributionType cannot be instantiated.
     */
    public ContinuousDist getEPTObsBivarMarginalDist(final DistributionType type) throws DefaultFactoryException
    {
        final Double[] parameters = new Double[4];
        for(int i = 0; i < parameters.length; i++)
        {
            parameters[i] = getEPTObsBivarMarginalDistParm(i);
        }
        return MEFPPrecipitationModelTools.instantiateOperationalDistribution(type, parameters);
    }

    public void setEPTObsBivarMarginalDist(final ContinuousDist dist)
    {
        setParametersFromDistribution(32, dist);
    }

    public void setEPTObsBivarMarginalDistParm(final int distParameterIndex, final double value)
    {
        if(distParameterIndex >= 4)
        {
            throw new IllegalArgumentException("Distribution parameter index cannot be 4 or greater.");
        }
        setValue(32 + distParameterIndex, value);
    }

    public double getEPTRho()
    {
        return getValue(36);
    }

    public void setEPTRho(final double value)
    {
        setValue(36, value);
    }

    public double getEPTXmin()
    {
        return getValue(37);
    }

    public void setEPTXmin(final double value)
    {
        setValue(37, value);
    }

    public double getEPTMu()
    {
        return getValue(38);
    }

    public void setEPTMu(final double value)
    {
        setValue(38, value);
    }

    public double getEPTSig()
    {
        return getValue(39);
    }

    public void setEPTSig(final double value)
    {
        setValue(39, value);
    }

}
