package ohd.hseb.hefs.mefp.models.precipitation;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.MEFPModelTools;
import ohd.hseb.hefs.mefp.models.MEFPParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPAlgorithmModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.types.AvgParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.CAvgParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.CCVParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTBivariateProbOfNoPrecipParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTFcstBivarMarginalDistributionParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTFcstIntermittencyDistributionParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTFcstZeroObsDistributionParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTMuParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTObsBivarMarginalDistributionParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTObsZeroFcstDistributionParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTPOPFcstParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTPOPObsParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTRhoParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTSigParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTXminParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.NObsParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.NPosParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.POPParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.PThreshParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.RhoParameterType;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.GenericMEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationEstimationControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneSetParameterValues;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

/**
 * The order of the {@link ModelParameterType} instances returned herein must be kept in sync with those in
 * {@link PrecipitationOneSetParameterValues}.
 * 
 * @author hank.herr
 */
public class PrecipitationParameterEstimationModel extends MEFPParameterEstimationModel
{
    private static final Logger LOG = LogManager.getLogger(PrecipitationParameterEstimationModel.class);

//    public final static double FIXED_EPT_THRESHOLD = 0.25d;

    private final static ModelParameterType IPT_POP_TYPE = new POPParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                                false);
    private final static ModelParameterType EPT_RHO_TYPE = new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);
    private final static ModelParameterType EPT_POP_TYPE = new EPTBivariateProbOfNoPrecipParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID);

    /**
     * Initializes the algorithm model parameters.
     */
    public PrecipitationParameterEstimationModel()
    {
        setAlgorithmModelParameters(new MEFPAlgorithmModelParameters(true));
    }

    /**
     * Computes event values for a non-climatology source: RFC, GFS, CFSv2.
     * 
     * @param values The {@link SourceCanonicalEventValues} for which
     *            {@link SourceCanonicalEventValues#computeCanonicalEventValues(LocationAndDataTypeIdentifier, TimeSeriesArray, SourceDataHandler, int, int)}
     *            will be called.
     * @param identifier Identifier for which data is to be loaded.
     * @param source The source providing a {@link SourceDataHandler} to be used to load data.
     * @param sourceCtlParms Control parameters for the source. Why aren't these used???
     * @return The computed {@link SourceCanonicalEventValues}.
     * @throws ParameterEstimationException If the time series fail to load.
     */
    private void computeEventValuesForNonClimatologySource(final SourceCanonicalEventValues values,
                                                           final MEFPForecastSource source,
                                                           final MEFPSourceControlOptions sourceCtlParms) throws ParameterEstimationException
    {
        //Use the single historical ts by default.
        TimeSeriesArray obsTS = getSingleHistoricalTimeSeries();

        //Load the prepared time series for the source.
        try
        {
            source.getSourceDataHandler().loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new ParameterEstimationException("Could not load time series for source " + source.getName()
                + " (was source data prepared?):" + e.getMessage());
        }

        //Only use the source specific observed data if it is a non-empty time series.
        try
        {
            final TimeSeriesArray singleTS = source.getSourceDataHandler()
                                                   .getSingleObservedTimeSeries(getEstimatedIdentifier());
            if(!TimeSeriesArrayTools.isAllMissing(singleTS))
            {
                obsTS = singleTS;
            }
        }
        catch(final Exception e)
        {
            //If an exception occurs, use obsTS as given.
        }

        //Compute the event values for the source.
        values.computeCanonicalEventValues(getEstimatedIdentifier(), obsTS, source.getSourceDataHandler()); //Years are pulled from test scenario control file

    }

    /**
     * If problems occur during computation, parameters must still be recorded, but the {@link ModelParameterType}s
     * checked for missing must be set to {@link Double#NaN}.
     * 
     * @param modelParameters
     * @param source
     * @param modelCtlParms
     * @param values
     */
    private void calculateParametersForSource(final MEFPForecastSource source, final SourceCanonicalEventValues values) throws ParameterEstimationException
    {
        //Initialize the gatherers for IPT and EPT.
        final StandardCanonicalEventValuesGatherer iptGatherer = (StandardCanonicalEventValuesGatherer)source.constructCanonicalEventValuesGatherer(values,
                                                                                                                                                    getEstimationModelCtlParms(),
                                                                                                                                                    getEstimationCtlParms().getSourceControlOptions(source));
        StandardCanonicalEventValuesGatherer eptGatherer = null;
        if(source.canEPTModelBeUsedForSource())
        {
            eptGatherer = MEFPModelTools.constructEPTCanonicalEventValuesGatherer(source,
                                                                                  values,
                                                                                  getEstimationModelCtlParms(),
                                                                                  getEstimationCtlParms().getSourceControlOptions(source));
        }

        //The computed canonical events, used within a nested loop below.
        final CanonicalEventList computedEvents = values.getForecastValues().getEventsToCompute();

        //Loop through days of the year at a step dictated by control parameters.
        for(final int dayOfYear: getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters())
        {
            //Catch a user cancel before each day is computed.
            if(JobMessenger.isCanceled())
            {
                throw new ParameterEstimationException("Parameter estimation has been canceled (2).");
            }

            JobMessenger.madeProgress("Estimating parameters for day " + dayOfYear + "...");
            LOG.info("Computing parameters for day of year " + dayOfYear + "...");

            //Gather the event values for all events.
//            System.out.println("####>> ---------------- gather events...");
//            HStopWatch timer = new HStopWatch();
            iptGatherer.gatherEventValues(dayOfYear);
            if(eptGatherer != null)
            {
                eptGatherer.gatherEventValues(dayOfYear);
            }

            //For each canonical events, parameters must be computed and stored.
//            System.out.println("####>> ----------------     done in " + timer.getElapsedMillis());
//            System.out.println("####>> ---------------- compute parms by event...");
//            timer = new HStopWatch();
//            long iptMillis = 0L;
//            long eptMillis = 0L;
//            final long[] eptRunTimes = new long[7];
//            Arrays.fill(eptRunTimes, 0L);
            for(int canonicalEventIndex = 0; canonicalEventIndex < computedEvents.size(); canonicalEventIndex++)
            {
//                LOG.info("Computing parmameters for canonical event #" + canonicalEventIndex + ": "
//                    + computedEvents.get(canonicalEventIndex).toString() + "...");
                //Construct the one day parm storage object.
                final PrecipitationOneSetParameterValues dayParmValues = new PrecipitationOneSetParameterValues(dayOfYear,
                                                                                                                canonicalEventIndex,
                                                                                                                this,
                                                                                                                source);

                //Compute the IPT parameters.  Note that if not enough values were computed, then the IPT parameters
                //will be left as NaN (during construction, they are set to NaN).  This will be handled later.
//                final HStopWatch iptTimer = new HStopWatch();
                final IPTPrecipitationParameterCalculator iptCalc = new IPTPrecipitationParameterCalculator(iptGatherer.getLastComputedForecastThreshold(canonicalEventIndex),
                                                                                                            iptGatherer.getLastComputedObservedThreshold(canonicalEventIndex),
                                                                                                            getEstimationModelCtlParms().getIPT()
                                                                                                                                        .getMaximumConditionalMeanPrecipitationValue());
                if(iptGatherer.wereEnoughValuesGathered(canonicalEventIndex))
                {
                    iptCalc.calculate(iptGatherer.getForecastGatheredEventValues(canonicalEventIndex),
                                      iptGatherer.getObservedGatheredEventValues(canonicalEventIndex),
                                      getEstimationModelCtlParms().getMinimumNumberOfObservationsRequired());
                    addQuestionableParameterLogMessage(source,
                                                       computedEvents.get(canonicalEventIndex),
                                                       dayOfYear,
                                                       iptCalc.getQuestionableMessage());
                }
                else
                {
                    addQuestionableParameterLogMessage(source,
                                                       computedEvents.get(canonicalEventIndex),
                                                       dayOfYear,
                                                       "IPT: Not enough data for parameter estimation; check control options.");
                }
                iptCalc.populateOneSetParameterValues(dayParmValues);
                if(areParametersMissing(dayParmValues, true))
                {
                    LOG.debug("Unable to compute IPT parameters for day " + dayOfYear + " and event # "
                        + canonicalEventIndex + " (" + computedEvents.get(canonicalEventIndex).toString()
                        + ") probably due to insufficient data.");
                }
//                iptTimer.stop();
//                iptMillis += iptTimer.getElapsedMillis();

                //If the EPT option is on and the source allows for its computation, then compute the parameters.
                //This computation does not depend on determining if enough values are available.  However, there is 
                //a check in the calculator for enough values which throws an exception if not enough, but it uses
                //a hardcoded minimum required number.
                if(eptGatherer != null)
                {
//                    final HStopWatch eptTimer = new HStopWatch();
                    //Unlike IPT, checks for enough data are handled during calculations, as was done in the legacy code.
                    final EPTPrecipitationParameterCalculator eptCalc = new EPTPrecipitationParameterCalculator(getEstimationModelCtlParms().getEPT()
                                                                                                                                            .getEPTPrecipThreshold());
                    try
                    {
                        eptCalc.calculate(eptGatherer.getForecastGatheredEventValues(canonicalEventIndex),
                                          eptGatherer.getObservedGatheredEventValues(canonicalEventIndex),
                                          getEstimationModelCtlParms().getEPT().getEPTDistribution(),
                                          getEstimationModelCtlParms().getEPT().getEPTPrecipThreshold(),
                                          ((MEFPPrecipitationSourceControlOptions)getEstimationCtlParms().getSourceControlOptions(source)).getEPTAlpha());
               
//                        eptCalc.addRunTimes(eptRunTimes);
                        addQuestionableParameterLogMessage(source,
                                                           computedEvents.get(canonicalEventIndex),
                                                           dayOfYear,
                                                           eptCalc.getQuestionableMessage());
                    }
                    catch(final Exception e)
                    {
                        e.printStackTrace();
                        
                        //How are these exceptions handled?  They aren't!  We'll handle it here the same
                        //way we handle IPT problems: copy the latest.
                        LOG.debug("Exception occurred calculating EPT parameters: " + e.getMessage());
                        addQuestionableParameterLogMessage(source,
                                                           computedEvents.get(canonicalEventIndex),
                                                           dayOfYear,
                                                           e.getMessage());
//                        LoggingTools.outputDebugLines(LOG, ExceptionParser.multiLineStackTrace(e));  -- outputs lines to log
                    }
                    eptCalc.populateOneSetParameterValues(dayParmValues);
                    if(areParametersMissing(dayParmValues, false))
                    {
                        //NOTE: A questionable message should already be output above if this fails, either via the except thrown from EPT 
                        //or via the EPT itself.
                        LOG.debug("Unable to compute EPT parameters for day " + dayOfYear + " and event # "
                            + canonicalEventIndex + " (" + computedEvents.get(canonicalEventIndex).toString()
                            + ") probably due to insufficient data.");
                    }
//                    eptTimer.stop();
//                    eptMillis += eptTimer.getElapsedMillis();
                }

                //Put the values for this day into the modelParameters.
                getEstimatedMEFPModelParameters().recordValuesFrom(dayParmValues);
//
//                LOG.info("Done computing parameters for canonical event #" + canonicalEventIndex + ".");
            }
//            System.out.println("####>> ---------------- done in " + timer.getElapsedMillis() + " -- " + iptMillis
//                + ", " + eptMillis + " --- " + Arrays.toString(eptRunTimes));
        }
    }

    /**
     * Checks a designated variable in order to determine if the parameters should be considered missing. For IPT, its
     * {@link #IPT_POP_TYPE}, for EPT its {@link #EPT_POP_TYPE}.
     * 
     * @param sourceModelParameters Parameters to check.
     * @param dayOfYear Day of year for which to check.
     * @param canonicalEventIndex Event index for which to check.
     * @param ipt True to check IPT parameters, false for EPT.
     * @return True if the parameters are missing, false if not.
     */
    private boolean areParametersMissing(final MEFPSourceModelParameters sourceModelParameters,
                                         final int dayOfYear,
                                         final int canonicalEventIndex,
                                         final boolean ipt)
    {
        //NOTE: This is purposely coded independently of the other version in order to save time trying to construct a 
        //PrecipitationOneSetParameterValues object.
        ModelParameterType checkedType = IPT_POP_TYPE;
        if(!ipt)
        {
            checkedType = EPT_POP_TYPE;
        }
        return Double.isNaN(sourceModelParameters.getParameterValues(checkedType).getValue(dayOfYear,
                                                                                           canonicalEventIndex));
    }

    /**
     * Check if the provided one day of values is considered missing.
     * 
     * @param values The {@link PrecipitationOneSetParameterValues} to check.
     * @param ipt True to check IPT parameters, false for EPT.
     * @return True if the parameters are missing, false if not.
     */
    private boolean areParametersMissing(final PrecipitationOneSetParameterValues values, final boolean ipt)
    {
        ModelParameterType checkedType = IPT_POP_TYPE;
        if(!ipt)
        {
            checkedType = EPT_POP_TYPE;
        }
        return Double.isNaN(values.getValue(checkedType));
    }

    /**
     * Implements the Fortran filling algorithm for handling days for which parameters cannot be computed due to not
     * having enough observations or an exception of some type.
     * 
     * @param sourceModelParameters The parameters to fill.
     * @param ipt True if IPT parameters are to be filled, false for EPT.
     */
    private void fillInValuesForComputationalDaysForSource(final MEFPSourceModelParameters sourceModelParameters,
                                                           final boolean ipt) throws ParameterEstimationException
    {
        //NOTE: Any questionable messages relating to this filling should already have been handled.  

        final CanonicalEventList computedEvents = sourceModelParameters.getComputedEvents();
        boolean atLeastOneNonMissing = false;
        final List<Integer> daysOfYear = getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters();

        //For each canonical event...
        for(int canonicalEventIndex = 0; canonicalEventIndex < computedEvents.size(); canonicalEventIndex++)
        {
            //Find the first day with non-missing values.
            int initialDayOfYearIndex = 0;
            for(initialDayOfYearIndex = 0; initialDayOfYearIndex < daysOfYear.size(); initialDayOfYearIndex++)
            {
                if(!areParametersMissing(sourceModelParameters,
                                         daysOfYear.get(initialDayOfYearIndex),
                                         canonicalEventIndex,
                                         ipt))
                {
                    break;
                }
            }
            if(initialDayOfYearIndex == daysOfYear.size()) //If we make it to the end, then all data is missing.
            {
                LOG.warn("For canonical event # "
                    + canonicalEventIndex
                    + " ("
                    + computedEvents.get(canonicalEventIndex).toString()
                    + "), parameter estimation failed for all days of the year for which it was attempted, likely due to insufficient sample size.");
                continue;
            }
            atLeastOneNonMissing = true;

            //Now, starting from the initialDayOfYear, jump to the next day.  If the parameters for that day are 
            //missing based on checkedType, set them based on the previous day.  However, be sure to only set the 
            //parameters for EPT or IPT, not both, since missing is checked independently for each.  The if-check
            //wraps the working index to 0 if needed.
            //XXX The working index wrap check is done twice (see below).  Is there a better way to organize the loop so that it is done once?
            //Note that the wrap-check must be done before the while check is performed.
            int workingIndex = initialDayOfYearIndex + 1;
            if(workingIndex >= daysOfYear.size())
            {
                workingIndex = 0;
            }
            int previousIndex = initialDayOfYearIndex;
            while(workingIndex != initialDayOfYearIndex)
            {
                if(areParametersMissing(sourceModelParameters, daysOfYear.get(workingIndex), canonicalEventIndex, ipt))
                {
                    final PrecipitationOneSetParameterValues previousValues = (PrecipitationOneSetParameterValues)getEstimatedMEFPModelParameters().getSourceEstimatedModelParameterValues(sourceModelParameters.getForecastSource(),
                                                                                                                                                                                           daysOfYear.get(previousIndex),
                                                                                                                                                                                           canonicalEventIndex);
                    final PrecipitationOneSetParameterValues workingValues = (PrecipitationOneSetParameterValues)getEstimatedMEFPModelParameters().getSourceEstimatedModelParameterValues(sourceModelParameters.getForecastSource(),
                                                                                                                                                                                          daysOfYear.get(workingIndex),
                                                                                                                                                                                          canonicalEventIndex);
                    if(ipt)
                    {
//                        System.err.println("####>> COPYING IPT -- " + canonicalEventIndex + ", " + workingDayOfYear
//                            + ", " + previousDayOfYear);
                        workingValues.copyIPTParameters(previousValues); //IPT copy.
                    }
                    else
                    {
                        workingValues.copyEPTParameters(previousValues); // Does this match how EPT parms are filled in Fortran? -- yes, I confirmed it.
                    }
                    getEstimatedMEFPModelParameters().recordValuesFrom(workingValues);
                }

                //Go to the next day, wrapping to 0 if needed.
                previousIndex = workingIndex;
                workingIndex++;
                if(workingIndex >= daysOfYear.size())
                {
                    workingIndex = 0;
                }
            }
        }

        if(!atLeastOneNonMissing)
        {
            throw new ParameterEstimationException("No parameters were successfully computed for any canonical events over all days of the year for this source.");
        }
    }

    /**
     * Populates all other source model parameters as is appropriate.
     * 
     * @param sourceModelParameters
     */
    private void populateOtherSourceModelParameters(final MEFPSourceModelParameters sourceModelParameters)
    {
        final GenericMEFPSourceControlOptions genericCtlParms = (GenericMEFPSourceControlOptions)getEstimationCtlParms().getSourceControlOptions(sourceModelParameters.getForecastSource());
        sourceModelParameters.setNumberOfForecastLeadDays(genericCtlParms.getNumberOfForecastDaysUsed());
        sourceModelParameters.setNumberOfCanonicalEvents(sourceModelParameters.getComputedEvents().size());
        sourceModelParameters.determineTimesOfComputedCanonicalEvents();

        //Set the distributions based on control options. 
        final MEFPPrecipitationSourceControlOptions sourceCtlParms = (MEFPPrecipitationSourceControlOptions)getEstimationCtlParms().getSourceControlOptions(sourceModelParameters.getForecastSource());
        final MEFPSourceModelParameters parms = sourceModelParameters;
        parms.setDistributionOfObs(sourceCtlParms.getObservedDistribution());
        if(!sourceModelParameters.getForecastSource().isClimatologySource()) //For climatology, this is null
        {
            parms.setDistributionOfFcsts(sourceCtlParms.getForecastDistribution());
        }
    }

    /**
     * @return List of parameter types for EPT estimation. I find it convenient to have one method initialize EPT
     *         parameters, while the other method does the IPT parameters (and combines them).
     */
    private List<ModelParameterType> getEPTParameters()
    {
        final List<ModelParameterType> parameters = new ArrayList<ModelParameterType>();

        parameters.add(EPT_POP_TYPE);
        parameters.add(new EPTPOPObsParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new EPTPOPFcstParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   0));
        parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   1));
        parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   2));
        parameters.add(new EPTFcstZeroObsDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   3));
        parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   0));
        parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   1));
        parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   2));
        parameters.add(new EPTObsZeroFcstDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                   3));
        parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         0));
        parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         1));
        parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         2));
        parameters.add(new EPTFcstIntermittencyDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         3));
        parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         0));
        parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         1));
        parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         2));
        parameters.add(new EPTFcstBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                         3));
        parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                        0));
        parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                        1));
        parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                        2));
        parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                        3));
        parameters.add(EPT_RHO_TYPE);
        parameters.add(new EPTXminParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new EPTMuParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new EPTSigParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

        return parameters;
    }

    public List<ModelParameterType> getAllParametersRequiredForModel(final boolean eptOn)
    {
        final List<ModelParameterType> parameters = new ArrayList<ModelParameterType>();

        //Base parameters
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
        parameters.add(new PThreshParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new POPParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new CCVParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new PThreshParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
        parameters.add(new POPParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
        parameters.add(new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
        parameters.add(new CCVParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true));
        parameters.add(new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new NPosParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

        //EPT parameters.  I call the getEPTParameters method solely for organizational purposes.
        if(eptOn)
        {
            parameters.addAll(this.getEPTParameters());
        }
        return parameters;
    }

    @Override
    protected boolean estimateParametersForSource(final MEFPForecastSource source) throws ParameterEstimationException
    {
        //Get the control options and check if the source is to be used. 
        final MEFPSourceControlOptions sourceCtlParms = getEstimationCtlParms().getSourceControlOptions(source);

        //Get the source model parameters and initialize it.  We call this even if parameter estimation is not to be performed
        //in order to ensure that the tarball will still include all the files for a source even if not estimated.
        final MEFPSourceModelParameters srcModelParms = getEstimatedMEFPModelParameters().getSourceModelParameters(source);
        srcModelParms.setApplicableModel(this);
        srcModelParms.setupForStoringParametersAndEvents(sourceCtlParms.getNumberOfForecastDaysUsed());//Last arg differs from below.

        //Exit if the source is not enabled or the number of days is 0.  By doing it after the setup call above, this means that all files
        //will be output to the tar ball even if the source is not used.  That should simplify things when reading parameters, since
        //I won't need to check for file existence in the tar ball.
        if((!sourceCtlParms.isEnabled()) || (sourceCtlParms.getNumberOfForecastDaysUsed() <= 0))
        {
            LOG.info("Forecast source " + source.getName()
                + " is not to be used or the number of forecast days is 0.  Skipping source.");
            return false;
        }

        //Log messaging starts here, since we know parameters are to be estimated.
        LOG.info("Estimating parameters for forecast source " + source.getName() + "...");
        JobMessenger.newMonitorSubJob();
        JobMessenger.setMaximumNumberOfSteps(getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters()
                                                                              .size() + 1);
        JobMessenger.madeProgress("Loading time series and computing canonical events...");

        //Need to make sure the subjob is cleared, so I wrapped this in a try finally.
        try
        {
            //Catch a user cancel before events are computed.  The next catch is before the days are computed.
            if(JobMessenger.isCanceled())
            {
                throw new ParameterEstimationException("Parameter estimation has been canceled (1).");
            }

            //Compute canonical event values.
            final SourceCanonicalEventValues values = srcModelParms.getPrecipitationSourceEventValues();
            if(!source.isClimatologySource())
            {
                LOG.info("Computing canonical event values for non-climatology source...");
                //The method below loads the prepared data for the source, inlcuding both fcst and obs.
                computeEventValuesForNonClimatologySource(values, source, sourceCtlParms);
            }
            else
            {
                LOG.info("Computing canonical event values for climatology source...");
                values.computeCanonicalEventValues(getSingleHistoricalTimeSeries());
            }
            LOG.info("Computed " + values.getForecastValues().getNumberOfComputationalTimesWithValues()
                + " forecast canonical event values for source.");

            //Calculate the source parameters and populate the modelParameters.
            LOG.info("Calculating parameters...");
            calculateParametersForSource(source, values);
            LOG.info("Done calculating parameters.");

            //Fill in the missing or non-computed values.
            populateOtherSourceModelParameters(srcModelParms);
            fillInValuesForComputationalDaysForSource(srcModelParms, true); //IPT
            if(source.canEPTModelBeUsedForSource())
            {
                fillInValuesForComputationalDaysForSource(srcModelParms, false); //EPT, if necessary
            }

            LOG.info("Done estimating parameters for forecast source " + source.getName());
        }
        catch(final Throwable t)
        {
            //t.printStackTrace(); //DEBUG LINE
            final ParameterEstimationException e = new ParameterEstimationException("Problem occurred estimating parameters for "
                                                                                        + source.getName()
                                                                                        + ":\n"
                                                                                        + t.getMessage(),
                                                                                    t);
            e.setStackTrace(t.getStackTrace());
            throw e;
        }
        finally
        {
            JobMessenger.clearMonitorSubJob();
        }

        return true;
    }

    @Override
    protected void loadHistoricalData() throws ParameterEstimationException
    {
        try
        {
            getMEFPRunInfo().getHistoricalDataHandler()
                            .loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
        }
        catch(final Exception e)
        {
            throw new ParameterEstimationException(e.getMessage(), e);
        }

        TimeSeriesArray histTS = getMEFPRunInfo().getHistoricalDataHandler()
                                                 .getSingleObservedTimeSeries(getEstimatedIdentifier());

        //Get RFC obs data, if any (if the RFC source is being used in parameter estimation).  Extend historical.
        if(getMEFPRunInfo().getRFCForecastDataHandler() != null)
        {
            TimeSeriesArray rfcTS = null;
            if(this.getEstimationCtlParms()
                   .getSourceControlOptions(new RFCForecastSource())
                   .isSourceUsedInParameterEstimation())
            {
                try
                {
                    getMEFPRunInfo().getRFCForecastDataHandler()
                                    .loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
                    rfcTS = getMEFPRunInfo().getRFCForecastDataHandler()
                                            .getSingleObservedTimeSeries(getEstimatedIdentifier());
                    getMEFPRunInfo().getRFCForecastDataHandler().clearLoadedTimeSeries();
                }
                catch(final Exception e)
                {
                    LOG.warn("RFC forecasts are being used, but a problem was encountered trying to load RFC observed time series: "
                        + e.getMessage());
                    rfcTS = null;
                }
            }

            //Create the composite time series and add them to the superclass storage device.
            if(rfcTS != null)
            {
                try
                {
                    TimeSeriesArrayTools.fillInMissingFromOtherTimeSeries(histTS, rfcTS);
                    TimeSeriesArrayTools.extendFromOtherTimeSeries(histTS, rfcTS);
                    histTS = TimeSeriesArrayTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(histTS);

                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    throw new ParameterEstimationException("Cannot build historical time series due to an unexpected exception creating them: "
                                                               + e.getMessage(),
                                                           e);
                }
            }
            else
            {
                LOG.warn("RFC forecasts will not be used in construction of composite historical "
                    + "time series because the source is not being used to estimate parameters.");
            }
        }

        addHistoricalTimeSeries(histTS);
    }

    @Override
    public ControlOption createControlOptions()
    {
        return new MEFPPrecipitationModelControlOptions();
    }

    @Override
    public List<ModelParameterType> getAllParametersRequiredForModel(final ForecastSource source)
    {
        return getAllParametersRequiredForModel(((MEFPForecastSource)source).canEPTModelBeUsedForSource());
    }

    @Override
    public MEFPPrecipitationModelControlOptions getEstimationModelCtlParms()
    {
        return (MEFPPrecipitationModelControlOptions)super.getEstimationModelCtlParms();
    }

    @Override
    public List<ModelParameterType> getObservationParametersRequiredForModel(final ForecastSource source)
    {
        //IPT climatology source parameters.
        final List<ModelParameterType> parameters = new ArrayList<ModelParameterType>();
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new PThreshParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(IPT_POP_TYPE);
        parameters.add(new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new CCVParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
        parameters.add(new NPosParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));

        //EPT Specific climatology source parameters.
        if(((MEFPForecastSource)source).canEPTModelBeUsedForSource())
        {
            parameters.add(EPT_POP_TYPE);
            parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                            0));
            parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                            1));
            parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                            2));
            parameters.add(new EPTObsBivarMarginalDistributionParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                            3));
        }

        return parameters;
    }

    @Override
    public Type getDataType()
    {
        return ParameterId.Type.PRECIPITATION;
    }

    @Override
    public OneSetParameterValues initializeOneSetParameterValues(final ForecastSource source,
                                                                 final int dayOfYear,
                                                                 final int canonicalEventIndex)
    {
        return new PrecipitationOneSetParameterValues(dayOfYear, canonicalEventIndex, this, (MEFPForecastSource)source);
    }

    /**
     * @param runInfo {@link MEFPParameterEstimatorRunInfo} that provides the needed data handlers for loading time
     *            series.
     * @param identifier Identifier for which to load historical time series.
     * @return The historical time series used in parameter estimation for the provided identifier. It is a combination
     *         of historical with RFC extensions.
     * @throws ParameterEstimationException See {@link #loadHistoricalData()}.
     */
    public static TimeSeriesArray retrieveHistoricalData(final MEFPParameterEstimatorRunInfo runInfo,
                                                         final LocationAndDataTypeIdentifier identifier) throws ParameterEstimationException
    {
        if(!identifier.isPrecipitationDataType())
        {
            throw new IllegalArgumentException("Identifier " + identifier.buildStringToDisplayInTree()
                + " is not a precipitation identifier.");
        }
        final PrecipitationParameterEstimationModel model = new PrecipitationParameterEstimationModel();

        final MEFPPrecipitationEstimationControlOptions ctlOptions = (MEFPPrecipitationEstimationControlOptions)MEFPTools.constructEstimationControlOptions(identifier.getParameterIdType(),
                                                                                                                                                            runInfo.getForecastSources());
        ctlOptions.getSourceControlOptions(new RFCForecastSource()).setEnabled(true);
        ((GenericMEFPSourceControlOptions)ctlOptions.getSourceControlOptions(new RFCForecastSource())).setDaysUsed(1);
        model.setEstimationCtlOptions(ctlOptions);

        model.setMEFPRunInfo(runInfo);
        model.setEstimatedIdentifier(identifier);
        model.loadHistoricalData();
        return model.getSingleHistoricalTimeSeries();
    }
}
