package ohd.hseb.hefs.mefp.models.temperature;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.ForecastEnsembleCalculator;
import ohd.hseb.hefs.mefp.models.MEFPEnsembleGeneratorModel;
import ohd.hseb.hefs.mefp.models.SchaakeShuffleApplier;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.MeasuringUnitType;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Basic temperature ensemble generation model using the {@link MEFPEnsembleGeneratorModel} superclass.
 * 
 * @author hankherr
 */
public class TemperatureEnsembleGenerationModel extends MEFPEnsembleGeneratorModel
{
    private static final Logger LOG = LogManager.getLogger(TemperatureEnsembleGenerationModel.class);
    public final static ParameterId[] PROCESSED_DATA_TYPES = {ParameterId.TFMN, ParameterId.TFMX};

    /**
     * Flag indicates if the model will be applied to maximum (true) or minimum temperature data. This controls which
     * parameters are pulled from the {@link MEFPFullModelParameters} returned by {@link #getModelParameters()}.
     */
    private boolean _tmaxData;

    @Override
    protected int getCanonicalEventPeriodStepSizeInHours()
    {
        return CanonicalEvent.determineCanonicalEventPeriodUnitInHours(false);
    }

    @Override
    protected ForecastEnsembleCalculator instantiateCalculator(final MEFPForecastSource source, final int dayOfYear) throws Exception
    {

        final TemperatureForecastEnsembleCalculator ensCalculator = new TemperatureForecastEnsembleCalculator(source,
                                                                                                              getModelParameters(),
                                                                                                              dayOfYear,
                                                                                                              isTmaxData(),
                                                                                                              getStratifiedSampling());
        return ensCalculator;
    }

    @Override
    protected double getRhoValue(final MEFPForecastSource source, final int dayOfYear, final CanonicalEvent evt0)
    {
        double rho = getModelParameters().getSourceModelParameters(source).getTMAXRhoParameterValue(dayOfYear, evt0);
        if(!isTmaxData())
        {
            rho = getModelParameters().getSourceModelParameters(source).getTMINRhoParameterValue(dayOfYear, evt0);
        }
//XXX Used in experiments to force an event to be processed last!        
//        if(evt0.equals(new CanonicalEvent(-1, 61, 90, -1)))
//        {
//            rho = 0.9999;
//        }
        return rho;
    }

    public boolean isTmaxData()
    {
        return _tmaxData;
    }

    public void setTmaxData(final boolean tmaxData)
    {
        _tmaxData = tmaxData;
    }

    @Override
    protected TimeSeriesArray getSingleHistoricalTimeSeries()
    {
        for(final TimeSeriesArray ts: getModelParameters().getHistoricalTimeSeries())
        {
            final ParameterId currentId = ParameterId.of(ts.getHeader());
            if(isTmaxData())
            {
                if(currentId.isMax() && currentId.isTemperature())
                {
                    return ts;
                }
            }
            else
            {
                if(currentId.isMin() && currentId.isTemperature())
                {
                    return ts;
                }
            }
        }
        throw new IllegalStateException("Parameters do not include historical temperature time series for maximum = "
            + isTmaxData() + ".");
    }

    @Override
    public ParameterId[] getProcessedDataTypes()
    {
        return PROCESSED_DATA_TYPES;
    }

    @Override
    protected ParameterId determineForecastEnsembleParameterId(final String baseEnsembleHistoricalParameterId)
    {
        final ParameterId baseParm = ParameterId.valueOf(baseEnsembleHistoricalParameterId);
        if(baseParm.isTemperature() && baseParm.isMax())
        {
            return ParameterId.TFMX;
        }
        else if(baseParm.isTemperature() && baseParm.isMin())
        {
            return ParameterId.TFMN;
        }
        throw new IllegalArgumentException("Base ensemble parameter id '" + baseEnsembleHistoricalParameterId
            + "' is not a valid temperature min/max parameter.");
    }

    @Override
    public ParameterId checkForValidityAndConvertToProcessedType(final String inputParameterStr) throws Exception
    {
        //Check...
        final ParameterId inputParameterId = ParameterId.valueOf(inputParameterStr);
        if(inputParameterId == null)
        {
            throw new Exception("Parameter id of " + inputParameterStr + " is not recognized by MEFP.");
        }
        if(!inputParameterId.isTemperature())
        {
            throw new Exception("Parameter id of " + inputParameterStr + " is not temperature.");
        }
        if((!inputParameterId.isMin()) && (!inputParameterId.isMax()))
        {
            throw new Exception("Parameter id of " + inputParameterStr + " is neither a minimum or maximum data type.");
        }

        //Convert...
        ParameterId result;
        if(inputParameterId.isMin())
        {
            result = ParameterId.TFMN;
        }
        else
        {
            result = ParameterId.TFMX;
        }
        if(!result.equals(inputParameterId))
        {
            LOG.warn("Provided parameter id " + inputParameterId + " is being treated as " + result
                + " the purposes of running this model.");
        }
        return result;
    }

    @Override
    public void checkForValidityAndConvertUnits(final TimeSeriesArray inputTS) throws Exception
    {
        //Null unit is treated as MM.  So if its null, it better be in metric!
        if(inputTS.getHeader().getUnit() == null)
        {
            ((DefaultTimeSeriesHeader)inputTS.getHeader()).setUnit("DEGC");
        }
        //MeasuringUnit does not recognize the unit:
        else if(MeasuringUnit.getMeasuringUnit(inputTS.getHeader().getUnit()) == null)
        {
            throw new Exception("Unit string, '" + inputTS.getHeader().getUnit() + "', is not recognized internally.");
        }
        //Unit is invalid:
        else if(!MeasuringUnit.getMeasuringUnit(inputTS.getHeader().getUnit())
                              .getType()
                              .equals(MeasuringUnitType.temperature))
        {
            throw new Exception("Units of time series must be temperature (degc, degf, etc.), but was "
                + inputTS.getHeader().getUnit() + ".");
        }

        TimeSeriesArrayTools.convertUnits(inputTS, "degc");
    }

    @Override
    protected SchaakeShuffleApplier constructSchaakeShuffleApplier(final TimeSeriesEnsemble baseEnsemble)
    {
        return new SchaakeShuffleApplier(baseEnsemble, false);
    }
}
