package ohd.hseb.hefs.mefp.models.temperature;

import java.util.Arrays;

import ohd.hseb.hefs.mefp.models.ForecastEnsembleCalculator;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.dist.types.NormalDist;
import ohd.hseb.hefs.utils.tools.SamplingTools;
import ohd.hseb.hefs.utils.tools.ThreadedRandomTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * A {@link ForecastEnsembleCalculator} for use with either maximum or minimum temperature.
 * 
 * @author hankherr
 */
public class TemperatureForecastEnsembleCalculator implements ForecastEnsembleCalculator
{
    private static final Logger LOG = LogManager.getLogger(TemperatureForecastEnsembleCalculator.class);

//    private static int MAXIMUM_NUMBER_OF_NORMAL_DEVIATES = 1000;

    private final MEFPForecastSource _source;
    private final MEFPFullModelParameters _fullParameters;
    private final int _dayOfYear;
    private final boolean _tmaxFlag;
    //private boolean _stratifiedSampling;
    private SamplingTools.SamplingTechnique _stratifiedSampling;

//    private final double[] _fullRealizationsInNormalSpace = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //vval
//    private final double[] _fullRealizationPDFValues = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //cpdfv
//    private final double[] _fullRealizationCDFValues = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //ccdfv
//    private final double[] _fullRealizationsInOriginalSpace = new double[MAXIMUM_NUMBER_OF_NORMAL_DEVIATES]; //p

    /**
     * Test constructor. After calling this, call
     * {@link #calculateForecastEnsemble(double, int, double, double, double, double, double)} directly.
     */
    protected TemperatureForecastEnsembleCalculator()
    {
        this(null, null, -1, false, SamplingTools.SamplingTechnique.RANDOM);
    }

    /**
     * Constructor is meant to be called one time for each source, with
     * {@link #calculateForecastEnsemble(CanonicalEvent, double, int)} being called one time per {@link CanonicalEvent}
     * for that source.
     * 
     * @param source The source.
     * @param fullParameters The parameters of the model.
     * @param dayOfYear The day of the year for execution.
     * @param tmaxFlag Indicates if it will be run for maximum (true) or minimum temperature data.
     * @param stratifiedSampling Indicates if the sampling should be random or stratified-random.
     */

    public TemperatureForecastEnsembleCalculator(final MEFPForecastSource source,
                                                 final MEFPFullModelParameters fullParameters,
                                                 final int dayOfYear,
                                                 final boolean tmaxFlag,
                                                 //final boolean stratifiedSampling
                                                 final SamplingTools.SamplingTechnique stratifiedSampling)
    {
        _source = source;
        _fullParameters = fullParameters;
        _dayOfYear = dayOfYear;
        _tmaxFlag = tmaxFlag;
        _stratifiedSampling = stratifiedSampling;
    }

    public void setStratifiedSampling(final SamplingTools.SamplingTechnique stratifiedSampling)
    {
        _stratifiedSampling = stratifiedSampling;
    }

    @Override
    public double[] calculateForecastEnsemble(final CanonicalEvent event,
                                              final double canonicalEventValue,
                                              final int numberOfEnsembleMembers) throws Exception
    {
//XXX Forcing parameters to use those for another event
//        if(event.equals(new CanonicalEvent(-1, 151, 180, -1)))
//        {
//            event = new CanonicalEvent(-1, 181, 210, -1);
//        }
        final TemperatureOneSetParameterValues parms = (TemperatureOneSetParameterValues)_fullParameters.getSourceEstimatedModelParameterValues(_source,
                                                                                                                                                _dayOfYear,
                                                                                                                                                event);

        if(_tmaxFlag)
        {
            return calculateForecastEnsemble(canonicalEventValue,
                                             numberOfEnsembleMembers,
                                             parms.getTmaxFcstAvg(),
                                             parms.getTmaxFcstStdDev(),
                                             parms.getTmaxObsAvg(),
                                             parms.getTmaxObsStdDev(),
                                             parms.getTmaxRho());
        }
        return calculateForecastEnsemble(canonicalEventValue,
                                         numberOfEnsembleMembers,
                                         parms.getTminFcstAvg(),
                                         parms.getTminFcstStdDev(),
                                         parms.getTminObsAvg(),
                                         parms.getTminObsStdDev(),
                                         parms.getTminRho());
    }

//    public double[] calculateForecastEnsemble(final CanonicalEvent event,
//                                              final double canonicalEventValue,
//                                              final int numberOfEnsembleMembers,
//                                              final boolean stratifiedSampling) throws Exception
//    {
////XXX Forcing parameters to use those for another event
////        if(event.equals(new CanonicalEvent(-1, 151, 180, -1)))
////        {
////            event = new CanonicalEvent(-1, 181, 210, -1);
////        }
//        final TemperatureOneSetParameterValues parms = (TemperatureOneSetParameterValues)_fullParameters.getSourceEstimatedModelParameterValues(_source,
//                                                                                                                                                _dayOfYear,
//                                                                                                                                                event);
//
//        if(_tmaxFlag)
//        {
//            return calculateForecastEnsemble(canonicalEventValue,
//                                             numberOfEnsembleMembers,
//                                             parms.getTmaxFcstAvg(),
//                                             parms.getTmaxFcstStdDev(),
//                                             parms.getTmaxObsAvg(),
//                                             parms.getTmaxObsStdDev(),
//                                             parms.getTmaxRho(),
//                                             stratifiedSampling);
//        }
//        return calculateForecastEnsemble(canonicalEventValue,
//                                         numberOfEnsembleMembers,
//                                         parms.getTminFcstAvg(),
//                                         parms.getTminFcstStdDev(),
//                                         parms.getTminObsAvg(),
//                                         parms.getTminObsStdDev(),
//                                         parms.getTminRho(),
//                                         stratifiedSampling);
//    }

    protected double[] calculateForecastEnsemble(final double canonicalEventValue,
                                                 final int numberOfEnsembleMembers,
                                                 double forecastMean,
                                                 double forecastStdDev,
                                                 final double obsMean,
                                                 final double obsStdDev,
                                                 double rho)
    {
        //If either is 0, then the call to MEFPModelTools.computeEnsembleMembers will be garbage and will go into an 
        //infinite loop.  Not to mention it should be mathematically impossible.
        if(forecastStdDev == 0)
        {
            throw new IllegalArgumentException("Forecast standard deviation is 0, which is not valid.");
        }
        if(obsStdDev == 0)
        {
            throw new IllegalArgumentException("Observed standard deviation is 0, which is not valid.");
        }

        if((_source != null) && (_source.isClimatologySource()))
        {
            forecastMean = obsMean;
            forecastStdDev = obsStdDev;
            rho = 0;
        }

        //Determine the rho and forecast value used based on limit checking and missing.
        final double usedRho = rho; //xrho
        final double usedForecast = canonicalEventValue; //xfcst
//        if(usedForecast < MISSING_TEMP)
//        {
//            LOG.debug("** Canonical event value is missing; using resampled climatology instead... may cause discontinuity "
//                + "(this message is expected for climatology/historical source).");
//            usedForecast = 0.0;
//            usedRho = 0;
//        }
        // HEFS TASK ASSIGNMENTS #44, remove the checking usedForecast against the forecast mean +/- stddev
//        if((usedForecast > forecastMean + 3 * forecastStdDev) || (usedForecast < forecastMean - 3 * forecastStdDev))
//        {
//            LOG.debug("** Canonical event value, "
//                + usedForecast
//                + ", is outside reasonable range of mean +/- 3*stddev where mean is "
//                + forecastMean
//                + "and stddev is "
//                + forecastStdDev
//                + "; forcing rho to 0, which will cause using resampled climatology (see debug printed next)... may cause discontinuity.");
//            usedForecast = forecastMean;
//            usedRho = 0;
//        }

        //Compute the difference between the used forecast and mean and standardize it.
        final double standardizedDeviationFromMean = (usedForecast - forecastMean) / forecastStdDev; //u

        //Prep work for the generation of ensembles.  Includes determining step to use in standard normal space.
//        final int numberOfRealizations = this._fullRealizationsInNormalSpace.length; //nv
//        final int halfNumberOfRealizations = (numberOfRealizations / 2);
        final double rhoThresh = 0.9999;
        final double normalSpaceStdDevThresh = Math.sqrt(1.0d - rhoThresh * rhoThresh);
//        final double deltaInStdNormalSpace = 5.0d / numberOfRealizations; //dz; 5.0d = z2 - z1 in Fortran
        double conditionalNormalMean, conditionalNormlStdDev; //vavg, vstd

        //Determine the mean and standard deviation of the bivariate conditional normal distribution.
        if(usedRho <= 0)
        {
            LOG.debug("** Negative or zero rho found; using resampled climatology... may cause discontinuity.");
            conditionalNormalMean = 0.0d;
            conditionalNormlStdDev = 1.0d;
        }
        else if(usedRho < rhoThresh)
        {
            conditionalNormalMean = usedRho * standardizedDeviationFromMean;
            conditionalNormlStdDev = Math.sqrt(1.0d - usedRho * usedRho);
        }
        else
        {
            conditionalNormalMean = rhoThresh * standardizedDeviationFromMean;
            conditionalNormlStdDev = normalSpaceStdDevThresh;
        }

        //Generate stratified random sample
        final double[] ensemble = new double[numberOfEnsembleMembers];
        Arrays.fill(ensemble, 0d);
        for(int i = 0; i < numberOfEnsembleMembers; i++)
        {
            double zValue =0.0d;
            double w;
//            if(_stratifiedSampling)
//            {
//                zValue = SamplingTools.generateStratifiedRandomSample(NormalDist.STD_NORM_DIST,
//                                                                            numberOfEnsembleMembers,
//                                                                            i);
//            }
//            else
//            {
//                zValue = ThreadedRandomTools.getRandomNumGen().nextGaussian();
//            }
            
	        switch(_stratifiedSampling) 
	        { 
	            case STRATIFIED_RANDOM: 
	                zValue = SamplingTools.generateSample(NormalDist.STD_NORM_DIST, numberOfEnsembleMembers, i, null, null, _stratifiedSampling);
	                break; 
	            case QUANTILE: 
	                zValue = SamplingTools.generateSample(NormalDist.STD_NORM_DIST, numberOfEnsembleMembers, i, null, null, _stratifiedSampling);
	                break;
	            case RANDOM:
	            	zValue = ThreadedRandomTools.getRandomNumGen().nextGaussian();
	        }

            w = conditionalNormalMean + conditionalNormlStdDev * zValue;
            ensemble[i] = obsMean + obsStdDev * w;
        }

        // Sorting array ensemble into ascending order.
        Arrays.sort(ensemble);

        return ensemble;

        // Determine the step to use in the bivariate conditional normal distribution.
//        double deltaConditionalNormalSpace; //dv
//        if(usedRho <= rhoThresh)
//        {
//            deltaConditionalNormalSpace = conditionalNormlStdDev * deltaInStdNormalSpace;
//        }
//        else
//        {
//            deltaConditionalNormalSpace = 0;
//        }
//
//        //Populate realizations in original space.
//        for(int i = 0; i < numberOfRealizations; i++)
//        {
//            _fullRealizationsInNormalSpace[i] = conditionalNormalMean + (i - halfNumberOfRealizations)
//                * deltaConditionalNormalSpace;
//        }
//
//        double sum = 0d;
//
//        //Populate the _fullRealizationPDFValues and _fullRealizationCDFValues arrays.
//        for(int i = 0; i < numberOfRealizations; i++)
//        {
//            if(usedRho < rhoThresh)
//            {
//                final double normalSpaceValue = (_fullRealizationsInNormalSpace[i] - conditionalNormalMean)
//                    / conditionalNormlStdDev; //zv
//                _fullRealizationPDFValues[i] = NormalDist.STD_NORM_DIST.functionPDF(normalSpaceValue)
//                    * deltaConditionalNormalSpace;
//            }
//            else
//            {
//                _fullRealizationPDFValues[i] = 1.0 / numberOfRealizations;
//            }
//            sum += _fullRealizationPDFValues[i];
//            _fullRealizationCDFValues[i] = sum;
//        }
//
//        //Compute the _fullRealizationsInOriginalSpace values.
//        for(int i = 0; i < numberOfRealizations; i++)
//        {
//            _fullRealizationCDFValues[i] = _fullRealizationCDFValues[i] / sum;
//            _fullRealizationPDFValues[i] = _fullRealizationPDFValues[i] / sum;
//            _fullRealizationsInOriginalSpace[i] = obsMean + obsStdDev * _fullRealizationsInNormalSpace[i];
//        }
//
//        //Generate the forecast ensemble
//        return MEFPModelTools.computeEnsembleMembers(0.0d,
//                                                     numberOfEnsembleMembers,
//                                                     _fullRealizationsInOriginalSpace,
//                                                     _fullRealizationCDFValues);

    }

//    subroutine tfcst2ensem (ilog,fcst,fcstavg,fcststd,
//   x                        obsavg,obsstd,rho,
//   x                        ntt,genavg,genstd,vmax,exprobmin,tt,
//   x                        tmin,tmax)
//c
//c     produce a distribution of precipitation values, tt, given the
//c     deterministic forecast, fcst.
//c
//c  Input Arguments:
//c     ilog    = output device
//c     fcst    = deterministic forecast
//c     favg   = climatological mean of fcst 
//c     fstd    = climatological std of fcst 
//c     obsavg = climatological mean of obs 
//c     obsstd  = climatological std of obs 
//c     rho     = coefficient of correlation between fcst and obs
//c     ntt     = number of values of tt (number of ensemble members)
//c
//c  Output Arguments:
//c     genavg  = average value of forecast ensemble distribution (internal)
//c     genstd  = std of forecast ensemble distribution (internal)
//c     vmax    = maximum value of forecast ensemble distribution (internal)
//c     exprobmin = excedence probability associated with vmax
//c         note:  The output ensemble distribution is tt and has ntt members.  
//c                This is computed from a more detailed internal distribution 
//c                with nvmax members that has a maximum value of vmax
//c     tt      = vector of precipitation values sorted with tt(1) = smallest 
//c               value
//c     tmin    = value climatological cdf probability of observing a value 
//c               corresponding to the first (lowest) point on the internal 
//c               ensemble forecast distribution
//c     tmax    = value climatological cdf probability of observing a value 
//c               corresponding to the last (highest) point on the internal 
//c               ensemble forecast distribution
//c
//    parameter (nvmax=1000)
//    real*4 tt(ntt),p(nvmax),vval(nvmax)
//    real*8 cpdfv(nvmax),ccdfv(nvmax),sum,u
//    real*8 f1,ufcst,exprobmin,obscdfmax
//c
//    xrho = rho
//    xfcst = fcst
//    if (fcst.lt.-90.) then
//      xfcst = 0.
//      xrho = 0.
//    endif
//    fmax = fcstavg + 3*fcststd
//    fmin = fcstavg - 3*fcststd
//    if (xfcst.gt.fmax.or.xfcst.lt.fmin) then
//      xfcst = fcstavg
//      xrho = 0.
//    endif
//    u = (xfcst - fcstavg)/fcststd
//    genavg = 0.
//    genstd = 0.
//    nv = nvmax
//c
//c     compute conditional distribution of bivariate std normal
//c     variable, v, given the other std normal value is equal to u.
//c
//c     u     = given value of correlated standard normal deviate
//c     rho   = coefficient of correlation between u and v
//c     nv    = number of values of vval
//c     vmin  = minimum value of vval range
//c     vmax  = maximum value of vval range
//c     vval  = values of standard normal deviate v
//c     cpdfv = probability density function of v
//c     ccdfv = cumulative distribution function of v
//c
//    rhothresh = 0.9999
//    vstdthresh = sqrt(1. - rhothresh**2)
//    theta = 0.25
//    z1 = -2.5
//    z2 = 2.5
//    dz = (z2 - z1)/float(nv)
//    if (xrho.le.0.) then
//      vavg = 0.
//      vstd = 1.
//    elseif (xrho.lt.rhothresh) then
//      vavg = xrho*u  !  conditional mean of v given u
//      vstd = sqrt(1. - xrho**2)  !  conditional std dev of v given u
//    else
//      vavg = rhothresh*u
//      vstd = vstdthresh
//    endif
//c
//    nv2 = nv/2
//    if (xrho.le.rhothresh) then
//      dv = vstd*dz
//    else
//      dv = 0.
//    endif
//    do i=1,nv
//      vval(i) = vavg + (i-1-nv2)*dv
//    enddo
//c 
//    sum = 0.
//    if (xrho.lt.rhothresh) then
//      do i=1,nv
//        zv = (vval(i) - vavg)/vstd  !  standardized conditional v
//        cpdfv(i) = xgausspdf(zv)*dv
//        sum = sum + cpdfv(i)
//        ccdfv(i) = sum
//      enddo
//    else
//      do i=1,nv
//        cpdfv(i) = 1./float(nv)
//        sum = sum + cpdfv(i)
//        ccdfv(i) = sum
//      enddo
//    endif
//    sum = ccdfv(nv)
//    do i=1,nv
//      ccdfv(i) = ccdfv(i)/sum
//      cpdfv(i) = cpdfv(i)/sum
//    enddo
//c 
//    vmax = vval(nv)
//    obscdfmax = xgaussp(vmax)
//    exprobmin = 1. - obscdfmax
//c
//    avg = 0.
//    std = 0.
//    do i=1,nv
//      p(i) = obsavg + obsstd*vval(i)
//      avg = avg + p(i)*cpdfv(i)
//      std = std + (p(i)**2)*cpdfv(i)
//    enddo
//    std = sqrt(abs(std - avg**2))
//c XXX
//    genavg = avg
//    genstd = std
//    tmin = p(1)
//    tmax = p(nv)
//    pz = 0.
//    call xextractp (nv,p,ccdfv,pz,ntt,tt)
//    return
//    end

}
