package ohd.hseb.hefs.mefp.models.temperature;

import static java.lang.Math.abs;
import static java.lang.Math.pow;
import static java.lang.Math.sqrt;

import java.util.List;

import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.tools.HEFSTools;

/**
 * Calculates parameters for the MEFP temperature algorithm. It also provides a method to populate a
 * {@link TemperatureOneSetParameterValues} object based on the computed parameters. That object can then be put into
 * the {@link FullModelParameters} for the appropriate day-of-year and canonical event.
 * 
 * @author hank.herr
 */
public class TemperatureParameterCalculator
{
    private double _obsAvg = Double.NaN;
    private double _obsStdDev = Double.NaN;
    private double _fcstAvg = Double.NaN;
    private double _fcstStdDev = Double.NaN;
    private double _rmsFcst = Double.NaN; //Not needed for MEFP calculations, so it is not in the parm file.
    private double _rho = Double.NaN;
    private int _numObs = -1;

    private String _questionableMessage = null;

    public String getQuestionableMessage()
    {
        return _questionableMessage;
    }

    /**
     * @param forecasts Forecast event values.
     * @param observations Observed event values.
     * @param tmax Only used for generating a questionable message that indicates if it is tmin or tmax.
     */
    public void calculate(final List<Float> forecasts, final List<Float> observations, final boolean tmax)
    {
        _numObs = observations.size();
        _obsAvg = 0;
        _obsStdDev = 0;
        _fcstAvg = 0;
        _fcstStdDev = 0;
        _rmsFcst = 0;
        _rho = 0;

        for(int i = 0; i < _numObs; i++)
        {
            _obsAvg += observations.get(i);
            _obsStdDev += pow(observations.get(i), 2);
            _fcstAvg += forecasts.get(i);
            _fcstStdDev += pow(forecasts.get(i), 2);
            _rmsFcst += pow(observations.get(i) - forecasts.get(i), 2);
            _rho += (double)observations.get(i) * (double)forecasts.get(i);
        }

        _obsAvg /= _numObs;
        _obsStdDev = sqrt(abs(_obsStdDev / _numObs - pow(_obsAvg, 2)));
        _fcstAvg /= _numObs;
        _fcstStdDev = sqrt(abs(_fcstStdDev / _numObs - pow(_fcstAvg, 2)));
        _rmsFcst = sqrt(abs(_rmsFcst / _numObs));

        if(_obsStdDev > 0 && _fcstStdDev > 0)
        {
            _rho /= _numObs;
            _rho -= _obsAvg * _fcstAvg;
            _rho /= _obsStdDev * _fcstStdDev;
        }
        else
        {
            _rho = 0;
        }

        if(_rho < 0)
        {
            _questionableMessage = HEFSTools.makeParameterId(HEFSTools.TEMPERATURE_DATA_TYPE_STRING, false, tmax)
                + ": Negative correlation coefficient computed for data: " + _rho;
        }
    }

    public String generateTestOutputString()
    {
        return "Temperature Parameters: _obsAvg = " + _obsAvg + "; _obsStdDev = " + _obsStdDev + "; _fcstAvg = "
            + _fcstAvg + "; _fcstStdDev = " + _fcstStdDev + "; _rmsFcst = " + _rmsFcst + "; _rho = " + _rho
            + "; _numObs = " + _numObs + ".";
    }

    public void populateOneSetParameterValues(final TemperatureOneSetParameterValues values, final boolean tmax)
    {
        if(tmax)
        {
            values.setTmaxFcstAvg(this._fcstAvg);
            values.setTmaxFcstStdDev(this._fcstStdDev);
            values.setTmaxNumObs(_numObs);
            values.setTmaxObsAvg(this._obsAvg);
            values.setTmaxObsStdDev(this._obsStdDev);
            values.setTmaxRho(this._rho);
        }
        else
        {
            values.setTminFcstAvg(this._fcstAvg);
            values.setTminFcstStdDev(this._fcstStdDev);
            values.setTminNumObs(_numObs);
            values.setTminObsAvg(this._obsAvg);
            values.setTminObsStdDev(this._obsStdDev);
            values.setTminRho(this._rho);
        }
    }
}
