package ohd.hseb.hefs.mefp.models.temperature;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.MEFPParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPAlgorithmModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.types.AvgParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.NObsParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.RhoParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.StdDevParameterType;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.GenericMEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPTemperatureEstimationControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.model.OneSetParameterValues;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.SourceModelParameters;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

/**
 * The order of the {@link ModelParameterType} instances returned herein must be kept in sync with those in
 * {@link TemperatureOneSetParameterValues}.
 * 
 * @author hank.herr
 */
public class TemperatureParameterEstimationModel extends MEFPParameterEstimationModel
{
    private static final Logger LOG = LogManager.getLogger(TemperatureParameterEstimationModel.class);
    private final static ModelParameterType TMIN_NOBS_TYPE = new NObsParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID);
    private final static ModelParameterType TMAX_NOBS_TYPE = new NObsParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID);

    /**
     * The position of the TMAX time series in the historical time series returned by {@link #getHistoricalTimeSeries()}
     * after the time series are loaded via {@link #loadHistoricalData()}.
     */
    public final static int TMAX_HISTORICAL_TIME_SERIES_INDEX = 1;

    /**
     * The position of the TMIN time series in the historical time series returned by {@link #getHistoricalTimeSeries()}
     * after the time series are loaded via {@link #loadHistoricalData()}.
     */
    public final static int TMIN_HISTORICAL_TIME_SERIES_INDEX = 0;

    /**
     * Constructor initializes the alogrithm model parameters.
     */
    public TemperatureParameterEstimationModel()
    {
        setAlgorithmModelParameters(new MEFPAlgorithmModelParameters(false));
    }

    /**
     * Extracts the time series from histTSs and rfcTSs that match the given parameterId and combines them, creating a
     * composite time series. The composite will start wiht the histTSs time series and add the rfcTSs time series to
     * fill in missing values and extend it.
     * 
     * @param histTSs
     * @param rfcTSs Null and emtpy lists are allowed. In such cases, the returned array is what is in histTSs.
     * @param parameterId
     * @return Composite time series, or histTSs time series if no rfcTSs time series are provided.
     * @throws IllegalStateException If no time series is found matching the parameterId or too many are found.
     */
    private TimeSeriesArray createCompositeHistoricalTimeSeries(final List<TimeSeriesArray> histTSs,
                                                                final List<TimeSeriesArray> rfcTSs,
                                                                final String parameterId) throws IllegalStateException
    {
        //The base compositeTS is from the Historical time series.  An IllegalStateException will be thrown if none is found.
        final TimeSeriesSorter histSorter = new TimeSeriesSorter(histTSs);
        TimeSeriesSorter tmpSorter = histSorter.restrictViewToParameters(parameterId);
        final TimeSeriesArray compositeTS = tmpSorter.getOnly();

        //That will be combined with the RFC data, but only if exactly one matching time series is available.
        if((rfcTSs == null) || (rfcTSs.size() == 0))
        {
            LOG.warn("Either the RFC source is not included or the RFC data handler returned no time series for location "
                + getEstimatedIdentifier().buildStringToDisplayInTree()
                + " and parameter "
                + parameterId
                + ".  RFC time series will not be used to build composite historical time series.");
            return compositeTS;
        }
        final TimeSeriesSorter rfcSorter = new TimeSeriesSorter(rfcTSs);
        tmpSorter = rfcSorter.restrictViewToParameters(parameterId);
        if(tmpSorter.size() != 1)
        {
            LOG.warn("The RFC data handler returned too many time series for location "
                + getEstimatedIdentifier().buildStringToDisplayInTree() + " and parameter " + parameterId
                + ".  RFC time series will not be used to build composite historical time series.");
            return compositeTS;
        }

        TimeSeriesArrayTools.fillInMissingFromOtherTimeSeries(compositeTS, tmpSorter.getOnly());
        TimeSeriesArrayTools.extendFromOtherTimeSeries(compositeTS, tmpSorter.getOnly());
        return compositeTS;
    }

    /**
     * If problems occur during computation, parameters must still be recorded, but the {@link ModelParameterType}s
     * checked for missing must be set to {@link Double#NaN}.
     * 
     * @param modelParameters
     * @param source
     * @param modelCtlParms
     * @param values
     * @param tmax
     */
    private void calculateParametersForSource(final MEFPForecastSource source,
                                              final SourceCanonicalEventValues values,
                                              final boolean tmax) throws ParameterEstimationException
    {
        //Initialize the gatherers for IPT and EPT.
        final StandardCanonicalEventValuesGatherer gatherer = (StandardCanonicalEventValuesGatherer)source.constructCanonicalEventValuesGatherer(values,
                                                                                                                                                 getEstimationModelCtlParms(),
                                                                                                                                                 getEstimationCtlParms().getSourceControlOptions(source));

        //The computed canonical events, used within a nested loop below.
        final CanonicalEventList computedEvents = values.getForecastValues().getEventsToCompute();

        //Loop through days of the year at a 5 day steps.  For temperature, the user cannot control the step size as for precip.
        for(final int dayOfYear: getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters())
        {
            //Catcg a user cancel before the parameters for each day are computed.
            if(JobMessenger.isCanceled())
            {
                throw new ParameterEstimationException("Parameter estimation has been canceled (2).");
            }

            JobMessenger.madeProgress("Estimating parameters for day " + dayOfYear + " and maximum temperature = "
                + tmax + "...");
            LOG.info("Computing parameters for day of year " + dayOfYear + " for ...");

            //Gather the event values for all events.
            gatherer.gatherEventValues(dayOfYear);

            //For each canonical events, parameters must be computed and stored.
            for(int canonicalEventIndex = 0; canonicalEventIndex < computedEvents.size(); canonicalEventIndex++)
            {
//                LOG.info("Computing parmameters for canonical event #" + canonicalEventIndex + ": "
//                    + computedEvents.get(canonicalEventIndex).toString() + "...");

                //Construct the one day parm storage object.
                final TemperatureOneSetParameterValues dayParmValues = new TemperatureOneSetParameterValues(dayOfYear,
                                                                                                            canonicalEventIndex,
                                                                                                            this,
                                                                                                            source);
                getEstimatedMEFPModelParameters().putValuesInto(dayParmValues); //Pick up any values already in place (like for the other data type).

                //Compute the IPT parameters.  Note that if not enough values were computed, then the IPT parameters
                //will be left as NaN (during construction, they are set to NaN).  This will be handled later.
                final TemperatureParameterCalculator calculator = new TemperatureParameterCalculator();
                if(gatherer.wereEnoughValuesGathered(canonicalEventIndex))
                {
                    calculator.calculate(gatherer.getForecastGatheredEventValues(canonicalEventIndex),
                                         gatherer.getObservedGatheredEventValues(canonicalEventIndex),
                                         tmax);
                }
                else
                {
                    addQuestionableParameterLogMessage(source,
                                                       computedEvents.get(canonicalEventIndex),
                                                       dayOfYear,
                                                       "Not enough data is avaialble for parameter estimation for "
                                                           + HEFSTools.getMinMaxStr(!tmax) + " check control options.");
                }
                calculator.populateOneSetParameterValues(dayParmValues, tmax);
                addQuestionableParameterLogMessage(source,
                                                   computedEvents.get(canonicalEventIndex),
                                                   dayOfYear,
                                                   calculator.getQuestionableMessage());
                if(areParametersMissing(dayParmValues, tmax))
                {
                    //A questionable message should have already been logged if within this if-check.
                    LOG.debug("Unable to compute temperature parameters for day " + dayOfYear + " and event # "
                        + canonicalEventIndex + " (" + computedEvents.get(canonicalEventIndex).toString()
                        + ") probably due to insufficient data.");
                }

                //Put the values for this day into the modelParameters.
                getEstimatedMEFPModelParameters().recordValuesFrom(dayParmValues);
//
//                LOG.info("Done computing parameters for canonical event #" + canonicalEventIndex + ".");
            }
        }
    }

    /**
     * Checks a designated variable in order to determine if the parameters should be considered missing. For TMAX data,
     * its {@link #TMAX_NOBS_TYPE}, for TMIN data its {@link #TMIN_NOBS_TYPE}.
     * 
     * @param sourceModelParameters Parameters to check.
     * @param dayOfYear Day of year for which to check.
     * @param canonicalEventIndex Event index for which to check.
     * @param tmax True to check TMAX parameters, false for TMIN.
     * @return True if the parameters are missing, false if not.
     */
    private boolean areParametersMissing(final MEFPSourceModelParameters sourceModelParameters,
                                         final int dayOfYear,
                                         final int canonicalEventIndex,
                                         final boolean tmax)
    {
        //NOTE: This is purposely coded independently of the other version in order to save time trying to construct a 
        //TemperatureOneSetParameterValues object.
        ModelParameterType checkedType = TMIN_NOBS_TYPE;
        if(tmax)
        {
            checkedType = TMAX_NOBS_TYPE;
        }
        return sourceModelParameters.getParameterValues(checkedType).getValue(dayOfYear, canonicalEventIndex) < 0;
    }

    /**
     * Checks a designated variable in order to determine if the parameters should be considered missing. For TMAX data,
     * its {@link #TMAX_NOBS_TYPE}, for TMIN data its {@link #TMIN_NOBS_TYPE}.
     * 
     * @param values The {@link TemperatureOneSetParameterValues} to check.
     * @param tmax True to check TMAX parameters, false for TMIN.
     * @return True if the parameters are missing, false if not.
     */
    private boolean areParametersMissing(final TemperatureOneSetParameterValues values, final boolean tmax)
    {
        ModelParameterType checkedType = TMIN_NOBS_TYPE;
        if(tmax)
        {
            checkedType = TMAX_NOBS_TYPE;
        }
        return Double.isNaN(values.getValue(checkedType));
    }

    /**
     * Populates all other source model parameters as is appropriate.
     * 
     * @param sourceModelParameters
     */
    private void populateOtherSourceModelParameters(final SourceModelParameters sourceModelParameters)
    {
        final GenericMEFPSourceControlOptions genericCtlParms = (GenericMEFPSourceControlOptions)getEstimationCtlParms().getSourceControlOptions(sourceModelParameters.getForecastSource());
        ((MEFPSourceModelParameters)sourceModelParameters).setNumberOfForecastLeadDays(genericCtlParms.getNumberOfForecastDaysUsed());
        ((MEFPSourceModelParameters)sourceModelParameters).setNumberOfCanonicalEvents(((MEFPSourceModelParameters)sourceModelParameters).getComputedEvents()
                                                                                                                                        .size());
        ((MEFPSourceModelParameters)sourceModelParameters).determineTimesOfComputedCanonicalEvents();
    }

    /**
     * Implements the Fortran filling algorithm for handling days for which parameters cannot be computed due to not
     * having enough observations or an exception of some type.
     * 
     * @param sourceModelParameters The parameters to fill.
     */
    private void fillInValuesForComputationalDaysForSource(final MEFPSourceModelParameters sourceModelParameters,
                                                           final boolean tmaxFlag) throws ParameterEstimationException
    {
        final CanonicalEventList computedEvents = sourceModelParameters.getComputedEvents();
        boolean atLeastOneNonMissing = false;
        final List<Integer> daysOfYear = getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters();

        //For each canonical event...
        for(int canonicalEventIndex = 0; canonicalEventIndex < computedEvents.size(); canonicalEventIndex++)
        {
            //Find the first day with non-missing values.
            int initialDayOfYearIndex = 0;
            for(initialDayOfYearIndex = 0; initialDayOfYearIndex < daysOfYear.size(); initialDayOfYearIndex++)
            {
                if(!areParametersMissing(sourceModelParameters,
                                         daysOfYear.get(initialDayOfYearIndex),
                                         canonicalEventIndex,
                                         tmaxFlag))
                {
                    break;
                }
            }
            if(initialDayOfYearIndex == daysOfYear.size()) //If we make it to the end, then all data is missing.
            {
                LOG.warn("For canonical event # "
                    + canonicalEventIndex
                    + " ("
                    + computedEvents.get(canonicalEventIndex).toString()
                    + "), parameter estimation failed for all days of the year for which it was attempted, likely due to insufficient sample size.");
                continue;
            }
            atLeastOneNonMissing = true;

            //Now, starting from the initialDayOfYear, jump to the next day.  If the parameters for that day are 
            //missing based on checkedType, set them based on the previous day.  However, be sure to only set the 
            //parameters for EPT or IPT, not both, since missing is checked independently for each.
            int workingIndex = daysOfYear.get(initialDayOfYearIndex) + 1;
            if(workingIndex >= daysOfYear.size())
            {
                workingIndex = 0;
            }
            int previousIndex = daysOfYear.get(initialDayOfYearIndex);
            while(workingIndex != initialDayOfYearIndex)
            {
                if(areParametersMissing(sourceModelParameters,
                                        daysOfYear.get(workingIndex),
                                        canonicalEventIndex,
                                        tmaxFlag))
                {
                    final TemperatureOneSetParameterValues previousValues = (TemperatureOneSetParameterValues)getEstimatedMEFPModelParameters().getSourceEstimatedModelParameterValues(sourceModelParameters.getForecastSource(),
                                                                                                                                                                                       daysOfYear.get(previousIndex),
                                                                                                                                                                                       canonicalEventIndex);
                    final TemperatureOneSetParameterValues workingValues = (TemperatureOneSetParameterValues)getEstimatedMEFPModelParameters().getSourceEstimatedModelParameterValues(sourceModelParameters.getForecastSource(),
                                                                                                                                                                                      daysOfYear.get(previousIndex),
                                                                                                                                                                                      canonicalEventIndex);

                    workingValues.copyParameters(previousValues);
                    getEstimatedMEFPModelParameters().recordValuesFrom(workingValues);
                }

                //Go to the next day.
                previousIndex = workingIndex;
                workingIndex++;
                if(workingIndex >= daysOfYear.size())
                {
                    workingIndex = 0;
                }
            }
        }

        if(!atLeastOneNonMissing)
        {
            throw new ParameterEstimationException("No parameters were successfully computed for any canonical events over all days of the year for this source.");
        }
    }

    @Override
    protected boolean estimateParametersForSource(final MEFPForecastSource source) throws ParameterEstimationException
    {
        //Get the control options and check if the source is to be used. 
        final MEFPSourceControlOptions sourceCtlParms = getEstimationCtlParms().getSourceControlOptions(source);

        //Get the source model parameters and initialize it.  We call this even if parameter estimation is not to be performed
        //in order to ensure that the tarball will still include all the files for a source even if not estimated.
        final MEFPSourceModelParameters srcModelParms = getEstimatedMEFPModelParameters().getSourceModelParameters(source);
        srcModelParms.setApplicableModel(this);
        srcModelParms.setupForStoringParametersAndEvents(sourceCtlParms.getNumberOfForecastDaysUsed());

        //Exit if the source is not enabled or the number of days is 0.  By doing it after the setup call above, this means that all files
        //will be output to the tar ball even if the source is not used.  That should simplify things when reading parameters, since
        //I won't need to check for file existence in the tar ball.
        if((!sourceCtlParms.isEnabled()) || (sourceCtlParms.getNumberOfForecastDaysUsed() <= 0))
        {
            LOG.info("Forecast source " + source.getName()
                + " is not to be used or the number of forecast days is 0.  Skipping source.");
            return false;
        }

        //Log messaging starts here since we know parameters are to be estimated.
        LOG.info("Estimating parameters for forecast source " + source.getName() + "...");
        JobMessenger.newMonitorSubJob();
        JobMessenger.setMaximumNumberOfSteps(2 * (getEstimatedMEFPModelParameters().generateDaysOfTheYearForWhichToEstimateParameters()
                                                                                   .size() + 1));
        JobMessenger.madeProgress("Loading time series and computing canonical events...");

        //Load the prepared time series for the source outside the tmin/tmax loop.
        //This is wrapped in a try-finally in order to ensure that the subjob is cleared.
        try
        {
            try
            {
                source.getSourceDataHandler().loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
            }
            catch(final Exception e)
            {
                throw new ParameterEstimationException(e.getMessage());
            }

            //Loop over TMIN and TMAX (TMIN first; histIndex = 1).
            for(int typeIndex = 0; typeIndex < 2; typeIndex++)
            {
                //Prep some stuff for parameter estimation.
                final TimeSeriesArray histTS = getHistoricalTimeSeries().get(typeIndex);
                final boolean tmaxFlag = (typeIndex == 1);
                ParameterId fcstParmId = null;
                if(tmaxFlag)
                {
                    fcstParmId = ParameterId.TFMX;
                    LOG.info("Estimating parameters for maximum temperature (TMAX) data...");
                }
                else
                {
                    fcstParmId = ParameterId.TFMN;
                    LOG.info("Estimating parameters for minimum temperature (TMIN) data...");
                }

                //Catch a user cancel before the events are computed.  The next cancel is before each day's parameters are calculated.
                if(JobMessenger.isCanceled())
                {
                    throw new ParameterEstimationException("Parameter estimation has been canceled (1).");
                }

                //For non-climatology forecast sources...
                JobMessenger.madeProgress("Computing canonical events for maximum temperature = " + tmaxFlag + "...");
                final SourceCanonicalEventValues values = srcModelParms.getTemperatureSourceEventValues(tmaxFlag);
                if(source != getMEFPRunInfo().getHistoricalForecastSource())
                {
                    LOG.info("Computing canonical event values for non-climatology source...");
                    values.computeCanonicalEventValues(getEstimatedIdentifier(),
                                                       fcstParmId,
                                                       histTS,
                                                       source.getSourceDataHandler()); //Years are pulled from test scenario control file
                    LOG.info("Computed " + values.getForecastValues().getNumberOfComputationalTimesWithValues()
                        + " forecast canonical event values for source.");
                }
                //Otherwise, compute events based on the historical observed time series...
                else
                {
                    LOG.info("Computing canonical event values for climatology source...");
                    values.computeCanonicalEventValues(histTS);
                    LOG.info("Computed " + values.getForecastValues().getNumberOfComputationalTimesWithValues()
                        + " forecast canonical event values for source.");
                }

                //Calculate the source parameters and populate the modelParameters.
                LOG.info("Calculating parameters...");
                calculateParametersForSource(source, values, tmaxFlag);
                LOG.info("Done calculating parameters.");
            }

            //Populate the other parameters and then fill in missing values.  For the algorithm model parameters, the tmaxFlag is true.
            populateOtherSourceModelParameters(srcModelParms);
            fillInValuesForComputationalDaysForSource(srcModelParms, true);

            LOG.info("Done estimating parameters for forecast source " + source.getName());
        }
        finally
        {
            JobMessenger.clearMonitorSubJob();
        }

        return true;
    }

    /**
     * See {@link #TMIN_HISTORICAL_TIME_SERIES_INDEX} and {@link #TMAX_HISTORICAL_TIME_SERIES_INDEX} for the relative
     * positions in the historical time series attribute for the data. This calls
     * {@link #addHistoricalTimeSeries(TimeSeriesArray)} to add the two time series in order.
     */
    @Override
    protected void loadHistoricalData() throws ParameterEstimationException
    {
        try
        {
            getMEFPRunInfo().getHistoricalDataHandler()
                            .loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
        }
        catch(final Exception e)
        {
            throw new ParameterEstimationException(e.getMessage());
        }

        final List<TimeSeriesArray> histTSs = Lists.newArrayList(getMEFPRunInfo().getHistoricalDataHandler()
                                                                                 .getLoadedObservedTimeSeries(getEstimatedIdentifier()));

        //Get RFC obs data, if any, but only if RFC is included as a forecast source (???)
        List<TimeSeriesArray> rfcTSs = new ArrayList<TimeSeriesArray>();
        if(getMEFPRunInfo().getRFCForecastDataHandler() != null)
        {
            if(this.getEstimationCtlParms()
                   .getSourceControlOptions(new RFCForecastSource())
                   .isSourceUsedInParameterEstimation())
            {
                try
                {
                    getMEFPRunInfo().getRFCForecastDataHandler()
                                    .loadPreparedTimeSeries(Lists.newArrayList(getEstimatedIdentifier()));
                    rfcTSs = Lists.newArrayList(getMEFPRunInfo().getRFCForecastDataHandler()
                                                                .getLoadedObservedTimeSeries(getEstimatedIdentifier()));

                    //If the sizes don't match, we have a problem... do not continue.  The sizes represent the number of data types, which
                    //should be two: TMIN, TMAX.
                    if(rfcTSs.size() != histTSs.size())
                    {
                        LOG.warn("The number of RFC observed time series returned for "
                            + getEstimatedIdentifier().buildStringToDisplayInTree() + ", " + rfcTSs.size()
                            + ", does not equal that found using " + "the historical data handler, " + histTSs.size()
                            + ".  RFC time series will not be used to build composite historical time series.");
                        rfcTSs.clear();
                    }
                }
                catch(final Exception e)
                {
                    LOG.warn("RFC forecasts are being used, but a problem was encountered trying to load RFC observed time series: "
                        + e.getMessage());
                }
            }
            else
            {
                LOG.warn("RFC forecasts will not be used in construction of composite historical "
                    + "time series because the source is not being used to estimate parameters.");
            }
        }

        //Create the composite time series and add them to the superclass storage device.
        try
        {
            //TMIN IS ALWAYS ADDED FIRST!!! Must be consistent with the constants above!!!
            addHistoricalTimeSeries(createCompositeHistoricalTimeSeries(histTSs,
                                                                        rfcTSs,
                                                                        HEFSTools.DEFAULT_TMIN_PARAMETER_ID));
            addHistoricalTimeSeries(createCompositeHistoricalTimeSeries(histTSs,
                                                                        rfcTSs,
                                                                        HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
        }
        catch(final IllegalStateException e)
        {
            throw new ParameterEstimationException("Cannot build historical time series due to an unexpected exception creating them: "
                + e.getMessage());
        }
    }

    @Override
    public ControlOption createControlOptions()
    {
        return new MEFPTemperatureModelControlOptions();
    }

    @Override
    public List<ModelParameterType> getAllParametersRequiredForModel(final ForecastSource source)
    {
        final List<ModelParameterType> parameters = new ArrayList<ModelParameterType>();
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false));
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, true));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, true));
        parameters.add(new RhoParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID));
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, false));
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, true));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, false));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, true));
        parameters.add(new RhoParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
        return parameters;
    }

    @Override
    public List<ModelParameterType> getObservationParametersRequiredForModel(final ForecastSource source)
    {
        final List<ModelParameterType> parameters = new ArrayList<ModelParameterType>();
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID));
        parameters.add(new AvgParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, false));
        parameters.add(new StdDevParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, false));
        parameters.add(new NObsParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
        return parameters;
    }

    @Override
    public MEFPTemperatureModelControlOptions getEstimationModelCtlParms()
    {
        return (MEFPTemperatureModelControlOptions)super.getEstimationModelCtlParms();
    }

    @Override
    public Type getDataType()
    {
        return ParameterId.Type.TEMPERATURE;
    }

    @Override
    public OneSetParameterValues initializeOneSetParameterValues(final ForecastSource source,
                                                                 final int dayOfYear,
                                                                 final int secondaryIndex)
    {
        return new TemperatureOneSetParameterValues(dayOfYear, secondaryIndex, this, (MEFPForecastSource)source);
    }

    /**
     * @param runInfo {@link MEFPParameterEstimatorRunInfo} that provides the needed data handlers for loading time
     *            series.
     * @param identifier Identifier for which to load historical time series.
     * @return The historical time series, TMIN and TMAX (in that order), used in parameter estimation for the provided
     *         identifier. It is a combination of historical with RFC extensions.
     * @throws ParameterEstimationException See {@link #loadHistoricalData()}.
     */
    public static List<TimeSeriesArray> retrieveHistoricalData(final MEFPParameterEstimatorRunInfo runInfo,
                                                               final LocationAndDataTypeIdentifier identifier) throws ParameterEstimationException
    {
        if(!identifier.isTemperatureDataType())
        {
            throw new IllegalArgumentException("Identifier " + identifier.buildStringToDisplayInTree()
                + " is not a temperature identifier.");
        }
        final TemperatureParameterEstimationModel model = new TemperatureParameterEstimationModel();
        model.setEstimatedIdentifier(identifier);

        final MEFPTemperatureEstimationControlOptions ctlOptions = (MEFPTemperatureEstimationControlOptions)MEFPTools.constructEstimationControlOptions(identifier.getParameterIdType(),
                                                                                                                                                        runInfo.getForecastSources());
        ctlOptions.getSourceControlOptions(new RFCForecastSource()).setEnabled(true);
        ((GenericMEFPSourceControlOptions)ctlOptions.getSourceControlOptions(new RFCForecastSource())).setDaysUsed(1);
        model.setEstimationCtlOptions(ctlOptions);

        model.setMEFPRunInfo(runInfo);
        model.loadHistoricalData();
        return model.getHistoricalTimeSeries();
    }
}
