package ohd.hseb.hefs.mefp.pe.acceptance;

import java.io.File;
import java.io.IOException;

import nl.wldelft.fews.common.config.GlobalProperties;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.status.BooleanStatus;
import ohd.hseb.hefs.utils.tools.FileTools;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Handler for parameter files
 * 
 * @author Xuning.Tan
 */
public class MEFPIdentifierFileHandler
{
    ParameterEstimatorRunInfo _runInfo;
    private final File _destPath;
    private static final Logger LOG = LogManager.getLogger(MEFPIdentifierFileHandler.class);

    protected MEFPIdentifierFileHandler(final ParameterEstimatorRunInfo runInfo)
    {
        _runInfo = runInfo;

        String destPath = null;
        try
        {
            destPath = GlobalProperties.resolvePropertyTags(GlobalProperties.get("MEFP_ROOT_DIR"));
        }
        catch(final Exception e)
        {
            LOG.warn("Unable to find or parse MEFP_ROOT_DIR global property; using test version: " + e.getMessage());
        }

        if(destPath == null)
        {
            destPath = "testoutput" + File.separator + "guiTestOutput" + File.separator + "mefpRootDir"
                + File.separator + "mefpParameters";
        }
        else
        {
            destPath += File.separator + "mefpParameters";
        }
        _destPath = new File(destPath);

        LOG.info("Destination path for accepted parameters: " + _destPath);
    }

    /**
     * Check whether the parameter file has been estimated.
     * 
     * @param identifier
     * @return true if parameter file exists in the run area, false otherwise
     */

    public boolean isFileAvailable(final LocationAndDataTypeIdentifier identifier)
    {
        final File srcFile = _runInfo.getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);
        if(srcFile.exists())
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Get the estimation and acceptance status of the parameter file
     * 
     * @param identifier
     * @return
     */
    public BooleanStatus getFileStatus(final LocationAndDataTypeIdentifier identifier)
    {
        final File srcFile = _runInfo.getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);
        final File destFile = FileTools.newFile(_destPath, srcFile.getName());

        if(!destFile.exists() || !srcFile.exists())
        {
            return BooleanStatus.get(false);
        }
        else
        {
            // return true if the destFile exists and has been modified more recently than the srcFile
            if(destFile.lastModified() >= srcFile.lastModified())
            {
                return BooleanStatus.get(true);
            }
            else
            {
                return BooleanStatus.get(null);
            }
        }
    }

    /**
     * Accept the estimated parameters by placing them under mefpRootDir/parameters
     * 
     * @param identifier
     */
    public void copyFileOver(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File srcFile = _runInfo.getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);

        if(this.isFileAvailable(identifier))
        {
            if(!_destPath.exists())
            {
                throw new Exception("Destination for parameters, " + _destPath.getAbsolutePath() + ", does not exist.");
            }
            try
            {
                // Copy the contents of the srcFile to a file of the same name in the destDir directory.
                // The destDir is created if it does not exist.  We don't want that, so the above check was added.
                // If the destination file exists, then this method will overwrite it.
                FileUtils.copyFileToDirectory(srcFile, _destPath, false);
            }
            catch(final IOException e)
            {
                throw new Exception("Unable to copy file " + srcFile.getAbsolutePath() + " to destination "
                    + _destPath.getAbsolutePath() + ": " + e.getMessage());
            }
            LOG.info(srcFile.getName() + " has been copied over to " + _destPath);
        }
        else
        {
            throw new Exception(srcFile + " does not exist. Please estimate the parameters.");
        }
    }
}
