package ohd.hseb.hefs.mefp.pe.core;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.ExecutorService;

import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimatedParametersFileHandler;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.sources.historical.HistoricalDataHandler;
import ohd.hseb.hefs.mefp.sources.historical.HistoricalForecastSource;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCDataOptions;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastDataHandler;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPPESFTPSettings;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventsManager;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.EstimationLogFileHandler;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.sources.ForecastSourceTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifierList;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.ParameterId.Type;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

/**
 * Adds additional run information objects to the defaults provided in ParameterEstimatorRunInfo. Includes the
 * following: instances of the precipitation and temperature models, estimated parameters file handler, accepted zip
 * file handler, index entry file handler, estimation log file handler, working canonical events file handler, gridded
 * data SFTP settings, and RFC Forecast data options.<br>
 * <br>
 * For MEFPPE, working identifiers are initialized in three steps: (1) constructed based on the historical data handler
 * found time series, (2) overridden based on the system file settings, and (3) overridden again based on imported
 * old-format index files. The canonical events are initialized in two steps: (1) constructed based on system file
 * settings (possibly empty) and (2) overridden based on imported old-format canonical event file. The control
 * parameters are initialized based on the system file. <br>
 * <br>
 * Note that the steps to override based on index file or canonical events file must be called by an external
 * application since not all instance of this object will include importing those values.
 * 
 * @author hank.herr
 */
public class MEFPParameterEstimatorRunInfo extends ParameterEstimatorRunInfo
{
    private static final Logger LOG = LogManager.getLogger(MEFPParameterEstimatorRunInfo.class);

    public final static String FORECAST_SOURCES_DEFINITION_FILENAME = "forecastSourcesDefinition.xml";

    /**
     * Records the canonical events, including any changes created by the user.
     */
    private CanonicalEventsManager _canonicalEventsMgr;

    /**
     * Records information needed to ftp and acquire processed ASCII grid files for MEFP.
     */
    private final MEFPPESFTPSettings _griddedDataSFTPSettings;

    /**
     * Records options for the RFC Forecast source.
     */
    private RFCDataOptions _rfcDataOptions;

    /**
     * This constructor can be called to acquire the default run-time information stored as a system resource under
     * nonsrc. It will only initialize and read the attributes necessary to process that XML file.
     */
    private MEFPParameterEstimatorRunInfo() throws Exception
    {
        super();
        _griddedDataSFTPSettings = new MEFPPESFTPSettings("griddedDataSFTPParameters", "", "");
        _canonicalEventsMgr = new CanonicalEventsManager(getBaseDirectory(), getSystemFilesDirectory(), this);

        //Load default run-time info
        final InputStream stream = ClassLoader.getSystemResource(getDefaultRunInformationXMLFile()).openStream();
        if(stream == null)
        {
            throw new FileNotFoundException("No default runtime information found as system resource: "
                + getDefaultRunInformationXMLFile());
        }
        XMLTools.readXMLFromStreamAndClose(stream, false, this); //Closes the stream for me!
    }

    /**
     * Standard constructor to call for operationally usable run-time info.
     * 
     * @param executor {@link ExecutorService} used to initialize info from system files.
     * @param baseDirectory The base directory for the MEFPPE.
     * @param configDirectory The config directory corresponding to the stand-alone running the MEFPPE.
     * @throws Exception
     */
    public MEFPParameterEstimatorRunInfo(final ExecutorService executor,
                                         final File baseDirectory,
                                         final File configDirectory) throws Exception
    {
        super(executor, baseDirectory, configDirectory);

        _griddedDataSFTPSettings = new MEFPPESFTPSettings("griddedDataSFTPParameters", "", "");
        loadGriddedDataFTPSettingsFromStandardFile();
    }

    private void loadGriddedDataFTPSettingsFromStandardFile() throws Exception
    {
        final File standardFile = new File(getBaseDirectory() + "/.systemFiles/griddedDataSFTPParameters.xml");
        if(standardFile.exists())
        {
            LOG.info("Reading gridded data ftp settings...");
            XMLTools.readXMLFromFile(standardFile, this._griddedDataSFTPSettings);
            LOG.info("Done reading gridded data ftp settings.");
        }
    }

    /**
     * Load the forecast sources from a plug-in file. If no such file exists, use the default via
     * {@link MEFPTools#instantiateDefaultMEFPForecastSources()}.
     * 
     * @throws Exception
     */
    private void loadForecastSources() throws Exception
    {
        final File standardFile = new File(getBaseDirectory() + "/.systemFiles/" + FORECAST_SOURCES_DEFINITION_FILENAME);
        if(standardFile.exists())
        {
            LOG.info("Loading forecast source definitions from " + standardFile.getAbsolutePath() + "...");
            final List<MEFPForecastSource> sources = new ArrayList<>();
            ForecastSourceTools.readSourcesFromXML(this, standardFile, sources);

            for(final MEFPForecastSource source: sources)
            {
                addForecastSource(source);
            }
            LOG.info("DONE loading forecast source definitions from " + standardFile.getAbsolutePath() + ".");
        }
        else
        {
            LOG.info("Forecast source definition file " + standardFile.getAbsolutePath()
                + " was not found.  Using default sources.");
            //Order is important in terms of consistent GUI appearance, but not in terms of parameters any more.
            //However, it is important in terms of application when generating an ensemble as it determines
            //preferential ordering for combining forecast ensembles from different sources. 
            for(final MEFPForecastSource source: MEFPTools.instantiateDefaultMEFPForecastSources())
            {
                addForecastSource(source);
            }
        }

        //Make sure the historical source exists.
        if(getHistoricalForecastSource() == null)
        {
            LOG.warn("This Historical forecast source was not included in the forecastSourceDefinition.xml file.  It will be added now.");
            addForecastSource(new HistoricalForecastSource());
        }
    }

    /**
     * This method will result in files being removed from under the import directory. As such, only call this when
     * actually executing the main GUI. DO NOT RUN THIS DURING TESTING!!!
     */
    public void importOldCanonicalEvents()
    {
        // Canonical events, first.  Messaging is done within the loadFromImportFiles method.
        try
        {
            _canonicalEventsMgr.loadFromImportFiles();
        }
        catch(final Exception e)
        {
            LOG.warn("At least one import canonical event file was improperly formatted; no events are imported, but all import files are still removed!  Message: "
                + e.getMessage());
        }
    }

    /**
     * @param type The type of parameter to examine.
     * @return List of the forecast sources that are used... for which parameters are to be estimated.
     */
    public List<? extends MEFPForecastSource> getUsedForecastSources(final ParameterId.Type type)
    {
        final List<MEFPForecastSource> list = Lists.newArrayList();
        final EstimationControlOptions typeParms = getEstimationControlOptions(type);
        for(final MEFPForecastSource source: this.getForecastSources())
        {
            final MEFPSourceControlOptions parm = (MEFPSourceControlOptions)typeParms.getControlOptions(source);
            if(parm.isEnabled() && parm.getNumberOfForecastDaysUsed() > 0)
            {
                list.add(source);
            }
        }
        return list;
    }

    /**
     * @return Calls {@link #getSourceControlOptions(Type, MEFPForecastSource)} and returns the value of a call to
     *         {@link MEFPSourceControlOptions#isSourceUsedInParameterEstimation()}.
     */
    public boolean isForecastSourceUsedInEstimation(final ParameterId.Type type, final MEFPForecastSource source)
    {
        final MEFPSourceControlOptions parm = getSourceControlOptions(type, source);
        return parm.isSourceUsedInParameterEstimation();
    }

    /**
     * @return The {@link MEFPSourceControlOptions} for the provided {@link Type} and {@link MEFPForecastSource},
     *         acquired from
     */
    public MEFPSourceControlOptions getSourceControlOptions(final ParameterId.Type type, final MEFPForecastSource source)
    {
        return (MEFPSourceControlOptions)super.getSourceControlOptions(type, source);
    }

    public HistoricalDataHandler getHistoricalDataHandler()
    {
        return getHistoricalForecastSource().getSourceDataHandler();
    }

    public RFCForecastDataHandler getRFCForecastDataHandler()
    {
        if(getRFCForecastSource() == null)
        {
            return null;
        }
        return getRFCForecastSource().getSourceDataHandler();
    }

    public HistoricalForecastSource getHistoricalForecastSource()
    {
        return (HistoricalForecastSource)getForecastSource(HistoricalForecastSource.SOURCE_ID);
    }

    /**
     * @return The {@link RFCForecastSource} or null if it does not exist.
     */
    public RFCForecastSource getRFCForecastSource()
    {
        return (RFCForecastSource)getForecastSource(RFCForecastSource.SOURCE_ID);
    }

    public CanonicalEventsManager getCanonicalEventsMgr()
    {
        return _canonicalEventsMgr;
    }

    public MEFPPESFTPSettings getGriddedDataSFTPSettings()
    {
        return this._griddedDataSFTPSettings;
    }

    public RFCDataOptions getRfcDataOptions()
    {
        return _rfcDataOptions;
    }

    @Override
    protected void initializeSourcesHandlersControlOptionsFileAndAvailableIdentifiers() throws Exception
    {
        LOG.info("Initialize MEFPPE run-time information from the system files and run area...");
        //        getSourceDataHandlers().clear();

        //Load the forecast sources is stop one.
        loadForecastSources();

        // Add any identifiers that do not already exist in the _identifiers list.
        for(final LocationAndDataTypeIdentifier identifier: getHistoricalDataHandler().getIdentifiersWithData())
        {
            if(!getAvailableIdentifiers().contains(identifier))
            {
                getAvailableIdentifiers().add(identifier);
            }
        }

        //The estimated parameter file handlers: regular and backup.
        setEstimatedParametersFileHandler(new MEFPEstimatedParametersFileHandler(this));

        //The zip file handlers
        //setAcceptedZipFileHandler(new AcceptedParameterFileHandler(this, MEFPIdentifierZipFileHandler.makeCache(this)));

        //Everything else...
        _canonicalEventsMgr = new CanonicalEventsManager(getBaseDirectory(), getSystemFilesDirectory(), this);

        //Log file handler must be created and have its working log file overridden.
        setEstimationLogFileHandler(new EstimationLogFileHandler(getBaseDirectory()));
        getEstimationLogFileHandler().setWorkingLogFile(new File(getBaseDirectory().getAbsolutePath() + File.separator
            + "working" + File.separator + "log" + File.separator + "mefppe.ctl.log"));

        LOG.info("Done initializing MEFPPE run-time information.");
    }

    @Override
    public MEFPEstimationControlOptions getEstimationControlOptions(final ParameterId.Type type)
    {
        return (MEFPEstimationControlOptions)super.getEstimationControlOptions(type);
    }

    @Override
    public MEFPEstimationControlOptions getEstimationControlOptions(final LocationAndDataTypeIdentifier identifier)
    {
        return (MEFPEstimationControlOptions)super.getEstimationControlOptions(identifier);
    }

    @Override
    public List<MEFPForecastSource> getForecastSources()
    {
        return ListTools.convertCollection(super.getForecastSources(), (MEFPForecastSource)null);
    }

    @Override
    public List<? extends MEFPSourceDataHandler> getSourceDataHandlers()
    {
        return (List<? extends MEFPSourceDataHandler>)super.getSourceDataHandlers();
    }

    @Override
    public void addForecastSource(final ForecastSource source) throws Exception
    {
        Preconditions.checkArgument(source instanceof MEFPForecastSource,
                                    "%s is not an instance of %s.",
                                    source,
                                    MEFPForecastSource.class.getName());
        super.addForecastSource(source);
    }

    @Override
    public MEFPEstimatedParametersFileHandler getEstimatedParametersFileHandler()
    {
        return (MEFPEstimatedParametersFileHandler)super.getEstimatedParametersFileHandler();
    }

    @Override
    public MEFPEstimatedParametersFileHandler getEstimatedParametersBackupFileHandler()
    {
        return (MEFPEstimatedParametersFileHandler)super.getEstimatedParametersBackupFileHandler();
    }

    @Override
    public ParameterEstimationModel getModel(final LocationAndDataTypeIdentifier identifier)
    {
        return getEstimationControlOptions(identifier).getModel();
    }

    @Override
    public LocationAndDataTypeIdentifierList getCurrentlyWorkingIdentifiers()
    {
        final LocationAndDataTypeIdentifierList identifiersWithData = new LocationAndDataTypeIdentifierList();
        identifiersWithData.addAll(getAvailableIdentifiers());
        identifiersWithData.retainAll(getHistoricalDataHandler().getIdentifiersWithData());
        return identifiersWithData;
    }

    @Override
    public CompositeXMLReader getReader()
    {
        final CompositeXMLReader reader = super.getReader();
        reader.addComponent(_rfcDataOptions);
        reader.addComponent(_canonicalEventsMgr);
        return reader;
    }

    @Override
    public CompositeXMLWriter getWriter()
    {
        final CompositeXMLWriter writer = (CompositeXMLWriter)super.getWriter();
        writer.addComponent(_canonicalEventsMgr);
        writer.addComponent(_rfcDataOptions);
        return writer;
    }

    @Override
    protected void initializeXMLReaders()
    {
        _rfcDataOptions = new RFCDataOptions(getExecutor());
        _rfcDataOptions.register(this);
    }

    @Override
    public void updateAvailableIdentifiers()
    {
        //This only allows for adding identifiers.  The station mapping panel will allow for removing them.
        for(final LocationAndDataTypeIdentifier identifier: this.getHistoricalDataHandler().getIdentifiersWithData())
        {
            if(!getAvailableIdentifiers().contains(identifier))
            {
                getAvailableIdentifiers().add(identifier);
            }
        }
        Collections.sort(getAvailableIdentifiers());
    }

    @Override
    public EnumSet<Type> getSupportedDataTypes()
    {
        return MEFPTools.getSupportedDataTypes();
    }

    @Override
    public EstimationControlOptions constructEstimationControlOptions(final ParameterId.Type type)
    {
        return MEFPTools.constructEstimationControlOptions(type, this.getForecastSources());
    }

    @Override
    public String getDefaultRunInformationXMLFile()
    {
        return "mefppe/staticSystemFiles/defaultRunTimeInformation.xml";
    }

    @Override
    public String getProgramNameForAboutDialog()
    {
        return "MEFPPE";
    }

    /**
     * Tool to acquire the default run-time information. The empty constructor actually does the work.
     * 
     * @return A {@link MEFPParameterEstimatorRunInfo} object that only contains default options. It cannot be used
     *         operationally as many needed attributes are null!
     */
    public static MEFPParameterEstimatorRunInfo loadDefaultRunTimeInformation() throws Exception
    {
        final MEFPParameterEstimatorRunInfo runTimeInformation = new MEFPParameterEstimatorRunInfo();
        return runTimeInformation;
    }

    /**
     * Tests the method {@link #loadDefaultRunTimeInformation()}.
     * 
     * @param args
     */
    public static void main(final String[] args)
    {
        try
        {
            final MEFPParameterEstimatorRunInfo info = MEFPParameterEstimatorRunInfo.loadDefaultRunTimeInformation();
            System.err.println("####>> DONE! >>>");
            System.err.println("####>> " + info.getCanonicalEventsMgr());
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
    }
}
