package ohd.hseb.hefs.mefp.pe.core;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationPEStepProcessor;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.gui.PerformAllStepsJob;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Overrides PerformAllStepsJob because of an extra option, _assumeZeroNumDaysFlag, which if true results in 0 being the
 * number of days assumed for any source data that doesn't exist, avoiding an error in the fotran code. It also it needs
 * to runs the estimation step is a non-standard way to account for hte _assumeZeroNumDaysFlag.
 * 
 * @author hank.herr
 */
public class MEFPPerformAllStepsJob extends PerformAllStepsJob
{
    private static final Logger LOG = LogManager.getLogger(MEFPPerformAllStepsJob.class);

    private boolean _assumeZeroNumDaysFlag = true;

    public MEFPPerformAllStepsJob()
    {
        super();
    }

    /**
     * A special method is needed to handle assigning zeros, which required creating a copy of the control parameters
     * and editing as needed.
     * 
     * @param estStep
     * @param identifier
     * @throws Exception
     */
    private void runEstimationStep(final MEFPEstimationPEStepProcessor estStep,
                                   final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        //Get a working copy of the control parameters.
        final MEFPEstimationControlOptions usedControlOptions = estStep.getTopLevelControlOptions(identifier.getParameterIdType())
                                                                       .clone();

        //Modify the source data handler parameters to have 0 for the number of days if appropriate.
        for(final MEFPForecastSource forecastSource: usedControlOptions.getForecastSources())
        {
            if(this.isCanceled())
            {
                return;
            }

            final MEFPSourceControlOptions parms = usedControlOptions.getSourceControlOptions(forecastSource);

            if(parms.getNumberOfForecastDaysUsed() > 0)
            {
                //If data is not available for a forecast source, then...
                if(!forecastSource.getSourceDataHandler().havePreparedDataFilesBeenCreatedAlready(identifier))
                {
                    //If _assumeZeroNumDaysFlag is true, assign the number of days to zero.  Otherwise,
                    //error out.
                    if(_assumeZeroNumDaysFlag)
                    {
                        LOG.warn("For source " + forecastSource.getName()
                            + ", the prepared data files do not exist, but the number of days used is not zero.");
                        LOG.warn("Setting to zero days so that no parameters are estimated for the source.");

                        parms.setEnabled(false);
                    }
                    else
                    {
                        throw new Exception("Cannot perform parameter estimation because for source "
                            + forecastSource.getName()
                            + " the number of days is not zero but no prepared data was found.");
                    }
                }
            }
        }

        //Ensure that all sources are estimated.
        estStep.setSelectedSource(null);

        //Perform the step and always backup parameters!!!
        estStep.performStep(identifier, usedControlOptions, true);
    }

    @Override
    protected JPanel constructOptionsPanel()
    {
        final JPanel optionPanel = super.constructOptionsPanel();

        final JLabel assumeZeroDaysLabel = new JLabel("<html>All steps are run using one set of estimation options. In the options, if<br>"
            + "a data source is to be used (number of days is not zero), but the data source<br>"
            + "is not found (possible step failure), then should the number of days be set to<br>"
            + "zero or should the Estimation and Acceptance steps be skipped?</html>");
        final JRadioButton assumeZeroDaysYesButton = new JRadioButton("Assume Zero Days");
        final JRadioButton assumeZeroDaysNoButton = new JRadioButton("Skip");
        final ButtonGroup assumeZeroDaysButtonGroup = new ButtonGroup();
        assumeZeroDaysYesButton.setSelected(true);
        assumeZeroDaysButtonGroup.add(assumeZeroDaysYesButton);
        assumeZeroDaysButtonGroup.add(assumeZeroDaysNoButton);
        final JPanel assumeZeroDaysButtonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        assumeZeroDaysButtonPanel.add(assumeZeroDaysYesButton);
        assumeZeroDaysButtonPanel.add(assumeZeroDaysNoButton);
        final JPanel assumeZeroDaysPanel = new JPanel(new BorderLayout());
        assumeZeroDaysPanel.add(assumeZeroDaysLabel, BorderLayout.NORTH);
        assumeZeroDaysPanel.add(assumeZeroDaysButtonPanel, BorderLayout.CENTER);
        assumeZeroDaysPanel.setBorder(BorderFactory.createEtchedBorder());

        final ActionListener assumeZeroActionListener = new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                _assumeZeroNumDaysFlag = assumeZeroDaysYesButton.isSelected();
            }
        };
        assumeZeroDaysYesButton.addActionListener(assumeZeroActionListener);
        assumeZeroDaysNoButton.addActionListener(assumeZeroActionListener);

        optionPanel.add(assumeZeroDaysPanel);

        return optionPanel;
    }

    @Override
    protected void executeStep(final ParameterEstimatorStepProcessor step,
                               final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        if(step instanceof MEFPEstimationPEStepProcessor)
        {
            runEstimationStep((MEFPEstimationPEStepProcessor)step, identifier);
        }
        else
        {
            step.performStep(identifier);
        }
    }
}
