package ohd.hseb.hefs.mefp.pe.estimation;

import java.util.List;

import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.pe.estimation.options.BooleanEstimationOption;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.estimation.options.EditableLeafControlOption;
import ohd.hseb.hefs.pe.estimation.options.IntegerEstimationOption;
import ohd.hseb.hefs.pe.estimation.options.OptionalCompositeControlOption;
import ohd.hseb.hefs.utils.tools.ArrayTools;

import com.google.common.base.Supplier;

/**
 * Generic superclass for MEFP source control parameters.
 * 
 * @author alexander.garbarino
 */
public class GenericMEFPSourceControlOptions extends OptionalCompositeControlOption implements
MEFPSourceControlOptions<List<? extends ControlOption>>
{

    private MEFPForecastSource _forecastSource;

    /**
     * Constructor is only called during cloning.
     */
    private GenericMEFPSourceControlOptions(final String xmlTag)
    {
        super(xmlTag);
    }

    /**
     * A standard constructor to use that does not allow for extra control options.
     * 
     * @param xmlTag XML tag for the control options.
     * @param source The associated {@link MEFPForecastSource}. Only its {@link MEFPForecastSource#getSourceId()} method
     *            is called.
     */
    public GenericMEFPSourceControlOptions(final String xmlTag, final MEFPForecastSource source)
    {
        super(xmlTag,
              new BooleanEstimationOption("using", "Use " + source.getName() + "?", true),
              constructBaseParameters(source));
        _forecastSource = source;
        mapAttributes();
    }

    /**
     * The standard constructor to use allows for additional control options to be specified.
     * 
     * @param xmlTag XML tag for the control options.
     * @param source The associated {@link MEFPForecastSource}. Only its {@link MEFPForecastSource#getSourceId()} method
     *            is called.
     * @param parameters Additional options to include.
     */
    public GenericMEFPSourceControlOptions(final String xmlTag,
                                           final MEFPForecastSource source,
                                           final ControlOption... parameters)
    {
        super(xmlTag,
              new BooleanEstimationOption("using", "Use " + source.getName() + "?", true),
              ArrayTools.concat(constructBaseParameters(source), parameters));
        _forecastSource = source;
        mapAttributes();
    }

    /**
     * Override if you have attributes to map after copying parameters or cloning. {@link #copyFrom(ControlOption)}
     * calls this method after copying the parameters.
     */
    protected void mapAttributes()
    {
    }

    /**
     * A subclass should call this as needed. It is an instance of the applicable forecast source.
     * 
     * @return Forecast source for which these parameter exist.
     */
    protected MEFPForecastSource getForecastSource()
    {
        return _forecastSource;
    }

    @Override
    public boolean isSourceUsedInParameterEstimation()
    {
        return this.isEnabled() && (this.getNumberOfForecastDaysUsed() > 0);
    }

    @Override
    public int getNumberOfForecastDaysUsed()
    {
        return ((Supplier<Integer>)_parameters.get(1)).get();
    }

    @Override
    public int getInitialYear()
    {
        return ((Supplier<Integer>)_parameters.get(2)).get();
    }

    @Override
    public int getFinalYear()
    {
        return ((Supplier<Integer>)_parameters.get(3)).get();
    }

    public void setDaysUsed(final int days)
    {
        setDaysUsed(days, this);
    }

    public void setDaysUsed(final int days, final Object source)
    {
        ((EditableLeafControlOption<Integer>)_parameters.get(1)).set(days, source);
    }

    public void setInitialYear(final int year)
    {
        setInitialYear(year, this);
    }

    public void setInitialYear(final int year, final Object source)
    {
        ((EditableLeafControlOption<Integer>)_parameters.get(2)).set(year, source);
    }

    public void setFinalYear(final int year)
    {
        setFinalYear(year, this);
    }

    public void setFinalYear(final int year, final Object source)
    {
        ((EditableLeafControlOption<Integer>)_parameters.get(3)).set(year, source);
    }

    @Override
    public void assertValid()
    {
        if(getNumberOfForecastDaysUsed() <= 0 && isEnabled())
        {
            throw new IllegalStateException(String.format("%s is enabled, but has %s forecast days used.",
                                                          getXMLTagName(),
                                                          getNumberOfForecastDaysUsed()));
        }
    }

    @Override
    public void copyFrom(final ControlOption base)
    {
        super.copyFrom(base);
        _forecastSource = ((GenericMEFPSourceControlOptions)base)._forecastSource; //Not a copy, but shouldn't matter
        mapAttributes();
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        final GenericMEFPSourceControlOptions copy = new GenericMEFPSourceControlOptions(getXMLTagName());
        copy.copyFrom(this);
        return copy;
    }

    /**
     * Constructs an array of three control parameters: numDays, initialYearOfData, lastYearOfData.
     * 
     * @param source
     * @return
     */
    private static ControlOption[] constructBaseParameters(final MEFPForecastSource source)
    {
        final ControlOption[] parms = new ControlOption[3];

        //Pull the default values from the MEFP forecast source...
        int numDays = 0;
        if(source.getDefaultNumberOfForecastDays() >= 0)
        {
            numDays = source.getDefaultNumberOfForecastDays();
        }
        int initialYear = 1979;
        if((source.getDefaultInitialYear() != Integer.MIN_VALUE)
            && (source.getDefaultInitialYear() != Integer.MAX_VALUE))
        {
            initialYear = source.getDefaultInitialYear();
        }
        int lastYear = 1979;
        if((source.getDefaultLastYear() != Integer.MIN_VALUE) && (source.getDefaultLastYear() != Integer.MAX_VALUE))
        {
            lastYear = source.getDefaultLastYear();
        }

        //Setup the options.
        parms[0] = new IntegerEstimationOption("numDays",
                                               "Number of " + source.getSourceId() + " Forecast Days:",
                                               numDays);
        parms[1] = new IntegerEstimationOption("initialYearOfData",
                                               "Initial Year of Parameter Estimation:",
                                               initialYear);
        parms[2] = new IntegerEstimationOption("lastYearOfData", "Last Year of Parameter Estimation:", lastYear);
        return parms;
    }

}
