package ohd.hseb.hefs.mefp.pe.estimation;

import static com.google.common.collect.Lists.newArrayList;
import static ohd.hseb.hefs.utils.tools.FileTools.newAbsoluteFile;
import static ohd.hseb.hefs.utils.tools.FileTools.newFile;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.pe.estimation.EstimatedParametersFileHandler;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ParameterId;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ImmutableList;

/**
 * MEFP specific instance of EstimatedParametersFileHandler that specifies that the FullModelParameters subclass to use
 * is MEFPFullModelParameters.
 * 
 * @author hank.herr
 */
public class MEFPEstimatedParametersFileHandler extends EstimatedParametersFileHandler
{
    @SuppressWarnings("unused")
    private static final Logger LOG = LogManager.getLogger(MEFPEstimatedParametersFileHandler.class);

    public MEFPEstimatedParametersFileHandler(final MEFPParameterEstimatorRunInfo runInfo)
    {
        super(runInfo);
    }

    /**
     * This goes out and finds the necessary source data handlers for the list maintained here.
     * 
     * @param baseDirectory
     */
    public MEFPEstimatedParametersFileHandler(final File baseDirectory)
    {
        this(newAbsoluteFile(baseDirectory, "parameters"), ImmutableList.<MEFPForecastSource>of());
    }

    public MEFPEstimatedParametersFileHandler(final File baseDirectory,
                                              final List<? extends MEFPForecastSource> forecastSources)
    {
        this(newAbsoluteFile(baseDirectory, "parameters"), ImmutableList.copyOf(forecastSources), true);
    }

    protected MEFPEstimatedParametersFileHandler(final File parametersDirectory,
                                                 final ImmutableList<MEFPForecastSource> forecastSources,
                                                 final boolean createBackup)
    {
        super(parametersDirectory, forecastSources, createBackup);
    }

    /**
     * @throws Exception Thrown if the file cannot be read or does not exist.
     */
    private void checkParameterFileForRead(final LocationAndDataTypeIdentifier identifier, final File parameterFile) throws Exception
    {
        if(!parameterFile.exists())
        {
            throw new FileNotFoundException("Parameter file for identifier " + identifier.buildStringToDisplayInTree()
                + ", " + parameterFile.getAbsolutePath() + ", was not found.");
        }
        if(!parameterFile.canRead())
        {
            throw new FileNotFoundException("Parameter file for identifier " + identifier.buildStringToDisplayInTree()
                + ", " + parameterFile.getAbsolutePath() + ", exists but cannot be read.");
        }
    }

    /**
     * @return The parameters with only the main parameters read in.
     * @throws Exception For standard reasons; example {@link #checkParameterFileForRead(File)}.
     */
    public MEFPFullModelParameters readMainParametersOnly(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File parameterFile = getPrimaryParameterFile(identifier);
        checkParameterFileForRead(identifier, parameterFile);
        final MEFPFullModelParameters parms = constructFullModelParameters(identifier);
        parms.readParametersTarArchive(getPrimaryParameterFile(identifier), new ArrayList<Integer>(), true, false);
        return parms;
    }

    /**
     * @param type Either PRECIPITATION or TEMPERATURE {@link Type}.
     * @return The parameters file directory associated with the given data type.
     */
    public File getParametersDirectory(final ParameterId.Type type)
    {
        switch(type)
        {
            case PRECIPITATION:
                return newAbsoluteFile(getParametersDirectory(), "precip_parms");
            case TEMPERATURE:
                return newAbsoluteFile(getParametersDirectory(), "temp_parms");
            default:
                throw new IllegalArgumentException("Unsupported parameter type: " + type);
        }
    }

    @Override
    protected MEFPEstimatedParametersFileHandler createBackupHandler()
    {
        final File backupDir = FileTools.newAbsoluteFile(getBaseDirectory(), "backupParameters");
        return new MEFPEstimatedParametersFileHandler(backupDir, this.getForecastSources(), false);
    }

    @Override
    public ImmutableList<MEFPForecastSource> getForecastSources()
    {
        return (ImmutableList<MEFPForecastSource>)super.getForecastSources();
    }

    @Override
    public File getParametersDirectory(final LocationAndDataTypeIdentifier identifier)
    {
        return getParametersDirectory(identifier.getParameterIdType());
    }

    @Override
    public List<File> getTrackedFiles(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> results = newArrayList();
        final String location = identifier.getLocationId();

        final File dir = getParametersDirectory(identifier);
        if((!identifier.isPrecipitationDataType()) && (!identifier.isTemperatureDataType()))
        {
            throw new IllegalArgumentException("Unsupported parameter type: " + identifier.getParameterIdType());
        }
        results.add(newFile(dir, MEFPTools.determineParameterFileName(identifier)));
        results.add(newFile(dir, location + ".parms.xml"));
        return results;
    }

    @Override
    public MEFPFullModelParameters readModelParameters(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File parameterFile = getPrimaryParameterFile(identifier);
        checkParameterFileForRead(identifier, parameterFile);
        final MEFPFullModelParameters parms = constructFullModelParameters(identifier);
        parms.readParametersTarArchive(getPrimaryParameterFile(identifier));
        return parms;
    }

    @Override
    public void writeParameterFile(final FullModelParameters parameters) throws Exception
    {
        ((MEFPFullModelParameters)parameters).writeParametersTarArchive(getPrimaryParameterFile(parameters.getIdentifier()),
                                                                        true);
    }

    @Override
    public MEFPFullModelParameters constructFullModelParameters(final LocationAndDataTypeIdentifier identifier)
    {
        return new MEFPFullModelParameters(identifier, getForecastSources());
    }
}
