package ohd.hseb.hefs.mefp.pe.estimation;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.diag.CreateBlockPlotForSelectedLocationsNotice;
import ohd.hseb.hefs.mefp.pe.estimation.diag.CreateParameterBlockDiagnosticStepProcessor;
import ohd.hseb.hefs.mefp.pe.estimation.diag.ParameterQCDiagnosticPanel;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.estimation.EstimationPEStepOptionsPanel;
import ohd.hseb.hefs.pe.gui.PerformAllStepsJob;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.GlobalDialogParent;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.StringTools;

/**
 * Overrides {@link EstimationPEStepOptionsPanel} adding a diagnostics button.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
@HelpFile("PDFHELP:helpManual.pdf#EstimationSubpanelRef")
@GlobalDialogParent
public class MEFPEstimationPEStepOptionsPanel
extends
    EstimationPEStepOptionsPanel
implements CreateBlockPlotForSelectedLocationsNotice.Subscriber
{
    private File _lastBlockPlotCreationDirectory = null;

    public MEFPEstimationPEStepOptionsPanel(final MEFPEstimationPEStepProcessor stepProc)
    {
        super(stepProc.getRunInfo(), stepProc, true);
    }

    @Override
    protected List<Component> getAdditionalComponentsToDisplayInParameterSummaryPanel()
    {
        final List<Component> components = Lists.newArrayList();
        components.add(new DiagnosticsButton());
        return components;
    }

    /**
     * Button that opens up the diagnostics block panel. Its a {@link JobListener} so that it receives a message
     * whenever a parameter file is opened so that it can be enabled.
     * 
     * @author hankherr
     */
    private class DiagnosticsButton extends SelfListeningButton
    implements JobListener
    {

        private DiagnosticsButton()
        {
            super("blockChart20x20");
            setToolTipText("View Parameter Diagnostics");
            setEnabled(false);
        }

        @Override
        public void actionPerformed(final ActionEvent e)
        {
            final ParameterQCDiagnosticPanel panel =
                                                   new ParameterQCDiagnosticPanel((MEFPFullModelParameters)getLoadedParameters());
            panel.registerWithBus(MEFPEstimationPEStepOptionsPanel.this);
            fireDiagnostic(panel);
        }

        @Override
        public void processJobFailure(final Exception exc,
                                      final GenericJob theJob,
                                      final boolean displayMessage)
        {
            this.setEnabled(getLoadedParameters() != null);
        }

        @Override
        public void processSuccessfulJobCompletion(final GenericJob theJob)
        {
            this.setEnabled(getLoadedParameters() != null);
        }

    }

    @Override
    protected void loadOptionsFromParameterFile()
    {
        final int option =
                         JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(MEFPEstimationPEStepOptionsPanel.this),
                                                       "This will update the Estimation Options panel within the Estimations Panel.\n"
                                                           + "Do you wish to also load the canonical events from the parameter file?\n\n"
                                                           + "(Click Yes to load events and update the Canonical Events Subpanel within the Setup Panel.)\n",
                                                       "Load Canonical Events and Continue?",
                                                       JOptionPane.YES_NO_CANCEL_OPTION);
        if(option == JOptionPane.CANCEL_OPTION)
        {
            return;
        }

        //Use the super class version which reads from the basic options file, not the parameters file.
        if(option == JOptionPane.NO_OPTION)
        {
            super.loadOptionsFromParameterFile();
            return;
        }

        final LocationAndDataTypeIdentifier identifier =
                                                       getSelectedIdentifier();
        if((identifier != null)
            && (getStepProcessor().doFilesExist(identifier)))
        {
            try
            {
                final MEFPFullModelParameters parms =
                                                    (MEFPFullModelParameters)getRunInfo().getEstimatedParametersFileHandler()
                                                                                         .readModelParameters(identifier);

                //NOTE that the read in parms may not have the same forecast sources as those inside the run-time information,
                //which is configured by the user through the sources XML.  So, what I need to do here is a source-by-source 
                //copy from the read in parms into run-time information.  Its then followed by a call to copy the defaults
                //into those options so that the newly copied options are properly linked to the default options.
                for(final MEFPForecastSource source: parms.getOrderedForecastSources())
                {
                    getRunInfo().getSourceControlOptions(identifier.getParameterIdType(),
                                                         source)
                                .copyFrom(parms.getEstimationControlOptions()
                                               .getSourceControlOptions(source));

                }
                getRunInfo().copyDefaultEstimationOptionsIntoWorkingOptions();

                //The parameters loaded correctly.  So, clear out the existing events.
                ((MEFPParameterEstimatorRunInfo)getRunInfo()).getCanonicalEventsMgr()
                                                             .getBaseCanonicalEventList(identifier)
                                                             .clear();
                ((MEFPParameterEstimatorRunInfo)getRunInfo()).getCanonicalEventsMgr()
                                                             .getModulationCanonicalEventList(identifier)
                                                             .clear();

                //Copy in the new events, relying on the run-info event bus to update the canonical events table.
                ((MEFPParameterEstimatorRunInfo)getRunInfo()).getCanonicalEventsMgr()
                                                             .copyEvents(parms.getAlgorithmModelParameters()
                                                                              .getBaseCanonicalEvents());
                ((MEFPParameterEstimatorRunInfo)getRunInfo()).getCanonicalEventsMgr()
                                                             .copyEvents(parms.getAlgorithmModelParameters()
                                                                              .getModulationCanonicalEvents());

                //Call other methods as needed primarily for the estimation options stuff, not the events.
                addToUndoIfNeeded(identifier.getParameterIdType());
                updateCurrentlyEditedControlOptionsPanel(getRunInfo().getSelectedIdentifiers());
                refreshLocationSummaryTable();
            }
            catch(final Throwable t)
            {
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                              "Unable to load parameters for "
                                                  + identifier.buildStringToDisplayInTree()
                                                  + ",\nbut the selected locations are for "
                                                  + getRunInfo().getSelectedDataType()
                                                  + ".",
                                              "Selected Parameters are not Compatible",
                                              JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    @Override
    @Subscribe
    public void reactToCreateBlockPlotForSelectedLocationsNotice(final CreateBlockPlotForSelectedLocationsNotice evt)
    {
        final List<? extends StepUnit> units = getStepUnitsToPerform();

        //Check data type of source diagnostic panel with the selected unit.
        if(!getRunInfo().getSelectedDataType()
                        .equals(evt.getSource().getParameterFileDataType()))
        {
            JOptionPane.showMessageDialog(evt.getSource(),
                                          "The diagnostic panel is displaying diagnostics for "
                                              + evt.getSource()
                                                   .getParameterFileDataType()
                                              + ",\nbut the selected locations are for "
                                              + getRunInfo().getSelectedDataType()
                                              + ".",
                                          "Selected Parameters are not Compatible",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }

        //Get the file name prefix.
        final String imageFilePrefix =
                                     JOptionPane.showInputDialog(evt.getSource(),
                                                                 "Specify the prefix used for all images files created.\n"
                                                                     + "Files generated will have the name:\n\n"
                                                                     + "    <prefix>.<locationId>.<parameterId>.png\n\n"
                                                                     + "If no prefix is provided, the files will have the name:\n\n"
                                                                     + "    <locationId>.<parameterId>.png\n",
                                                                 "Specify Image File Name Prefix",
                                                                 JOptionPane.QUESTION_MESSAGE);
        if(imageFilePrefix == null)
        {
            return;
        }

        //Choose the output directory.
        final JFileChooser chooser =
                                   new JFileChooser(_lastBlockPlotCreationDirectory);
        chooser.setDialogTitle("Choose Directory to Contain Output Image Files");
        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        final int result =
                         chooser.showDialog(evt.getSource(), "Generate Images");
        if(result == JFileChooser.CANCEL_OPTION)
        {
            return;
        }
        File outputDirectory = chooser.getSelectedFile();
        if(!outputDirectory.isDirectory())
        {
            outputDirectory = outputDirectory.getParentFile();
        }
        _lastBlockPlotCreationDirectory = outputDirectory;

        //Construct a single step processor.
        final ParameterEstimatorStepProcessor stepProcessor =
                                                            new CreateParameterBlockDiagnosticStepProcessor(getRunInfo(),
                                                                                                            evt.getSource(),
                                                                                                            outputDirectory,
                                                                                                            imageFilePrefix);

        //Initialize the job
        final PerformAllStepsJob performAllJob = new PerformAllStepsJob();
        performAllJob.setLocationMessageStr("Images being generated");
        try
        {
            performAllJob.initialize(getRunInfo(),
                                     Lists.newArrayList(stepProcessor),
                                     ListTools.convertCollection(units,
                                                                 (LocationAndDataTypeIdentifier)units.get(0)),
                                     false);
        }
        catch(final Exception e)
        {
            JOptionPane.showMessageDialog(evt.getSource(),
                                          StringTools.wordWrap("Unable to generate diagnostic block plots for all locations:\n"
                                              + e.getMessage(), 100),
                                          "Error Generating Diagnostic Block Plots!",
                                          JOptionPane.ERROR_MESSAGE);
            return;
        }
        final HJobMonitorDialog jobDialog =
                                          new HJobMonitorDialog(evt.getSource(),
                                                                "Generating Diagnostic Block Plots",
                                                                performAllJob,
                                                                true);
        performAllJob.setParentComponentForMessages(evt.getSource());
        performAllJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc,
                                          final GenericJob theJob,
                                          final boolean displayMessage)
            {
                if(theJob.isCanceled())
                {
                    ((PerformAllStepsJob)theJob).displayExceptions("Image Generation Errors",
                                                                   "Generating diagnostic block plots has been cancelled.");
                }
                else
                {
                    JOptionPane.showMessageDialog(evt.getSource(),
                                                  StringTools.wordWrap(exc.getMessage(),
                                                                       100),
                                                  "Generate Diagnostic Block Plots Error Occurred!",
                                                  JOptionPane.ERROR_MESSAGE);
                }
                jobDialog.setVisible(false);
                setVisible(false);
                setVisible(true);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
                ((PerformAllStepsJob)theJob).displayExceptions("Done generating image files. ");
                setVisible(false);
                setVisible(true);
            }
        });

        //Run all steps.  I need to create another thread to run the job because I do not plan on 
        //reconstructing _performAllJob.  Since you cannot stop and restart a thread in Java, I need
        //to perform the run() method of _performAllJob within another thread that can be disposed of.
        performAllJob.startJob();
        jobDialog.setMinimumSize(new Dimension(700, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }
}
