package ohd.hseb.hefs.mefp.pe.estimation;

import java.io.File;
import java.util.HashMap;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.estimation.GenericEstimationPEStepProcessor;
import ohd.hseb.hefs.pe.estimation.GenericParameterEstimationRunner;
import ohd.hseb.hefs.pe.estimation.options.EstimationControlOptions;
import ohd.hseb.hefs.pe.notice.StepUnitsUpdatedNotice;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.Triad;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.status.StatusLabel;
import ohd.hseb.hefs.utils.status.WorkingStatus;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.util.misc.HStopWatch;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Maps;

public class MEFPEstimationPEStepProcessor extends GenericEstimationPEStepProcessor
{
    private static final Logger LOG = LogManager.getLogger(MEFPEstimationPEStepProcessor.class);

    private static final String STATUS_RUNNING = "Parameters are being estimated for this point.";
    private static final String STATUS_FALSE = "<html>The parameter tgz file has not been created. Please estimate the parameters.</html>";
    private static final String STATUS_NULL = "<html>The parameter tgz file exists, but there is at least one prepared data file newer than the<br>"
        + "parameter file. Please re-estimate the parameters to account for potentially new data.</html>";
    private static final String STATUS_CTL_PARMS = "<html>The control parameters have been changed since the parameter tgz file has last been generated.</html>";
    private static final String STATUS_PARMS_NOT_FOUND = "The used control parameters were not saved for the estimated parameters.";
    private static final String STATUS_TRUE = "The parameter file has been created and is up-to-date. All other required files are present.";
    private static final String STATUS_CANNOT_READ_MAIN_PARMS = "The parameter file exists but it cannot be read.";
    private static final String STATUS_EVENTS = "The canonical events have been changed since the parameter tgz file was last generated.";

    /**
     * The identifier for which parameters are currently being estimated. This is used to indicate that parameters are
     * currently being estimated and for which location they are being estimated. It should not be used for any other
     * purpose. See {@link #getStatus(StepUnit)}.
     */
    private LocationAndDataTypeIdentifier _runningIdent = null;

    /**
     * Used to remember the status acquired from the parameter files. Specifically, the status depends on the list of
     * canonical events contained in the parameter file relative to the user selected events in the interface. This
     * remembers for each identifier the user selected events the last time a check was performed along with the
     * returned state and the last mod time for the parameter file. A new read, then, is only needed if the events are
     * different from those checked previously or the parameter file has been modified. See {@link #getStatus(StepUnit)}
     * .
     */
    private final HashMap<LocationAndDataTypeIdentifier, Triad<CanonicalEventList, Boolean, Long>> _identifierToStatusAndTimeMap = Maps.newHashMap();

    public MEFPEstimationPEStepProcessor(final MEFPParameterEstimatorRunInfo runInformation)
    {
        setRunInfo(runInformation);
    }

    /**
     * @return The top level control options (i.e., the composite one that stores the others) for the given type.
     */
    public MEFPEstimationControlOptions getTopLevelControlOptions(final ParameterId.Type type)
    {
        return getRunInfo().getEstimationControlOptions(type);
    }

    /**
     * This will make use of {@link #_identifierToStatusAndTimeMap} if possible. Specifically, if the GUI selected
     * events match those stored in the map -AND- the time of the file has not changed since the last time the file was
     * read, then the previously remembered state is used. Otherwise, the file will be read from scratch and the state
     * found put back into the map.<br>
     * <br>
     * Since the table tends to get drawn often, there should be many times that this is called in rapid succession for
     * the same identifier. Hence, by remembering the results, we should be able to make the table update more
     * efficient.
     * 
     * @param identifier Identifier to check.
     * @return True if the canonical events in the parm file match the selected events in the interface.
     * @throws Exception If a problem is encountered reading the parm file.
     */
    private boolean doSelectedEventsMatchParameterFileEvents(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        final File parmFile = getRunInfo().getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);
        final CanonicalEventList currentSelectedEvents = getRunInfo().getCanonicalEventsMgr()
                                                                     .buildFullCanonicalEventList(identifier); //Events chosen in interface!
        Triad<CanonicalEventList, Boolean, Long> stateAndTime = _identifierToStatusAndTimeMap.get(identifier);

        //If the events match the last tested, then...
        Boolean state = null;
        if((stateAndTime != null) && (stateAndTime.getFirst().equals(currentSelectedEvents)) && (parmFile.exists()))
        {
            //If the times match, use the remembered state.
            if(parmFile.lastModified() == stateAndTime.getThird())
            {
                state = stateAndTime.getSecond();
            }
        }
        //If the state is still not known, then read the main parameters and compare the events.
        if(state == null)
        {
            final MEFPFullModelParameters parms = getRunInfo().getEstimatedParametersFileHandler()
                                                              .readMainParametersOnly(identifier);
            final CanonicalEventList eventsInParameterFile = parms.getAlgorithmModelParameters()
                                                                  .getFullListOfEventsInOrder();
            state = currentSelectedEvents.equals(eventsInParameterFile);
            stateAndTime = new Triad<CanonicalEventList, Boolean, Long>(currentSelectedEvents,
                                                                        state,
                                                                        parmFile.lastModified());
            _identifierToStatusAndTimeMap.put(identifier, stateAndTime);
        }

        return state;
    }

    /**
     * Performs the step for the provided identifier using the provided options.
     * 
     * @param usedControlOptions The control options specified in the run-time information (user specified).
     */
    public void performStep(final LocationAndDataTypeIdentifier identifier,
                            final MEFPEstimationControlOptions userSpecifiedControlOptions,
                            final boolean backupParametersFirst) throws Exception
    {
        MEFPFullModelParameters parameters = null;

        //For all sources, create a new set of parameters.
        if(getSelectedSource() == null)
        {
            parameters = new MEFPFullModelParameters(identifier,
                                                     userSpecifiedControlOptions,
                                                     getRunInfo().getForecastSources());
        }
        //For a selected source, load them from an existing parameter file.
        else
        {
            if(getRunInfo().getEstimatedParametersFileHandler().haveParametersBeenCreatedAlready(identifier))
            {
                LOG.info("Loading parameters for " + identifier.buildStringToDisplayInTree()
                    + " for estimation of the one forecast source, " + getSelectedSource());
                final HStopWatch loadTimer = new HStopWatch();
                parameters = getRunInfo().getEstimatedParametersFileHandler().readModelParameters(identifier);
                LOG.debug("Loaded parameters successfully in " + loadTimer.getElapsedMillis() + " milliseconds.");
            }
            else
            {
                LOG.warn("No parameters exist for " + identifier.buildStringToDisplayInTree()
                    + ", so one-source parameter estimation cannot be done.  Moving on to next location.");
                return;
            }
        }

        final GenericParameterEstimationRunner runner = new GenericParameterEstimationRunner(identifier,
                                                                                             getRunInfo(),
                                                                                             1 + getRunInfo().getForecastSources()
                                                                                                             .size(),
                                                                                             backupParametersFirst,
                                                                                             getSelectedSource());

        //Set the currently running identifier and update the cell status to reflect parameters are currently being estimated.
        _runningIdent = identifier;
        getRunInfo().post(new StepUnitsUpdatedNotice<ParameterEstimatorStepProcessor, StepUnit>(this, this, identifier));

        try
        {
            runner.execute(parameters);
        }
        finally
        {
            _runningIdent = null;
        }
    }

    @Override
    public MEFPParameterEstimatorRunInfo getRunInfo()
    {
        return (MEFPParameterEstimatorRunInfo)super.getRunInfo();
    }

    @Override
    public void performStep(final StepUnit unit) throws Exception
    {
        final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;
        performStep(identifier,
                    getRunInfo().getEstimationControlOptions(identifier),
                    getBackupParametersBeforeEstimation());
    }

    @Override
    public StatusIndicator getStatus(final StepUnit unit)
    {
        if(!(unit instanceof LocationAndDataTypeIdentifier))
        {
            throw new IllegalArgumentException("Must be passed an Identifier.");
        }
        final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;

        //Added to show a waiting icon when displayed for a row that is currently being estimated.
        if((_runningIdent != null) && (_runningIdent.equals(identifier)))
        {
            return new WorkingStatus(STATUS_RUNNING);
        }

        if(!doFilesExist(identifier))
        {
            return StatusLabel.make(false, STATUS_FALSE);
        }

        if(!doControlOptionsExist(identifier))
        {
            //This should only happen during testing.
            return StatusLabel.make((Boolean)null, STATUS_PARMS_NOT_FOUND);
        }

        if(!areFilesCurrent(identifier))
        {
            return StatusLabel.make((Boolean)null, STATUS_NULL);
        }

        if(!areControlOptionsCurrent(identifier))
        {
            return StatusLabel.make((Boolean)null, STATUS_CTL_PARMS);
        }

        //MEFP specific check: Check that the canonical events are up to date.  This requires reading the
        //main parameters from the parameter files and checking the base and modulation events for equality
        //with those in the current run info.  We may want to turn this into a method in the future, but not now.
        try
        {
            if(!doSelectedEventsMatchParameterFileEvents(identifier))
            {
                return StatusLabel.make((Boolean)null, STATUS_EVENTS);
            }
        }
        catch(final Exception e)
        {
            return StatusLabel.make((Boolean)null, STATUS_CANNOT_READ_MAIN_PARMS);
        }

        return StatusLabel.make(true, STATUS_TRUE);
    }

    @Override
    public boolean doFilesExist(final LocationAndDataTypeIdentifier identifier)
    {
        return super.doFilesExist(identifier);
    }

    @Override
    public ParameterEstimatorStepOptionsPanel constructOptionsPanel()
    {
        return new MEFPEstimationPEStepOptionsPanel(this);
    }

    @Override
    public String getToolTipTextDescribingStep()
    {
        return "Specify control options and execute parameter estimation.";
    }

    @Override
    public String getShortNameForIdentifierTableColumn()
    {
        return "Est";
    }

    @Override
    public String getToolTipTextForIdentifierTableColumnHeader()
    {
        return "<html>Displays if the parameters have been estimated for a location and data type.</html>";
    }

    @Override
    public String getTabNameForStep()
    {
        return "Estimation";
    }

    @Override
    public String getStepNameForRunButton()
    {
        return "Estimate Parameters";
    }

    @Override
    public String getPerformStepPrefix()
    {
        return "Estimating parameters";
    }

    @Override
    public String toString()
    {
        return getStepNameForRunButton();
    }

    @Override
    protected boolean canParametersBeEstimatedForIndividualSources()
    {
        return true;
    }

    @Override
    protected EstimationControlOptions instantiateEstimationControlOptions(final LocationAndDataTypeIdentifier identifier)
    {
        return getRunInfo().constructEstimationControlOptions(identifier.getParameterIdType());
    }
}
