package ohd.hseb.hefs.mefp.pe.estimation;

import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.estimation.options.DistributionEstimationOption;
import ohd.hseb.hefs.pe.estimation.options.FloatEstimationOption;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.hefs.utils.tools.ArrayTools;

public class MEFPPrecipitationSourceControlOptions extends GenericMEFPSourceControlOptions
{
    private DistributionEstimationOption _distObserved;
    private DistributionEstimationOption _distForecast;

    private FloatEstimationOption _alpha;

    /**
     * Standard constructor does not allow for additional parameters.
     * 
     * @param xmlTag The tag to use.
     * @param source The associated forecast source. Its is only used to call its
     *            {@link MEFPForecastSource#getSourceId()} method.
     * @param includeForecastDistribution True if distribution of forecasts is to be included as an options, false if
     *            not. For a climatology source, pass in false.
     */
    public MEFPPrecipitationSourceControlOptions(final String xmlTag, final MEFPForecastSource source)
    {
        super(xmlTag, source, constructPrecipitationBaseParameters(source));
    }

    /**
     * @param xmlTag The tag to use.
     * @param source The associated forecast source. Its is only used to call its
     *            {@link MEFPForecastSource#getSourceId()} method.
     * @param includeForecastDistribution True if distribution of forecasts is to be included as an options, false if
     *            not. For a climatology source, pass in false.
     * @param parameters Additiona parameteters to include, if any.
     */
    public MEFPPrecipitationSourceControlOptions(final String xmlTag,
                                                 final MEFPForecastSource source,
                                                 final ControlOption... parameters)
    {
        super(xmlTag, source, ArrayTools.concat(constructPrecipitationBaseParameters(source), parameters));
    }

    @Override
    protected void mapAttributes()
    {
        _alpha = (FloatEstimationOption)_parameters.get(4);
        
        _distObserved = (DistributionEstimationOption)_parameters.get(5);

        if(!getForecastSource().isClimatologySource())
        {
            _distForecast = (DistributionEstimationOption)_parameters.get(6);
        }
    }

    public DistributionType getObservedDistribution()
    {
        return _distObserved.get();
    }

    public void setObservedDistribution(final DistributionType distribution)
    {
        _distObserved.set(distribution);
    }

    public DistributionType getForecastDistribution()
    {
        if(_distForecast == null)
        {
            return null;
        }
        return _distForecast.get();
    }

    public void setForecastDistribution(final DistributionType distribution)
    {
        _distForecast.set(distribution);
    }

    public float getEPTAlpha()
    {
        return _alpha.get();
    }

    public void setEPTAlpha(final float eptAlpha)
    {
        _alpha.set(eptAlpha);
    }

    /**
     * Constructs an array containing distributionOfObs and distributionOfFcsts control parameters. The
     * distributionOfFcsts control parameter is not included if the source is a climatology source.
     * 
     * @param source The source for which to build control parameters for precip.
     * @return Array of control parameters to include.
     */
    private static ControlOption[] constructPrecipitationBaseParameters(final MEFPForecastSource source)
    {
        ControlOption[] parms = null;
        if(!source.isClimatologySource())
        {
            parms = new ControlOption[3];
        }
        else
        {
            parms = new ControlOption[2];
        }

        parms[0] = new FloatEstimationOption("eptAlpha",
                                  "Fixed Conditional Bias Penalty Weight:",
                                  0.0f,
                                  0.1f,
                                  0.0f,
                                  null);
        parms[1] = new DistributionEstimationOption("distributionOfObs",
                                                    DistributionType.MOMENTS_FITTING_POSTIVE_BOUNDED_DISTRIBUTIONS,
                                                    "Distribution of Positive Observations:");
        if(!source.isClimatologySource())
        {
            parms[2] = new DistributionEstimationOption("distributionOfFcsts",
                                                        DistributionType.MOMENTS_FITTING_POSTIVE_BOUNDED_DISTRIBUTIONS,
                                                        "Distribution of Positive Forecasts:");
        }
        
        return parms;
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        final MEFPPrecipitationSourceControlOptions copy = new MEFPPrecipitationSourceControlOptions(getXMLTagName(),
                                                                                                     getForecastSource());
        copy.copyFrom(this);
        return copy;
    }
}
