package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.event.RowSorterEvent;
import javax.swing.event.RowSorterListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellRenderer;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.gui.components.MoveableJListPanel;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.jtable.GenericTablePanel;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.renderers.ForwardingTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;

/**
 * Panel that displays a row-checkable table of canonical events and checkboxes for base and modulation events tied to
 * the table. It makes use of a {@link GenericTable} that uses a {@link CanonicalEventSelectionTableModel}. An
 * {@link EventBus} is used to communicate events outside this panel. To hear an event, create an instance of event's
 * subscriber; for example, {@link CanonicalEventsSelectedNotice.Subscriber}. Then pass that to the
 * {@link #registerWithBus(Object)} method.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class CanonicalEventSelectionPanel extends JPanel
{
    /**
     * The event table will make use of the private {@link CanonicalEventTypeCellRenderer} via the wrap renderer method
     * in {@link GenericTable}.
     */
    private final GenericTable _eventTable;

    /**
     * Stores the canonical event list in the order of the parameter file.
     */
    private Collection<CanonicalEvent> _defaultCanonicalEventListInOrder;

    /**
     * A listener is added to this table model triggering a {@link CanonicalEventsSelectedNotice} posted to
     * {@link #_eventBus} whenever a selection of events is made.
     */
    private final CanonicalEventSelectionTableModel _eventTableModel;

    private final JCheckBox _baseEventsCheckBox = HSwingFactory.createJCheckBox("All Base Events", null, true);

    private final JCheckBox _modulationEventsCheckBox = HSwingFactory.createJCheckBox("All Modulation Events",
                                                                                      null,
                                                                                      true);
    private final EventBus _eventBus = new EventBus();

    /**
     * @param parameters Full model parameters.
     * @param source The source for which to display selectable events.
     */
    public CanonicalEventSelectionPanel(final MEFPFullModelParameters parameters, final MEFPForecastSource source)
    {
        _defaultCanonicalEventListInOrder = Lists.newArrayList(parameters.getSourceModelParameters(source)
                                                                         .getComputedEvents());

        _eventTableModel = new CanonicalEventSelectionTableModel(parameters.getSourceModelParameters(source),
                                                                 parameters.getAlgorithmModelParameters()
                                                                           .getBaseCanonicalEvents());
        _eventTableModel.addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(final TableModelEvent e)
            {
                _eventBus.post(new CanonicalEventsSelectedNotice(this, getSelectedEvents()));
            }
        });

        //The event table is created using a canonical event type cell renderer.  
        //Further, it must include a listener for when events are sorted, which means the 
        //initial row sorter must be listened to and new row sorters must be listened to when
        //changed.
        _eventTable = new GenericTable<CanonicalEvent>(_eventTableModel)
        {
            @Override
            protected TableCellRenderer wrapRenderer(final TableCellRenderer renderer)
            {
                return new CanonicalEventTypeCellRenderer(super.wrapRenderer(renderer));
            }
        };
        _eventTable.addPropertyChangeListener(new PropertyChangeListener()
        {
            @Override
            public void propertyChange(final PropertyChangeEvent evt)
            {
                //Model changes, so the row sorter changed... must add a new listener.
                if(evt.getPropertyName().equals("model"))
                {
                    _eventTable.getRowSorter().addRowSorterListener(new RowSorterListener()
                    {
                        @Override
                        public void sorterChanged(final RowSorterEvent e)
                        {
                            _eventBus.post(new CanonicalEventsSelectedNotice(this, getSelectedEvents()));
                        }
                    });
                }
            }
        });
        //Initial row sorter and listener.
        _eventTable.setAutoCreateRowSorter(true);
        _eventTable.getRowSorter().addRowSorterListener(new RowSorterListener()
        {
            @Override
            public void sorterChanged(final RowSorterEvent e)
            {
                _eventBus.post(new CanonicalEventsSelectedNotice(this, getSelectedEvents()));
            }
        });
        _eventTable.setRowSelectionAllowed(true);
        _eventTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);

        final JButton checkAllButton = TableTools.createCheckAllButton(_eventTableModel, "Check all rows for viewing");
        final JButton uncheckAllButton = TableTools.createUncheckAllButton(_eventTableModel,
                                                                           "Uncheck all rows for viewing");
        final JButton checkSelectedButton = TableTools.createCheckSelectedButton(_eventTableModel,
                                                                                 "Check selected rows for viewing");
        final JButton uncheckSelectedButton = TableTools.createUncheckSelectedButton(_eventTableModel,
                                                                                     "Uncheck selected rows for viewing");
        final JButton reorderRowsButton = new SelfListeningButton("reorderAllRows20x20")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                reorderRows();
            }
        };
        reorderRowsButton.setToolTipText("Reorder row indices (shown in # column; allows for custom ordering of events)");

        final GenericTablePanel tablePanel = new GenericTablePanel(_eventTable,
                                                                   Lists.newArrayList(GenericTablePanel.SELECT_ALL_BUTTON,
                                                                                      GenericTablePanel.UNSELECT_BUTTON,
                                                                                      GenericTablePanel.SEPARATOR,
                                                                                      checkAllButton,
                                                                                      uncheckAllButton,
                                                                                      checkSelectedButton,
                                                                                      uncheckSelectedButton,
                                                                                      GenericTablePanel.SEPARATOR,
                                                                                      reorderRowsButton),
                                                                   "Canonical Events (Sortable)");

        final JPanel mainPanel = new JPanel(new GridBagLayout());
        final GridBagConstraints cons = new GridBagConstraints();
        cons.gridy = 0;
        cons.weighty = 0;
        cons.weightx = 1; //XXX Must be positive to ensure that it fills in the mainPanel width with the table.
        cons.anchor = GridBagConstraints.NORTHWEST;
        cons.fill = GridBagConstraints.BOTH;
        mainPanel.add(_baseEventsCheckBox, cons);
        cons.gridy = 1;
        mainPanel.add(_modulationEventsCheckBox, cons);
        cons.gridy = 2;
        cons.weighty = 1;
        mainPanel.add(tablePanel, cons);

        if(!_eventTableModel.areAnyModulationEventsDefined())
        {
            _modulationEventsCheckBox.setEnabled(false);
            _modulationEventsCheckBox.setSelected(false);
        }

        this.setLayout(new BorderLayout());
        this.add(mainPanel, BorderLayout.CENTER);
        this.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1),
                                                        "Select Events to Display",
                                                        null));

        addCheckBoxListeners();
    }

    /**
     * @return {@link List} of {@link Dyad} instances, each containing a {@link String} to be displayed in the reorder
     *         dialog and the corresponding {@link CanonicalEvent}.
     */
    private List<Dyad<String, CanonicalEvent>> createReorderManagedList(final Collection<CanonicalEvent> base)
    {
        final List<Dyad<String, CanonicalEvent>> managedList = Lists.newArrayList();
        for(final CanonicalEvent evt: base)
        {
            managedList.add(new Dyad<String, CanonicalEvent>(CanonicalEvent.createXMLAttributeString(evt), evt)
            {
                @Override
                public String toString()
                {
                    return getFirst() + " [" + _eventTableModel.getEventTypeCellValue(getSecond()) + "]";
                }
            });
        }
        return managedList;
    }

    /**
     * Opens a dialog to allow for reordering the base order of the canonical events.
     * 
     * @param sourceParameters Parameters for the active source that provides a starting point list of event for use
     *            with the default button.
     */
    private void reorderRows()
    {
        //Create the list to manage so that the simpled (#,#) string is displayed.  CanonicalEvent toString is too noisy to use here.
        final List<Dyad<String, CanonicalEvent>> managedList = createReorderManagedList(_eventTableModel.getEventsInTableOrder());

        //The movable list panel.
        final MoveableJListPanel listPanel = new MoveableJListPanel(managedList);
        listPanel.setPreferredSize(new Dimension(200, 250));

        //Default order button.
        final JButton defaultOrderButton = new SelfListeningButton("Default Order")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                managedList.clear();
                managedList.addAll(createReorderManagedList(_defaultCanonicalEventListInOrder));
                listPanel.updateJListToReflectManagedList();
            }
        };
        defaultOrderButton.setToolTipText("Sets order to match stored parameters (sorted by end period)");
        final JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        buttonPanel.add(defaultOrderButton);

        //The display panel.
        final JPanel panel = new JPanel(new BorderLayout());
        panel.add(new JLabel("<html>Reorder the '#' column indices of the rows.<br>Default order is by end period.<br>To use this order, sort table by '#' column.</html>"),
                  BorderLayout.NORTH);
        panel.add(listPanel, BorderLayout.CENTER);
        panel.add(buttonPanel, BorderLayout.SOUTH);

        final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(CanonicalEventSelectionPanel.this),
                                                         panel,
                                                         "Reorder Events (Modify # Column)",
                                                         JOptionPane.OK_CANCEL_OPTION);
        if(option == JOptionPane.OK_OPTION)
        {
            //Recover a list of canonical events in order to pass it to the model.
            final List<CanonicalEvent> newList = Lists.newArrayList();
            for(final Dyad<String, CanonicalEvent> dyad: managedList)
            {
                newList.add(dyad.getSecond());
            }
            _eventTable.getRowSorter().setSortKeys(null);
            _eventTableModel.setComputedEventList(newList);
            repaint();
        }
    }

    /**
     * @param parameters The {@link MEFPSourceModelParameters} to be used as a source for the canonical event selectable
     *            list.
     */
    public void setSourceModelParameters(final MEFPSourceModelParameters parameters)
    {
        _eventTableModel.setSourceModelParameters(parameters);
        _defaultCanonicalEventListInOrder = Lists.newArrayList(parameters.getComputedEvents());
    }

    /**
     * Initializes all listeners related to the check boxes, including keeping the check boxes in synch with the table
     * selections via an {@link TableModelListener} added to {@link #_eventTableModel}.
     */
    private void addCheckBoxListeners()
    {
        _baseEventsCheckBox.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(_baseEventsCheckBox.isSelected())
                {
                    _eventTableModel.selectBaseEvents();
                }
                else
                {
                    _eventTableModel.deselectBaseEvents();
                }
            }
        });
        _modulationEventsCheckBox.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(_modulationEventsCheckBox.isSelected())
                {
                    _eventTableModel.selectModulationEvents();
                }
                else
                {
                    _eventTableModel.deselectModulationEvents();
                }
            }
        });

        _eventTableModel.addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(final TableModelEvent e)
            {
                _baseEventsCheckBox.setSelected(_eventTableModel.areAllBaseEventsSelected());
                _modulationEventsCheckBox.setSelected(_eventTableModel.areAllModulationEventsSelected());
            }
        });
    }

    /**
     * Register the supplied Object to the {@link EventBus} that underlies this panel.
     */
    public void registerWithBus(final CanonicalEventsSelectedNotice.Subscriber registree)
    {
        _eventBus.register(registree);
    }

    /**
     * Unregister the provided subscriber.
     */
    public void unregisterWithBus(final CanonicalEventsSelectedNotice.Subscriber registree)
    {
        _eventBus.unregister(registree);
    }

    /**
     * Calls {@link CanonicalEventSelectionTableModel#getSelectedEvents()}.
     */
    public Collection<CanonicalEvent> getSelectedEvents()
    {
        return _eventTableModel.getSelectedEvents();
    }

    /**
     * Renderer will color the background of modulation events a darker shade of gray than base events.
     * 
     * @author hankherr
     */
    private class CanonicalEventTypeCellRenderer extends ForwardingTableCellRenderer
    {

        public CanonicalEventTypeCellRenderer(final TableCellRenderer delegate)
        {
            super(delegate);
        }

        @Override
        public Component getTableCellRendererComponent(final JTable table,
                                                       final Object value,
                                                       final boolean isSelected,
                                                       final boolean hasFocus,
                                                       final int row,
                                                       final int column)
        {
            final Component component = super.getTableCellRendererComponent(table,
                                                                            value,
                                                                            isSelected,
                                                                            hasFocus,
                                                                            row,
                                                                            column);

            int modelRow;
            {
                final RowSorter sorter = table.getRowSorter();
                if(sorter == null)
                {
                    modelRow = row;
                }
                else
                {
                    modelRow = sorter.convertRowIndexToModel(row);
                }
            }
            if(!isSelected)
            {
                if(_eventTableModel.isBaseEvent(modelRow))
                {
                    component.setBackground(new Color(224, 224, 224));
                }
                else
                {
                    component.setBackground(Color.LIGHT_GRAY);
                }
            }

            return component;
        }

    }
}
