package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.util.Collection;
import java.util.List;

import javax.swing.JTable;

import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.models.AbstractToolTipTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MaxWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowCheckableTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.SelectionMaintainingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.TableAwareTableModel;
import ohd.hseb.hefs.utils.tools.ListTools;

import com.google.common.collect.Lists;

/**
 * Table model used in the table displayed in the {@link CanonicalEventSelectionPanel}.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class CanonicalEventSelectionTableModel extends AbstractToolTipTableModel implements
SelectionMaintainingTableModel<CanonicalEvent>, RowCheckableTableModel, MaxWidthsTableModel
{

    private final boolean _isPrecipitationDataType;
    private final List<CanonicalEvent> _computedEvents;
    private final Collection<CanonicalEvent> _baseEvents;
    private final List<Boolean> _selectedStates;
    private JTable _table = null;

    /**
     * @param parameters Source parameters that supplies the computed events, via
     *            {@link MEFPSourceModelParameters#getComputedEvents()}.
     * @param baseEvents A list of the base events. It is assumed any event not in the list is a modulation event.
     */
    public CanonicalEventSelectionTableModel(final MEFPSourceModelParameters parameters,
                                             final Collection<CanonicalEvent> baseEvents)
    {
        _isPrecipitationDataType = parameters.getIdentifier().isPrecipitationDataType();
        _computedEvents = Lists.newArrayList(parameters.getComputedEvents());
        _baseEvents = baseEvents;

        _selectedStates = Lists.newArrayListWithCapacity(_computedEvents.size());
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            _selectedStates.add(true);
        }
    }

    /**
     * Call too change the underlying events and maintain an appropriate list of selected events.
     * 
     * @param parameters
     */
    public void setSourceModelParameters(final MEFPSourceModelParameters parameters)
    {
        final List<Boolean> newSelectedStates = Lists.newArrayList();
        for(int i = 0; i < parameters.getComputedEvents().size(); i++)
        {
            final CanonicalEvent event = parameters.getComputedEvents().get(i);
            if(_computedEvents.contains(event))
            {
                newSelectedStates.add(isSelected(event));
            }
            else
            {
                newSelectedStates.add(false);
            }
        }
        _computedEvents.clear();
        _computedEvents.addAll(parameters.getComputedEvents());
        _selectedStates.clear();
        _selectedStates.addAll(newSelectedStates);
        fireTableDataChanged();
    }

    /**
     * @param events Events for which to set the selected flag.
     * @param state The new flag value.
     */
    public void setSelected(final Collection<CanonicalEvent> events, final boolean state)
    {
        for(final CanonicalEvent evt: events)
        {
            _selectedStates.set(getRowIndex(evt), state);
        }
        this.fireTableDataChanged();
    }

    public String getEventTypeCellValue(final CanonicalEvent evt)
    {
        if(_baseEvents.contains(evt))
        {
            return "BASE";
        }
        return "MOD";
    }

    public boolean isBaseEvent(final int rowIndex)
    {
        return _baseEvents.contains(getRow(rowIndex));
    }

    public boolean isSelected(final CanonicalEvent evt)
    {
        return isSelected(getRowIndex(evt));
    }

    public boolean isSelected(final int index)
    {
        return _selectedStates.get(index);
    }

    /**
     * @return {@link Collection} of all events current selected according to the internal kept list of selected states,
     *         {@link #_selectedStates}.
     */
    public Collection<CanonicalEvent> getSelectedEvents()
    {
        final List<CanonicalEvent> evts = Lists.newArrayList();
        for(int i = 0; i < getRowCount(); i++)
        {
            final int modelIndex = getTable().convertRowIndexToModel(i);
            if(_selectedStates.get(modelIndex))
            {
                evts.add(_computedEvents.get(modelIndex));
            }
        }
        return evts;
    }

    /**
     * @return The {@link JTable} associated with this model, as part of the {@link TableAwareTableModel} framework.
     */
    public JTable getTable()
    {
        return _table;
    }

    /**
     * Sets the selection state for all base events to true; does not modify the selection state of modulation events.
     */
    public void selectBaseEvents()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(isBaseEvent(i))
            {
                _selectedStates.set(i, true);
            }
        }
        this.fireTableDataChanged();
    }

    /**
     * Sets the selection state for all modulation events to true; does not modify the selection state of base events.
     */
    public void selectModulationEvents()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(!isBaseEvent(i))
            {
                _selectedStates.set(i, true);
            }
        }
        this.fireTableDataChanged();
    }

    /**
     * Sets the selection state for all base events to false; does not modify the selection state of modulation events.
     */
    public void deselectBaseEvents()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(isBaseEvent(i))
            {
                _selectedStates.set(i, false);
            }
        }
        this.fireTableDataChanged();
    }

    /**
     * Sets the selection state for all modulation events to false; does not modify the selection state of base events.
     */
    public void deselectModulationEvents()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(!isBaseEvent(i))
            {
                _selectedStates.set(i, false);
            }
        }
        this.fireTableDataChanged();
    }

    /**
     * @return True if all base eventsa are selected, false otherwise.
     */
    public boolean areAllBaseEventsSelected()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(isBaseEvent(i))
            {
                if(!_selectedStates.get(i))
                {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * @return True if any modulation events are included in the current list of events.
     */
    public boolean areAnyModulationEventsDefined()
    {
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(!isBaseEvent(i))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @return True if all modulation eventsa are selected, false otherwise.
     */
    public boolean areAllModulationEventsSelected()
    {
        if(!areAnyModulationEventsDefined())
        {
            return false;
        }
        for(int i = 0; i < _computedEvents.size(); i++)
        {
            if(!isBaseEvent(i))
            {
                if(!_selectedStates.get(i))
                {
                    return false;
                }
            }
        }
        return true;
    }

    public List<CanonicalEvent> getComputedEvents()
    {
        return _computedEvents;
    }

    /**
     * @return List of the {@link CanonicalEvent}s sorted based on the current view in the table.
     */
    public List<CanonicalEvent> getEventsInTableOrder()
    {
        final List<CanonicalEvent> events = Lists.newArrayList();
        for(int i = 0; i < getRowCount(); i++)
        {
            events.add(getRow(getTable().convertRowIndexToModel(i)));
        }
        return events;
    }

    public void setComputedEventList(final Collection<CanonicalEvent> events)
    {
        final List<Boolean> newSelectedStates = Lists.newArrayList();
        for(final CanonicalEvent evt: events)
        {
            newSelectedStates.add(isSelected(evt));
        }

        _computedEvents.clear();
        _computedEvents.addAll(events);
        _selectedStates.clear();
        _selectedStates.addAll(newSelectedStates);
        fireTableDataChanged();
    }

    @Override
    public void checkSelected()
    {
        final int[] selectedRows = getTable().getSelectedRows();
        for(final int i: selectedRows)
        {
            this._selectedStates.set(getTable().convertRowIndexToModel(i), true);
        }
        this.fireTableDataChanged();
    }

    @Override
    public void uncheckSelected()
    {
        final int[] selectedRows = getTable().getSelectedRows();
        for(final int i: selectedRows)
        {
            this._selectedStates.set(getTable().convertRowIndexToModel(i), false);
        }
        this.fireTableDataChanged();
    }

    @Override
    public void checkAll()
    {
        ListTools.setAll(_selectedStates, true);
        this.fireTableDataChanged();
    }

    @Override
    public void uncheckAll()
    {
        ListTools.setAll(_selectedStates, false);
        this.fireTableDataChanged();
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == 0)
        {
            return "";
        }
        else if(column == 1)
        {
            return "#";
        }
        else if(column == 2)
        {
            return "Start";
        }
        else if(column == 3)
        {
            return "End";
        }
        else if(column == 4)
        {
            return "Dur";
        }
        else if(column == 5)
        {
            return "Type";
        }
        return null;
    }

    @Override
    public String getColumnHeaderToolTip(final int modelColIndex)
    {
        if(modelColIndex == 0)
        {
            return "Selected state; click to change the selected state of an event";
        }
        else if(modelColIndex == 1)
        {
            return "Click to sort by event indices";
        }
        else if(modelColIndex == 2)
        {
            return "Click to sort by the canonical event start period";
        }
        else if(modelColIndex == 3)
        {
            return "Click to sort by the canonical event end period";
        }
        else if(modelColIndex == 4)
        {
            return "Click to sort by the canonical period duration";
        }
        else if(modelColIndex == 5)
        {
            return "Click to sort by base or modulation event";
        }
        return null;
    }

    @Override
    public boolean isCellEditable(final int rowIndex, final int columnIndex)
    {
        return columnIndex == 0;
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return Boolean.class;
        }
        if(columnIndex == 5)
        {
            return String.class;
        }
        return Integer.class;
    }

    @Override
    public String getCellToolTip(final int modelRowIndex, final int modelColIndex)
    {
        //First line will match the domain axis tick label to make it easier to identify to which domain axis block
        //the event corresponds.
        final String topString = "<html>"
            + ParameterBlockDiagnosticPanel.determineDomaainAxisTickLabel(getRow(modelRowIndex),
                                                                          _isPrecipitationDataType) + "<br>";
        if(modelColIndex == 0)
        {
            if(Boolean.TRUE.equals(getValueAt(modelRowIndex, modelColIndex)))
            {
                return topString + "The canonical event is selected for display; click to change selection" + "</html>";
            }
            else
            {
                return topString + "The canonical event is not selected for display; click to change selection"
                    + "</html>";
            }
        }
        else if(modelColIndex == 1)
        {
            return topString + "Event index: " + (modelRowIndex + 1) + "</html>";
        }
        else if(modelColIndex == 2)
        {
            return topString + "Start period for event: " + getRow(modelRowIndex).getStartLeadPeriod() + "</html>";
        }
        else if(modelColIndex == 3)
        {
            return topString + "End period for event: " + getRow(modelRowIndex).getEndLeadPeriod() + "</html>";
        }
        else if(modelColIndex == 4)
        {
            return topString + "Duration for event: " + getRow(modelRowIndex).getDuration() + "</html>";
        }
        else if(modelColIndex == 5)
        {
            if(Boolean.TRUE.equals(getValueAt(modelRowIndex, modelColIndex)))
            {
                return topString + "The canonical event is a base event" + "</html>";
            }
            else
            {
                return topString + "The canonical event is a modulation event" + "</html>";
            }
        }
        return null;
    }

    @Override
    public int getRowCount()
    {
        return _computedEvents.size();
    }

    @Override
    public int getColumnCount()
    {
        return 6;
    }

    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return isSelected(rowIndex);
        }
        else if(columnIndex == 1)
        {
            return rowIndex + 1;
        }
        else if(columnIndex == 2)
        {
            return getRow(rowIndex).getStartLeadPeriod();
        }
        else if(columnIndex == 3)
        {
            return getRow(rowIndex).getEndLeadPeriod();
        }
        else if(columnIndex == 4)
        {
            return getRow(rowIndex).getDuration();
        }
        else if(columnIndex == 5)
        {
            return getEventTypeCellValue(getRow(rowIndex));
        }
        return null;
    }

    @Override
    public void setValueAt(final Object aValue, final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            final Boolean b = (Boolean)aValue;
            _selectedStates.set(rowIndex, b);
            fireTableDataChanged();
        }
    }

    @Override
    public CanonicalEvent getRow(final int index)
    {
        return _computedEvents.get(index);
    }

    @Override
    public int getRowIndex(final CanonicalEvent row)
    {
        return _computedEvents.indexOf(row);
    }

    @Override
    public void addTable(final JTable table)
    {
        _table = table;
    }

    @Override
    public void removeTable(final JTable table)
    {
        _table = null;
    }

    @Override
    public Integer getMaxWidth(final int column)
    {
        if(column == 0)
        {
            return TableTools.getStatusColumnFixedWidth() + 4;
        }
        if(column == 1)
        {
            return 40;
        }
        if(column == 5)
        {
            return 60;
        }
        return null;
    }

}
