package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSeparator;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.types.AvgParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.CAvgParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.EPTRhoParameterType;
import ohd.hseb.hefs.mefp.models.parameters.types.RhoParameterType;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.ParameterId;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.EventBus;

/**
 * Displays a combo box of selectable {@link ModelParameterType} names. It also allows for selecting custom difference
 * {@link ModelParameterType} instances. An {@link EventBus} is created for notices when a parameter selection is made:
 * {@link DisplayedParameterSelectedNotice}. To hear those events, call {@link #registerWithBus(Object)} and make sure
 * the passed in object implements {@link DisplayedParameterSelectedNotice.Subscriber}
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class DisplayedParameterSelectionPanel extends JPanel
{
    private final JComboBox _availableParametersChoiceBox;
    private final HashMap<String, ModelParameterType[]> _nameToTypesMap = Maps.newHashMap();
    private final HashMap<ModelParameterType, String> _singletonTypeToNameMap = Maps.newHashMap();
    private final EventBus _eventBus = new EventBus();
    private final ParameterId _parameter;
    private MEFPSourceModelParameters _activeSourceModelParameters = null;
    private final MEFPFullModelParameters _fullParameters;

    ActionListener _selectParameterActionListener = new ActionListener()
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            selectParameter();
        }
    };

    /**
     * Records the secondary source used for some difference parameters.
     */
    private MEFPForecastSource _secondarySource = null;

    /**
     * @param sourceParameters The source parameters used to extract a list of parameter names.
     */
    public DisplayedParameterSelectionPanel(final MEFPFullModelParameters fullParameters,
                                            final MEFPSourceModelParameters sourceParameters)
    {
        _fullParameters = fullParameters;
        _parameter = ParameterId.valueOf(sourceParameters.getIdentifier().getParameterId());
        _availableParametersChoiceBox = new JComboBox()
        {
            //Override so that the empty item, displayed as a separator, cannot be selected.
            @Override
            public void setSelectedItem(final Object anObject)
            {
                if(!((String)anObject).isEmpty())
                {
                    super.setSelectedItem(anObject);
                }
            }
        };
        setSourceModelParameters(sourceParameters);

        //Setup the listener after initializing combo box.
        _availableParametersChoiceBox.addActionListener(_selectParameterActionListener);
        _availableParametersChoiceBox.setRenderer(new DefaultListCellRenderer()
        {
            @Override
            public Component getListCellRendererComponent(final JList list,
                                                          final Object value,
                                                          final int index,
                                                          final boolean isSelected,
                                                          final boolean cellHasFocus)
            {
                if(((String)value).isEmpty())
                {
                    return new JSeparator(JSeparator.HORIZONTAL);
                }
                return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            }
        });
//        _availableParametersChoiceBox.addItemListener(new ItemListener()
//        {
//            @Override
//            public void itemStateChanged(final ItemEvent e)
//            {
//                if(e.getStateChange() == ItemEvent.SELECTED)
//                {
//                    if(!((String)_availableParametersChoiceBox.getSelectedItem()).isEmpty())
//                    {
//                        //Get the selected parameters.
//                        final ModelParameterType[] selectedParms = getSelectedParameters();
//
//                        //If the two parameters are identical, then the parameter is a difference across sources (see addDiff method below).
//                        //Ask the user what the secondary source is.
//                        if((selectedParms.length == 2) && (selectedParms[0].equals(selectedParms[1])))
//                        {
//                            if(!askForSecondarySource(selectedParms[0]))
//                            {
//                                return; //User clicked cancel or an exception occurred, so return without posting an event.
//                            }
//                        }
//                        else
//                        {
//                            _secondarySource = null;
//                        }
//
//                        _eventBus.post(new DisplayedParameterSelectedNotice(this,
//                                                                            getSelectedParameters(),
//                                                                            _secondarySource));
//                    }
//                }
//            }
//        });

        this.setLayout(new GridLayout(1, 0));
        add(_availableParametersChoiceBox);

        setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1),
                                                   "Select Parameter to Display",
                                                   null));
    }

    /**
     * If {@link #_availableParametersChoiceBox} selection is not empty, this posts a message indicating a parameter as
     * been selected. If necessary, a secondary source will be asked for.
     * 
     * @return True if a message was posted indicating a parameter is selected for display. False if nothing was done
     *         either due to the selection {@link String} in the combo box being empty (i.e., a separator combo box
     *         item) or the secondary source request resulting in a cancel.
     */
    private boolean selectParameter()
    {
        if(!((String)_availableParametersChoiceBox.getSelectedItem()).isEmpty())
        {
            //Get the selected parameters.
            final ModelParameterType[] selectedParms = getSelectedParameters();

            //If the two parameters are identical, then the parameter is a difference across sources (see addDiff method below).
            //Ask the user what the secondary source is.
            if((selectedParms.length == 2) && (selectedParms[0].equals(selectedParms[1])))
            {
                if(!askForSecondarySource(selectedParms[0]))
                {
                    return false; //User clicked cancel or an exception occurred, so return without posting an event.
                }
            }
            else
            {
                _secondarySource = null;
            }

            //This if check should not be needed, but we did see one instance where the secondary and primary sources
            //were identical.  This avoids that situation.  If the two sources are somehow equal, this returns as if cancel
            //were clicked and does nothing.
            _eventBus.post(new DisplayedParameterSelectedNotice(this, getSelectedParameters(), _secondarySource));
            return true;
        }
        return false;
    }

    /**
     * Opens a dialog located relative to the global parent of this asking for the user to choose a secondary source for
     * parameter computation.
     * 
     * @param requestedParameterType The parameter type requested, which is only used as part of the dialog's message.
     * @return False if the user cancelled the request or true if the user selected a secondary source and clicked OK.
     */
    private boolean askForSecondarySource(final ModelParameterType requestedParameterType)
    {
        //Base list of sources...
        final List<MEFPForecastSource> sources = MEFPTools.generateListOfSourcesWithParameterOfType(_fullParameters,
                                                                                                    requestedParameterType);

        //Remove the active source...
        sources.remove(_activeSourceModelParameters.getForecastSource());
        if(sources.isEmpty())
        {
            //This should not happen, but just in case.
            JOptionPane.showMessageDialog(DisplayedParameterSelectionPanel.this,
                                          "No other sources included a computed parameter of this type; this stat cannot be used.",
                                          "Only One Source Available!",
                                          JOptionPane.WARNING_MESSAGE);
            return false;
        }

        //Create the combo box.
        final JComboBox sourcesChoiceBox = new JComboBox(sources.toArray());

        //Select the previous secondary source if there was any.
        if(_secondarySource != null)
        {
            sourcesChoiceBox.setSelectedItem(_secondarySource);
        }

        final JPanel panel = new JPanel(new BorderLayout());
        panel.add(new JLabel("<html>The active forecast source is " + _activeSourceModelParameters.getForecastSource()
                      + ".<br>Please select the other forecast source for computation of differences:</html>"),
                  BorderLayout.NORTH);
        panel.add(sourcesChoiceBox, BorderLayout.CENTER);
        panel.add(new JLabel("<html>Difference computation will be the parameter value for <br>"
            + "the active source minus the difference source.</html>"), BorderLayout.SOUTH);

        //Allow for cancellation.
        final int confirm = JOptionPane.showConfirmDialog(DisplayedParameterSelectionPanel.this,
                                                          panel,
                                                          "Select Other Source for Difference Computation",
                                                          JOptionPane.OK_CANCEL_OPTION);
        if(confirm == JOptionPane.CANCEL_OPTION)
        {
            return false;
        }

        //Set the secondary source to that selected.
        _secondarySource = (MEFPForecastSource)sourcesChoiceBox.getSelectedItem();
        return true;
    }

    /**
     * For the given parameters, it determines the model parameter types based on the stored parameters and calls
     * {@link #addDifferenceParameterTypes(ParameterId)} to add additional display types.
     * 
     * @param sourceParameters
     */
    public void setSourceModelParameters(final MEFPSourceModelParameters sourceParameters)
    {
        _activeSourceModelParameters = sourceParameters;
        final String selectedName = (String)_availableParametersChoiceBox.getSelectedItem();

        //Put together maps
        _nameToTypesMap.clear();
        _singletonTypeToNameMap.clear();
        for(final ModelParameterType type: sourceParameters.getModelParameterTypesWithStoredValues())
        {
            String name = type.getParameterId() + ": " + type.getName();
            if(type.getUnit() != null)
            {
                name += " [" + type.getUnit() + "]";
            }
            _nameToTypesMap.put(name, new ModelParameterType[]{type});
            _singletonTypeToNameMap.put(type, name);
        }

        //Create names list
        final List<String> names = Lists.newArrayList();
        names.addAll(_nameToTypesMap.keySet());

        //Add additional difference parameter types that are not singletons.  Add them to the names list.
        Collections.sort(names);
        final Collection<String> diffParmTypes = addDifferenceParameterTypes(ParameterId.valueOf(sourceParameters.getIdentifier()
                                                                                                                 .getParameterId()));
        if(!diffParmTypes.isEmpty())
        {
            names.add(0, "");
            names.addAll(0, diffParmTypes);
        }

        //Set the model for the choice box and set the selected item, but only if we have a parameter selected.
        _availableParametersChoiceBox.setModel(new DefaultComboBoxModel(names.toArray()));
        if((selectedName != null) && (_nameToTypesMap.keySet().contains(selectedName)))
        {
            if(_nameToTypesMap.keySet().contains(selectedName))
            {
                //I need to be able to react to the secondary parm selection if necessary.
                _availableParametersChoiceBox.removeActionListener(_selectParameterActionListener);
                _availableParametersChoiceBox.setSelectedItem(selectedName);
                _availableParametersChoiceBox.addActionListener(_selectParameterActionListener);
                if(selectParameter())
                {
                    return;
                }
                else
                {
                    //User clicked cancel when the secondary source request opened.  The other possibility is that the 
                    //parameter chosen is the separator combo box item, but I don't think it will get past the _nameToTypesMap
                    //if check above.  
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                                  "Selected parameter,  '"
                                                      + selectedName
                                                      + "',\nrequires a secondary source, but the user clicked Cancel.\n"
                                                      + "The parameter cannot be computed or displayed.  Selecting default parameter.",
                                                  "Selected Parameter Changed",
                                                  JOptionPane.INFORMATION_MESSAGE);
                }
            }
            else
            {
                //Initial parameter is selected if the selected name is not within the name to types map (ex: if the separator
                //is selected).  A message is displayed and a default is chosen.
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                              "Selected parameter,  '"
                                                  + selectedName
                                                  + "',\nis not available for this source.  Selecting default parameter.",
                                              "Selected Parameter Changed",
                                              JOptionPane.INFORMATION_MESSAGE);
            }
            selectInitialParameter();
        }
    }

    /**
     * Adds parameter types to the {@link #_nameToTypesMap} computing differences between forecast and mean values. The
     * added types vary by parameter type.
     * 
     * @param parameter Parameter for which to add parameter types.
     * @return Names that can be added to the combo box.
     */
    private Collection<String> addDifferenceParameterTypes(final ParameterId parameter)
    {
        final List<String> names = Lists.newArrayList();

        //Precip...
        if(parameter.isPrecipitation())
        {
            //Only add these diffs if the source is NOT climatology.
            if(!_activeSourceModelParameters.getForecastSource().isClimatologySource())
            {
                names.add("Diff of Avg Forecast and Observed");
                _nameToTypesMap.put(names.get(0), new ModelParameterType[]{
                    new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true),
                    new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false)});
                names.add("Diff of Cond Avg Forecast and Observed");
                _nameToTypesMap.put(names.get(1), new ModelParameterType[]{
                    new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, true),
                    new CAvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID, false)});
            }

            //Correlation differences are also not added for climatology, but that will be handled via the checkList.

            //This will only be false if the parameters were estimated for only one source or the Rho parm is not defined for the source (e.g. climatology).
            List<MEFPForecastSource> checkList = MEFPTools.generateListOfSourcesWithParameterOfType(_fullParameters,
                                                                                                    new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
            if((checkList.size() > 1) && (checkList.contains(_activeSourceModelParameters.getForecastSource())))
            {
                names.add("Diff of IPT Correlation Between Sources");
                _nameToTypesMap.put(names.get(2), new ModelParameterType[]{
                    new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID),
                    new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)});
            }

            //This will be false if EPT parameters were estimated for only one source or the EPT parm is not defined for the source (e.g., climatology, CFSv2)
            checkList = MEFPTools.generateListOfSourcesWithParameterOfType(_fullParameters,
                                                                           new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
            if((checkList.size() > 1) && (checkList.contains(_activeSourceModelParameters.getForecastSource())))
            {
                names.add("Diff of EPT Correlation Between Sources");
                _nameToTypesMap.put(names.get(3), new ModelParameterType[]{
                    new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID),
                    new EPTRhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID)});
            }

        }

        //Temperature...
        else
        {
            //Only add these diffs if the source is NOT climatology.
            if(!_activeSourceModelParameters.getForecastSource().isClimatologySource())
            {
                names.add("TMIN Diff of Avg Forecast and Observed");
                _nameToTypesMap.put(names.get(0), new ModelParameterType[]{
                    new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, true),
                    new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false)});
                names.add("TMAX Diff of Avg Forecast and Observed");
                _nameToTypesMap.put(names.get(1), new ModelParameterType[]{
                    new AvgParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, true),
                    new AvgParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID, false)});
                names.add("Diff of TMIN Correlation Between Sources");

                //This will only be false if the parameters were estimated for only one source.  The rho parameter
                //always exists for all sources.
                if(MEFPTools.generateListOfSourcesWithParameters(_fullParameters).size() > 1)
                {
                    _nameToTypesMap.put(names.get(2), new ModelParameterType[]{
                        new RhoParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID),
                        new RhoParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID)});
                    names.add("Diff of TMAX Correlation Between Sources");
                    _nameToTypesMap.put(names.get(3), new ModelParameterType[]{
                        new RhoParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID),
                        new RhoParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID)});
                }
            }
        }

        return names;
    }

    /**
     * @return Array of either 1 or 2 (for custom difference types) of {@link ModelParameterType} instances
     *         corresponding to the selected name. If the array is of size two and the two are the same parameter type,
     *         it means that a difference is to be performed ACROSS forecast sources.
     */
    public ModelParameterType[] getSelectedParameters()
    {
        return _nameToTypesMap.get(_availableParametersChoiceBox.getSelectedItem());
    }

    public MEFPForecastSource getSecondarySource()
    {
        return _secondarySource;
    }

    /**
     * Register the supplied Object to the {@link EventBus} that underlies this panel.
     */
    public void registerWithBus(final DisplayedParameterSelectedNotice.Subscriber registree)
    {
        _eventBus.register(registree);
    }

    /**
     * Unregister the supplied Object to the {@link EventBus} that underlies this panel.
     */
    public void unregisterWithBus(final DisplayedParameterSelectedNotice.Subscriber registree)
    {
        _eventBus.unregister(registree);
    }

    /**
     * For precip, it selects the IPT precip rho parameter, since it is guaranteed to always exist. for temp, it selects
     * the temperature rho parameter.
     */
    public void selectInitialParameter()
    {
        String name = null;
        if(_parameter.isPrecipitation())
        {
            if(_activeSourceModelParameters.getForecastSource().isClimatologySource())
            {
                name = _singletonTypeToNameMap.get(new AvgParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID,
                                                                        false));
            }
            else
            {
                name = _singletonTypeToNameMap.get(new RhoParameterType(HEFSTools.DEFAULT_PRECIPITATION_IDENTIFIER_PARAMETER_ID));
            }
        }
        else
        {
            if(_activeSourceModelParameters.getForecastSource().isClimatologySource())
            {
                name = _singletonTypeToNameMap.get(new AvgParameterType(HEFSTools.DEFAULT_TMIN_PARAMETER_ID, false));
            }
            else
            {
                name = _singletonTypeToNameMap.get(new RhoParameterType(HEFSTools.DEFAULT_TMAX_PARAMETER_ID));
            }
        }
        _availableParametersChoiceBox.setSelectedItem(name);
    }
}
