package ohd.hseb.hefs.mefp.pe.estimation.diag;

import javax.swing.JPanel;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelControlOptions;
import ohd.hseb.hefs.mefp.sources.historical.DistributionDiagnosticPanel;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.utils.dist.DistributionType;
import ohd.hseb.util.data.DataSet;

import com.google.common.collect.Lists;

/**
 * Displays a {@link DistributionDiagnosticPanel} allowing for examining CDFs and Q-Q plots for potential fitted
 * distributions.
 * 
 * @author Hank.Herr
 */
public class DistributionEventDaySummaryDiagnostic extends EventDaySummaryDiagnostic
{
    /**
     * The panel being displayed.
     */
    private DistributionDiagnosticPanel _panel;

    /**
     * Restricts the events used to those for which the fcst is zero (false) or positive (true) only.
     */
    private final boolean _fcstPositive;

    /**
     * Restricts the events used to those for which the obs is zero (false) or positive (true) only.
     */
    private final boolean _obsPositive;

    /**
     * Version for data which does not need to be restricted to positive/zero, such as temperature.
     */
    public DistributionEventDaySummaryDiagnostic(final MEFPFullModelParameters fullParameters,
                                                 final MEFPSourceModelParameters sourceParameters,
                                                 final CanonicalEvent event,
                                                 final int dayOfYear,
                                                 final StandardCanonicalEventValuesGatherer gatherer)
    {
        initialize(fullParameters, sourceParameters, event, dayOfYear, gatherer);
        _obsPositive = false;
        _fcstPositive = false;
        buildDisplayData(2);
    }

    public DistributionEventDaySummaryDiagnostic(final MEFPFullModelParameters fullParameters,
                                                 final MEFPSourceModelParameters sourceParameters,
                                                 final CanonicalEvent event,
                                                 final int dayOfYear,
                                                 final StandardCanonicalEventValuesGatherer gatherer,
                                                 final boolean obsPositive,
                                                 final boolean fcstPositive)
    {
        initialize(fullParameters, sourceParameters, event, dayOfYear, gatherer);
        _obsPositive = obsPositive;
        _fcstPositive = fcstPositive;
        buildDisplayData(2);
    }

    public void clearPanel()
    {
        _panel = null;
    }

    @Override
    public String toString()
    {
        if(getFullParameters().getIdentifier().isTemperatureDataType())
        {
            return "Distribution Plots";
        }
        if(_obsPositive && _fcstPositive)
        {
            return "Distribution Plots for Both Observation and Forecast Positive";
        }
        //Obs positive
        else if(_obsPositive && !_fcstPositive)
        {
            return "Distribution Plots for Observation Positive, Forecast Zero";
        }
        //Fcst positive
        else if(!_obsPositive && _fcstPositive)
        {
            return "Distribution Plots for Forecast Positive, Observed Zero";
        }
        return null;
    }

    @Override
    public ChartEngine buildDiagnosticChartEngine() throws Exception
    {
        return null;
    }

    @Override
    public JPanel buildChartPanelForChart() throws Exception
    {
        if(_panel != null)
        {
            return _panel;
        }

        final DataSet eventData = getDisplayData();
        DataSet dataSubset = eventData;

        Double precipThreshold = null;

        //Precipitation...
        if(getFullParameters().getIdentifier().isPrecipitationDataType())
        {
            precipThreshold = (double)((MEFPPrecipitationModelControlOptions)getFullParameters().getEstimationControlOptions()
                                                                                                .getModelParameters()).getEPT()
                                                                                                                      .getEPTPrecipThreshold();

            //Both positive
            if(_obsPositive && _fcstPositive)
            {
                dataSubset = eventData.extractSubset(0, DataSet.GREATER_THAN_OR_EQUAL_TO, precipThreshold); //obs subsetting
                dataSubset = dataSubset.extractSubset(1, DataSet.GREATER_THAN_OR_EQUAL_TO, precipThreshold); //fcst subsetting
                _panel = new DistributionDiagnosticPanel(getFullParameters().getIdentifier(),
                                                         precipThreshold,
                                                         Lists.newArrayList("Observed", "Forecasts"),
                                                         Lists.newArrayList(dataSubset.getVariable(0),
                                                                            dataSubset.getVariable(1)));
                _panel.setCheckedDistributions(((MEFPPrecipitationModelControlOptions)getFullParameters().getEstimationControlOptions()
                                                                                                         .getModelParameters()).getEPT()
                                                                                                                               .getEPTDistribution());
            }
            //Obs positive
            else if(_obsPositive && !_fcstPositive)
            {
                dataSubset = eventData.extractSubset(0, DataSet.GREATER_THAN_OR_EQUAL_TO, precipThreshold); //obs subsetting
                dataSubset = dataSubset.extractSubset(1, DataSet.LESS_THAN, precipThreshold); //fcst subsetting
                _panel = new DistributionDiagnosticPanel(getFullParameters().getIdentifier(),
                                                         precipThreshold,
                                                         Lists.newArrayList("Observed"),
                                                         Lists.newArrayList(dataSubset.getVariable(0)));
                _panel.setCheckedDistributions(DistributionType.GAMMA);
            }
            //Fcst positive
            else if(!_obsPositive && _fcstPositive)
            {
                dataSubset = eventData.extractSubset(0, DataSet.LESS_THAN, precipThreshold); //obs subsetting
                dataSubset = dataSubset.extractSubset(1, DataSet.GREATER_THAN_OR_EQUAL_TO, precipThreshold); //fcst subsetting
                _panel = new DistributionDiagnosticPanel(getFullParameters().getIdentifier(),
                                                         precipThreshold,
                                                         Lists.newArrayList("Forecasts"),
                                                         Lists.newArrayList(dataSubset.getVariable(1)));
                _panel.setCheckedDistributions(DistributionType.GAMMA);
            }
        }
        //Temperature...
        else
        {
            _panel = new DistributionDiagnosticPanel(getFullParameters().getIdentifier(),
                                                     precipThreshold,
                                                     Lists.newArrayList("Observed", "Forecasts"),
                                                     Lists.newArrayList(eventData.getVariable(0),
                                                                        eventData.getVariable(1)));
            _panel.setCheckedDistributions(DistributionType.NORMAL);
        }

        return _panel;
    }
}
