package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.Color;
import java.awt.Font;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.DataSetXYChartDataSource;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.EPTPrecipitationParameterCalculator;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;

import com.google.common.collect.Lists;

public class EPTScatterEventDaySummaryDiagnostic extends EventDaySummaryDiagnostic
{

    public EPTScatterEventDaySummaryDiagnostic(final MEFPFullModelParameters fullParameters,
                                               final MEFPSourceModelParameters sourceParameters,
                                               final CanonicalEvent event,
                                               final int dayOfYear,
                                               final StandardCanonicalEventValuesGatherer gatherer)
    {
        initialize(fullParameters, sourceParameters, event, dayOfYear, gatherer);
        buildDisplayData(2);
    }

    /**
     * @return A string summarizing the counts of the specific events.
     */
    private String buildEPTCountString()
    {
        final EPTPrecipitationParameterCalculator calc = new EPTPrecipitationParameterCalculator(((MEFPPrecipitationModelControlOptions)getFullParameters().getEstimationControlOptions()
                                                                                                                                                           .getModelParameters()).getEPT()
                                                                                                                                                                                 .getEPTPrecipThreshold());
        final int[] counts = calc.computeCounts(getGatherer().getForecastGatheredEventValues(getEvent()),
                                                getGatherer().getObservedGatheredEventValues(getEvent()));
        return "Neither Positive = " + counts[0] + "   Obs Positive, Fcst Not = " + counts[1]
            + "   Fcst Positive, Obs Not = " + counts[2] + "   Both Positive = " + counts[3];
    }

    /**
     * Builds a scatter plot that displays the precipitation data in the original space.
     * 
     * @return The chart to display.
     * @throws Exception
     */
    @Override
    public ChartEngine buildDiagnosticChartEngine() throws Exception
    {
        final DataSetXYChartDataSource dataSource = new DataSetXYChartDataSource(null,
                                                                                 0,
                                                                                 getDisplayData(),
                                                                                 new int[]{0},
                                                                                 new int[]{1});
        dataSource.setSourceNameInTable("Scatter Plot of Forecast Against Observed");
        dataSource.addChartTableColumnHeader(0, "Forecast");
        dataSource.setDomainHeader("Observed");

        final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)dataSource));
        engine.getChartParameters().getDataSourceParameters(0).setPlotterName("LineAndScatter");
        final SeriesDrawingParameters seriesParms = engine.getChartParameters()
                                                          .getDataSourceParameters(0)
                                                          .getSeriesDrawingParametersForSeriesIndex(0);
        seriesParms.setLineWidth(0.0f);
        seriesParms.setShapeName("circle");
        seriesParms.setShapeFilled(true);
        seriesParms.setFillColor(seriesParms.getLineColor());
        seriesParms.setShapeSize(1.0d);

        //Correlation is displayed only if this is for a specific day-of-year.
        double value = Double.NaN;
        if(getDayOfYear() > 0)
        {
            value = this.getSourceParameters().getEPTPrecipitationRhoParameterValue(getDayOfYear(), getEvent());
        }

        //Determine the window width used to gather values.
        String windowWidthUsedForGathering = "";
        if(getDayOfYear() >= 0)
        {
            windowWidthUsedForGathering = "\nWindow Width Used to Gather Events = "
                + getGatherer().getWindowWidthUsedForGatheredValues(getEvent()) + " Days";
            if(getGatherer().getWindowWidthUsedForGatheredValues(getEvent()) < 0)
            {
                windowWidthUsedForGathering = "Window Width Used to Gather Events = "
                    + getGatherer().getMaximumWindowWidth() + " Days (not enough data found)";
            }
        }

        //Build the title text.
        String titleText = "EPT -- Scatter Plot of Forecast Against Observed";
        if(!Double.isNaN(value))
        {
            titleText += " (EPT Correlation = " + value + ")";
        }
        titleText += "\n"
            + buildEPTCountString()
            + "\nRain Threhold = "
            + ((MEFPPrecipitationModelControlOptions)getFullParameters().getEstimationControlOptions()
                                                                        .getModelParameters()).getEPT()
                                                                                              .getEPTPrecipThreshold()
            + " MM" + windowWidthUsedForGathering;

        engine.getChartParameters().getPlotTitle().setText(titleText);
        engine.getChartParameters().getPlotTitle().setFont(new Font("Serif", Font.BOLD, 14));
        engine.getChartParameters().getDomainAxis().getLabel().setText("Observed Canonical Event Value [MM]"); //Precip is always MM units in MEFP
        engine.getChartParameters()
              .getSubPlot(0)
              .getLeftAxis()
              .getLabel()
              .setText("Forecast Canonical Event Value [MM]");

        ThresholdParameters threshold = new ThresholdParameters("Fcst PThresh");
        threshold.setVisible(true);
        threshold.setupDefaultParameters();
        threshold.setNumericalAxisValueStartStr("" + getGatherer().getLastComputedForecastThreshold(getEvent()));
        threshold.getLabel().setText("Rain Threshold");
        threshold.setLabelAnchor("top_right");
        threshold.setLineWidth(2.0f);
        threshold.setColor(Color.BLUE);
        engine.getChartParameters().getThresholdList().addThreshold(threshold);

        threshold = new ThresholdParameters("Obs PThresh");
        threshold.setVisible(true);
        threshold.setupDefaultParameters();
        threshold.setNumericalAxisValueStartStr("" + getGatherer().getLastComputedObservedThreshold(getEvent()));
        threshold.getLabel().setText("Rain Threshold");
        threshold.setLabelAnchor("top_right");
        threshold.setLineWidth(2.0f);
        threshold.setColor(Color.BLUE);
        threshold.setAxisIdString(ChartConstants.DOMAIN_AXIS_STRING);
        engine.getChartParameters().getThresholdList().addThreshold(threshold);

        return engine;
    }

    @Override
    public String toString()
    {
        return "MAP: Scatter Plot of Raw Forecasts Against Observed Values and EPT Rain Threshold ";
    }
}
