package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.util.List;

import javax.swing.JPanel;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartPanelTools;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.panel.OHDFixedChartPanel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.util.data.DataSet;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;

/**
 * Generic super class of all diagnostic plug-ins to the {@link EventDaySummaryPanel}. It will be called as needed by
 * that panel. Note than every subclass of this must be constructed in the
 * {@link EventDaySummaryPanel#initializeEventDaySummaryDiagnostics()} method within the appropriate subclass of
 * {@link EventDaySummaryPanel}.
 * 
 * @author hankherr
 */
public abstract class EventDaySummaryDiagnostic
{

    private MEFPSourceModelParameters _sourceParameters;
    private MEFPFullModelParameters _fullParameters;
    private CanonicalEvent _event;
    private int _dayOfYear;
    private StandardCanonicalEventValuesGatherer _gatherer;

    /**
     * Call {@link #buildDisplayData(int)} to initialize this {@link DataSet}.
     */
    private DataSet _displayData;

    protected void initialize(final MEFPFullModelParameters fullParameters,
                              final MEFPSourceModelParameters sourceParameters,
                              final CanonicalEvent event,
                              final int dayOfYear,
                              final StandardCanonicalEventValuesGatherer gatherer)
    {
        _fullParameters = fullParameters;
        _sourceParameters = sourceParameters;
        _event = event;
        _dayOfYear = dayOfYear;
        _gatherer = gatherer;
        _displayData = null;
    }

    /**
     * Builds {@link #_displayData} to contain the number of variables specified as an argument. Variables 0 and 1 will
     * be the observed and forecast event values acquired from the {@link #_gatherer} for the {@link #_event}. Variable
     * 2 is the forecast T0 corresponding to the forecast event (from which the observed event is also computed) stored
     * as a long cast to a double. This is intended as a tool to initialize a {@link DataSet} for further use and should
     * only be called if needed.
     * 
     * @param numberOfVariables The number of variables to put in the {@link #_displayData}. If less than 3, 3 will be
     *            used.
     * @return A {@link DataSet} containing the canonical events gathered according to the provided
     *         {@link StandardCanonicalEventValuesGatherer}.
     */
    protected void buildDisplayData(int numberOfVariables)
    {
        if(numberOfVariables < 3)
        {
            numberOfVariables = 3;
        }

        //Gather events
        if(_dayOfYear >= 0)
        {
            _gatherer.gatherEventValues(_dayOfYear);
        }
        else
        {
            _gatherer.gatherEventValuesForAllDays();
        }
        final List<Float> obsValues = _gatherer.getObservedGatheredEventValues(_event);
        final List<Float> fcstValues = _gatherer.getForecastGatheredEventValues(_event);
        final List<Long> fcstT0s = _gatherer.getForecastT0s(_event);

        //Put the gathered events in the data set
        final DataSet dataSet = new DataSet(obsValues.size(), numberOfVariables);
        for(int i = 0; i < obsValues.size(); i++)
        {
            dataSet.addSample(new double[]{obsValues.get(i), fcstValues.get(i), fcstT0s.get(i)});
        }
        _displayData = dataSet;
    }

    @Override
    public abstract String toString();

    /**
     * @return A {@link ChartEngine} that is the diagnostic display to be shown.
     */
    public abstract ChartEngine buildDiagnosticChartEngine() throws Exception;

    /**
     * Builds a {@link JPanel} given the provided {@link ChartEngine}. It calls the {@link ChartEngine#buildChart()}
     * method to build the {@link JFreeChart} that will be displayed. If a failure occurs, the returned {@link JPanel}
     * will display an error message. If successful, the returned panel will be an instance of
     * {@link ChartEngineChartAndTablePanel}, allowing for display of the plot and data table.<br>
     * <br>
     * This method ensures that the axis limits are squared whenever the {@link ChartPanel#restoreAutoBounds()} is
     * called.
     * 
     * @return {@link JPanel} to display in a frame.
     * @throws Exception
     */
    @SuppressWarnings("serial")
    public JPanel buildChartPanelForChart() throws Exception
    {
        final ChartEngine engine = buildDiagnosticChartEngine();
        final JFreeChart chart = engine.buildChart();
        ChartTools.squareAxes(chart, "left");
        final JPanel panel = ChartPanelTools.buildPanelFromChartEngine(engine,
                                                                              chart,
                                                                              false,
                                                                              EventDaySummaryTableModel.class);

        if(panel instanceof ChartEngineChartAndTablePanel)
        {
            //Need a special chart panel that squares the axes after computation.  Hence, I override restoreAutoBounds.
            ((ChartEngineChartAndTablePanel)panel).setChartPanel(new OHDFixedChartPanel(chart,
                                                                                        false,
                                                                                        true,
                                                                                        true,
                                                                                        true,
                                                                                        true)
            {
                @Override
                public void restoreAutoBounds()
                {
                    super.restoreAutoBounds();
                    ChartTools.squareAxes(getChart(), "left");
                }
            });

        }
        return panel;
    }

    public MEFPSourceModelParameters getSourceParameters()
    {
        return _sourceParameters;
    }

    public void setSourceParameters(final MEFPSourceModelParameters sourceParameters)
    {
        _sourceParameters = sourceParameters;
    }

    public MEFPFullModelParameters getFullParameters()
    {
        return _fullParameters;
    }

    public void setFullParameters(final MEFPFullModelParameters fullParameters)
    {
        _fullParameters = fullParameters;
    }

    public CanonicalEvent getEvent()
    {
        return _event;
    }

    public void setEvent(final CanonicalEvent event)
    {
        _event = event;
    }

    public int getDayOfYear()
    {
        return _dayOfYear;
    }

    public void setDayOfYear(final int dayOfYear)
    {
        _dayOfYear = dayOfYear;
    }

    public StandardCanonicalEventValuesGatherer getGatherer()
    {
        return _gatherer;
    }

    public void setGatherer(final StandardCanonicalEventValuesGatherer gatherer)
    {
        _gatherer = gatherer;
    }

    /**
     * @return Built via {@link #buildDisplayData(int)}, the resulting {@link DataSet} includes obs event value in var
     *         0, fcst event value in var 1, and the forecast T0 (long mapped to double) in var 2.
     */
    public DataSet getDisplayData()
    {
        return _displayData;
    }
}
