package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRootPane;

import ohd.hseb.charter.panel.ChartEngineChartAndTablePanel;
import ohd.hseb.charter.panel.CombinedDomainChartNavigationPanel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.gui.components.ComponentPanelGlassPane;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

import com.google.common.collect.Lists;

/**
 * Super class of the data type specific event day summary panels.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public abstract class EventDaySummaryPanel extends JPanel
{

    private MEFPSourceModelParameters _sourceParameters;
    private MEFPFullModelParameters _fullParameters;
    private CanonicalEvent _event;
    private int _dayOfYear;
    private final List<EventDaySummaryDiagnostic> _diagnostics = Lists.newArrayList();
    private final JPanel _diagnosticPanel = new JPanel();
    private JPanel _panel;

    /**
     * Will contain the navigation panel.
     */
    private JPanel _navPanelHolder;

    private ComponentPanelGlassPane _glassPane;

    /**
     * Initializes the attributes and calls {@link #initializeEventDaySummaryDiagnostics()}, {@link #createDisplay()},
     * and {@link #updateDisplay(int)} using the first diagnostic.
     * 
     * @param sourceParameters
     * @param fullParameters
     * @param event
     * @param dayOfYear
     * @throws Exception
     */
    public void initialize(final MEFPSourceModelParameters sourceParameters,
                           final MEFPFullModelParameters fullParameters,
                           final CanonicalEvent event,
                           final int dayOfYear) throws Exception
    {
        _sourceParameters = sourceParameters;
        _fullParameters = fullParameters;
        _event = event;
        _dayOfYear = dayOfYear;

        initializeEventDaySummaryDiagnostics();
        createDisplay();
        updateDisplay(0);
    }

    private void createDisplay()
    {
        removeAll();
        _diagnosticPanel.setLayout(new BorderLayout());

        //Contains the navigation panel in a BorderLayout.
        _navPanelHolder = new JPanel(new BorderLayout());
        _navPanelHolder.setPreferredSize(new Dimension(190, 150));
        _navPanelHolder.setOpaque(false);

        final JPanel choicesPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        choicesPanel.add(new JLabel("Select Diagnostic: "));
        final JComboBox choices = new JComboBox(_diagnostics.toArray());
        choices.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                updateDisplay(choices.getSelectedIndex());
            }
        });
        choicesPanel.add(choices);

        //Create the root pane and add the glassPane to it, which contains the _navPanelHolder.
        final JRootPane rootPane = new JRootPane();
        rootPane.setContentPane(_diagnosticPanel);
        _glassPane = new ComponentPanelGlassPane(Lists.newArrayList((Component)_navPanelHolder), FlowLayout.TRAILING);
        _glassPane.addToPane(rootPane);
        _glassPane.setVisible(true);

        this.setLayout(new BorderLayout());
        this.add(choicesPanel, BorderLayout.NORTH);
        this.add(rootPane, BorderLayout.CENTER);
    }

    /**
     * Must be overridden to initialize the list of available diagnostics by calling
     * {@link #addDiagnostic(EventDaySummaryDiagnostic)} for each one. Diagnostics are listed in the combo box based on
     * their to string values.
     */
    protected abstract void initializeEventDaySummaryDiagnostics();

    /**
     * Called to update the display.
     * 
     * @throws Exception
     */
    private void updateDisplay(final int selectedDiagnostic)
    {
        try
        {
            //Build the chart and its display panel.  The nav panel will fill up the _navPanelHolder by using its getPreferredSize().
            final JPanel newPanel = _diagnostics.get(selectedDiagnostic).buildChartPanelForChart();

            //If this is the first time here, or the new panel displays an error, or the existing panel displays an error panel,
            //then the new panel must be directly used to display either the error or the chart/table.
            if((_panel == null) || !(newPanel instanceof ChartEngineChartAndTablePanel)
                || !(_panel instanceof ChartEngineChartAndTablePanel))
            {
                //Build it.
                _panel = newPanel;

                //If the panel was successfully constructed, update the nav panel.  Otherwise, the nav panel is forced to not be visible.
                if(_panel instanceof ChartEngineChartAndTablePanel)
                {
                    ((ChartEngineChartAndTablePanel)_panel).removeDataSourceChoiceBoxForTable();
                    final CombinedDomainChartNavigationPanel navPanel = CombinedDomainChartNavigationPanel.applyPopupNavigationPanelToChart(((ChartEngineChartAndTablePanel)_panel).getChartPanel(),
                                                                                                                                            _navPanelHolder.getPreferredSize(),
                                                                                                                                            true);
                    _navPanelHolder.removeAll();
                    _navPanelHolder.add(navPanel, BorderLayout.CENTER);
                    _navPanelHolder.setVisible(true);
                }
                else
                {
                    _navPanelHolder.setVisible(false);
                }

                //Put the new panel in place.
                _diagnosticPanel.removeAll();
                _diagnosticPanel.add(_panel, BorderLayout.CENTER);
            }
            //Otherwise, both the existing and new chart/table panels are valid, so just update the data displayed in the existing panel.
            else
            {
                ((ChartEngineChartAndTablePanel)_panel).setChartEngineAndChart(((ChartEngineChartAndTablePanel)newPanel).getChartEngine(),
                                                                               ((ChartEngineChartAndTablePanel)newPanel).getChartPanel()
                                                                                                                        .getChart());
            }

            SwingTools.forceComponentRedraw(this);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            _diagnosticPanel.add(HSwingFactory.createErrorMessagePane("Unable to display diagnostic:", e.getMessage()));
        }
    }

    /**
     * Call to add a diagnostic. This should only be called during initialization of the panel.
     * 
     * @param diagnostic The diagnostic to add.
     */
    public void addDiagnostic(final EventDaySummaryDiagnostic diagnostic)
    {
        _diagnostics.add(diagnostic);
    }

    public MEFPSourceModelParameters getSourceParameters()
    {
        return _sourceParameters;
    }

    public void setSourceParameters(final MEFPSourceModelParameters sourceParameters)
    {
        _sourceParameters = sourceParameters;
    }

    public MEFPFullModelParameters getFullParameters()
    {
        return _fullParameters;
    }

    public void setFullParameters(final MEFPFullModelParameters fullParameters)
    {
        _fullParameters = fullParameters;
    }

    public CanonicalEvent getEvent()
    {
        return _event;
    }

    public void setEvent(final CanonicalEvent event)
    {
        _event = event;
    }

    public int getDayOfYear()
    {
        return _dayOfYear;
    }

    public void setDayOfYear(final int dayOfYear)
    {
        _dayOfYear = dayOfYear;
    }

}
