package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.GridLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JPanel;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;

import com.google.common.eventbus.EventBus;

/**
 * Displays a combo box of selectable {@link MEFPForecastSource} names. An {@link EventBus} is created for notices when
 * a parameter selection is made: {@link ForecastSourceSelectedNotice}. To hear those events, call
 * {@link #registerWithBus(Object)} and make sure the passed in object implements
 * {@link ForecastSourceSelectedNotice.Subscriber}
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ForecastSourceSelectionPanel extends JPanel
{
    private final JComboBox _availableSourcesChoiceBox;
    private final EventBus _eventBus = new EventBus();

    /**
     * @param sourceParameters The source parameters used to extract a list of parameter names.
     */
    public ForecastSourceSelectionPanel(final MEFPFullModelParameters fullParameters)
    {
        //List of sources
        final List<MEFPForecastSource> sources = MEFPTools.generateListOfSourcesWithParameters(fullParameters);
        if(sources.isEmpty())
        {
            throw new IllegalStateException("No sources include estimated parameters in the parameters for "
                + fullParameters.getIdentifier().buildStringToDisplayInTree());
        }

        _availableSourcesChoiceBox = HSwingFactory.createJComboBox(sources, null);
        _availableSourcesChoiceBox.setSelectedIndex(0);
        _availableSourcesChoiceBox.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(final ItemEvent e)
            {
                if(e.getStateChange() == ItemEvent.SELECTED)
                {
                    _eventBus.post(new ForecastSourceSelectedNotice(this, getSelectedSource()));
                }
            }
        });

        this.setLayout(new GridLayout(1, 0));
        add(_availableSourcesChoiceBox);

        setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1), "Select Forecast Source", null));
    }

    public MEFPForecastSource getSelectedSource()
    {
        return (MEFPForecastSource)_availableSourcesChoiceBox.getSelectedItem();
    }

    /**
     * Register the supplied Object to the {@link EventBus} that underlies this panel.
     */
    public void registerWithBus(final ForecastSourceSelectedNotice.Subscriber registree)
    {
        _eventBus.register(registree);
    }

    /**
     * Unregister the supplied Object to the {@link EventBus} that underlies this panel.
     */
    public void unregisterWithBus(final ForecastSourceSelectedNotice.Subscriber registree)
    {
        _eventBus.unregister(registree);
    }
}
