package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.Color;
import java.awt.Font;
import java.text.DecimalFormat;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.charter.datasource.instances.DataSetXYChartDataSource;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.IPTPrecipitationParameterCalculator;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.util.misc.HNumber;

import com.google.common.collect.Lists;

public class IPTScatterEventDaySummaryDiagnostic extends EventDaySummaryDiagnostic
{

    public IPTScatterEventDaySummaryDiagnostic(final MEFPFullModelParameters fullParameters,
                                               final MEFPSourceModelParameters sourceParameters,
                                               final CanonicalEvent event,
                                               final int dayOfYear,
                                               final StandardCanonicalEventValuesGatherer gatherer)
    {
        initialize(fullParameters, sourceParameters, event, dayOfYear, gatherer);
        buildDisplayData(2);
    }

    /**
     * @return A string summarizing the counts of the specific events.
     */
    private String buildIPTCountString()
    {
        final IPTPrecipitationParameterCalculator calc = new IPTPrecipitationParameterCalculator(getGatherer().getLastComputedForecastThreshold(getEvent()),
                                                                                                 getGatherer().getLastComputedObservedThreshold(getEvent()),
                                                                                                 ((MEFPPrecipitationModelControlOptions)getFullParameters().getEstimationControlOptions()
                                                                                                                                                           .getModelParameters()).getIPT()
                                                                                                                                                                                 .getMaximumConditionalMeanPrecipitationValue());
        final int[] counts = calc.computeCounts(getGatherer().getForecastGatheredEventValues(getEvent()),
                                                getGatherer().getObservedGatheredEventValues(getEvent()));
        return "Neither Positive = " + counts[0] + "   Obs Positive, Fcst Not = " + counts[2]
            + "   Fcst Positive, Obs Not = " + counts[1] + "   Both Positive = " + counts[3];
    }

    /**
     * Builds a scatter plot that displays the precipitation data in the original space.
     * 
     * @return The chart to display.
     * @throws Exception
     */
    @Override
    public ChartEngine buildDiagnosticChartEngine() throws Exception
    {
        final DataSetXYChartDataSource dataSource = new DataSetXYChartDataSource(null,
                                                                                 0,
                                                                                 getDisplayData(),
                                                                                 new int[]{0},
                                                                                 new int[]{1});
        dataSource.setSourceNameInTable("Scatter Plot of Forecast Against Observed");
        dataSource.addChartTableColumnHeader(0, "Forecast");
        dataSource.setDomainHeader("Observed");

        final ChartEngine engine = new ChartEngine(Lists.newArrayList((XYChartDataSource)dataSource));
        engine.getChartParameters().getDataSourceParameters(0).setPlotterName("LineAndScatter");
        final SeriesDrawingParameters seriesParms = engine.getChartParameters()
                                                          .getDataSourceParameters(0)
                                                          .getSeriesDrawingParametersForSeriesIndex(0);
        seriesParms.setLineWidth(0.0f);
        seriesParms.setShapeName("circle");
        seriesParms.setShapeFilled(true);
        seriesParms.setFillColor(seriesParms.getLineColor());
        seriesParms.setShapeSize(1.0d);

        final DecimalFormat numFormat = new DecimalFormat("0.000");

        //The variable value is the correlation coefficient.  It is assumed to be 1.0 for a climatology source.
        double value = 1.0d;
        if(!this.getSourceParameters().getForecastSource().isClimatologySource())
        {
            value = this.getSourceParameters().getPrecipitationRhoParameterValue(getDayOfYear(), getEvent());
        }

        //Determine the window width used to gather values.
        String windowWidthUsedForGathering = "";
        if(getDayOfYear() >= 0)
        {
            windowWidthUsedForGathering = "\nWindow Width Used to Gather Events = "
                + getGatherer().getWindowWidthUsedForGatheredValues(getEvent()) + " Days";
            if(getGatherer().getWindowWidthUsedForGatheredValues(getEvent()) < 0)
            {
                windowWidthUsedForGathering = "Window Width Used to Gather Events = "
                    + getGatherer().getMaximumWindowWidth() + " Days (not enough data found)";
            }
        }

        //Build the title.
        String titleText = "IPT -- Scatter Plot of Forecast Against Observed";
        if(getDayOfYear() >= 0)
        {
            titleText += " (IPT Correlation = " + value + ")";
        }
        titleText += "\n"
            + buildIPTCountString()
            + "\n0-Thresholds Computed to Ensure the Fraction of Total Accumulated Precipitation Greater Than the Threshold is: "
            + numFormat.format(getGatherer().getFractionOfTotalPrecipGTZero()) + "\n" + "Forecast 0-Threshold = "
            + HNumber.roundDouble(getGatherer().getLastComputedForecastThreshold(getEvent()), 4)
            + "  Observed 0-Threshold = "
            + HNumber.roundDouble(getGatherer().getLastComputedObservedThreshold(getEvent()), 4)
            + windowWidthUsedForGathering;
        engine.getChartParameters().getPlotTitle().setText(titleText);

        engine.getChartParameters().getPlotTitle().setFont(new Font("Serif", Font.BOLD, 14));
        engine.getChartParameters().getDomainAxis().getLabel().setText("Observed Canonical Event Value [MM]"); //Precip is always MM units in MEFP
        engine.getChartParameters()
              .getSubPlot(0)
              .getLeftAxis()
              .getLabel()
              .setText("Forecast Canonical Event Value [MM]");

        ThresholdParameters threshold = new ThresholdParameters("Fcst PThresh");
        threshold.setVisible(true);
        threshold.setupDefaultParameters();
        threshold.setNumericalAxisValueStartStr("" + getGatherer().getLastComputedForecastThreshold(getEvent()));
        threshold.getLabel().setText("Forecast 0-Threshold");
        threshold.setLabelAnchor("top_right");
        threshold.setLineWidth(2.0f);
        threshold.setColor(Color.BLUE);
        engine.getChartParameters().getThresholdList().addThreshold(threshold);

        threshold = new ThresholdParameters("Obs PThresh");
        threshold.setVisible(true);
        threshold.setupDefaultParameters();
        threshold.setNumericalAxisValueStartStr("" + getGatherer().getLastComputedObservedThreshold(getEvent()));
        threshold.getLabel().setText("Observed 0-Threshold");
        threshold.setLabelAnchor("top_right");
        threshold.setLineWidth(2.0f);
        threshold.setColor(Color.BLUE);
        threshold.setAxisIdString(ChartConstants.DOMAIN_AXIS_STRING);
        engine.getChartParameters().getThresholdList().addThreshold(threshold);

        return engine;
    }

    @Override
    public String toString()
    {
        return "MAP: Scatter Plot of Raw Forecast Against Observed Values and IPT Precipitation Thresholds";
    }

}
