package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.MouseInfo;
import java.awt.Paint;
import java.awt.Point;
import java.awt.Shape;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Collection;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSpinner;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;

import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYBlockRenderer;
import ohd.hseb.charter.panel.OHDFixedChartPanel;
import ohd.hseb.charter.panel.notices.GeneralAxisLimitsChangedNotice;
import ohd.hseb.charter.panel.notices.GeneralTableCellSelectedNotice;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.utils.gui.jtable.EventPostingCellSelectableTable;
import ohd.hseb.hefs.utils.gui.jtable.MarkedTableScrollPanel;
import ohd.hseb.hefs.utils.gui.jtable.OutputDataToCSVFileMenuItem;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningMenuItem;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HNumber;

import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.AxisLocation;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.SymbolAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.block.BlockBorder;
import org.jfree.chart.entity.XYItemEntity;
import org.jfree.chart.labels.XYToolTipGenerator;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.LookupPaintScale;
import org.jfree.chart.renderer.PaintScale;
import org.jfree.chart.renderer.xy.XYBlockRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.chart.title.PaintScaleLegend;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.RectangleAnchor;
import org.jfree.ui.RectangleEdge;
import org.jfree.ui.RectangleInsets;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * Displays a block plot, created via n {@link XYBlockRenderer}, for the specified parameter. This is the starting point
 * for identifying questionable parameters and their cause.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ParameterBlockDiagnosticPanel extends ParameterEstimatorDiagnosticPanel implements NoticePoster,
GeneralTableCellSelectedNotice.Subscriber, GeneralAxisLimitsChangedNotice.Subscriber
{
    private final static DecimalFormat DOMAIN_TICK_LABEL_NUMBER_FORMATTER = new DecimalFormat("#.##");

    private final Font LEGEND_TICK_FONT = new Font("Dialog", Font.PLAIN, 10);
    private final Font AXIS_TICK_FONT = new Font("Dialog", Font.PLAIN, 10);
    private final Font AXIS_LABEL_FONT = new Font("Serif", Font.BOLD, 14);
    private final Font TITLE_FONT = new Font("Serif", Font.BOLD, 16);

    private GraphGenXYBlockRenderer _blockRenderer;
    private XYLineAndShapeRenderer _questionableRenderer;
    private ParameterBlockDiagnosticXYZDataset _blockDataset;
    private XYDataset _questionableDataset;

    /**
     * The paint scale employed the first time the block panel is constructed. This is only changed when
     * {@link #generatePaintScale(double, double)} is called and the {@link #_paintScaleBoundsOverride} is null (i.e.,
     * the first time the block panel is created for a new set of data selections).
     */
    private PaintScale _defaultPaintScale = null;

    /**
     * The {@link navSubPanel} displayed in the panel.
     */
    private final OHDFixedChartPanel _currentChartPanel;

    /**
     * The model for the {@link #_chartTable}.
     */
    private final ParameterBlockXYZDatasetTableModel _tableModel;

    /**
     * Table displayed within {@link #_scrollPanel}.
     */
    private final EventPostingCellSelectableTable _chartTable;

    /**
     * Container for the table allows for marking rows in scroll bars.
     */
    private final MarkedTableScrollPanel _scrollPanel;

    /**
     * If the paint scale limits are overridden via user interaction, then this will be non null with the x-value
     * storing the lower bound and y value storing the upper bound.
     */
    private Point2D _paintScaleBoundsOverride = null;

    /**
     * @param fullParameters Contains the parameters for display in the diagnostic plot.
     * @param sources Sources for which a diagnostic is to be displayed. Number of sources must match the number of
     *            parameters to display.
     * @param displayedEvents The events for which parameters are to be displayed.
     * @param parameterToDisplay The parameters to display. More accurately, these are the parameter types used in the
     *            computation of the parameter to display. If one is provided, then that parameter is directly
     *            displayed. If two are provided, then a difference is displayed.
     * @param prepareForPanelDisplay If true, then it is assumed that this panel will be displayed inside a container
     *            and therefore must be fully prepared. Otherwise, it is assumed that this is being called only to
     *            create a chart that can be used to generate an image.
     */
    public ParameterBlockDiagnosticPanel(final MEFPFullModelParameters fullParameters,
                                         final MEFPForecastSource[] sources,
                                         final Collection<CanonicalEvent> displayedEvents,
                                         final ModelParameterType[] parameterToDisplay,
                                         final boolean prepareForPanelDisplay)
    {

        _blockDataset = new ParameterBlockDiagnosticXYZDataset(fullParameters,
                                                               sources,
                                                               displayedEvents,
                                                               parameterToDisplay);
        final JFreeChart chart = buildXYBlockChart();

        setLayout(new BorderLayout());
        _currentChartPanel = new OHDFixedChartPanel(chart);

        if(prepareForPanelDisplay)
        {
            _currentChartPanel.addChartMouseListener(new ChartMouseListener()
            {
                @Override
                public void chartMouseClicked(final ChartMouseEvent event)
                {
                    //The legend was clicked.
                    if(((LegendBoundsRecordingPaintScaleLegend)event.getChart().getSubtitle(0)).getBounds()
                                                                                               .contains(event.getTrigger()
                                                                                                              .getX(),
                                                                                                         event.getTrigger()
                                                                                                              .getY()))
                    {
                        processLegendClick(event.getChart());
                    }

                    //A data item was clicked.
                    else if(event.getEntity() instanceof XYItemEntity)
                    {
                        processChartClick(event, (XYItemEntity)event.getEntity());
                    }
                }

                @Override
                public void chartMouseMoved(final ChartMouseEvent arg0)
                {
                }
            });
            addAxisChangeListener();

            _tableModel = new ParameterBlockXYZDatasetTableModel(_blockDataset);
            _chartTable = new EventPostingCellSelectableTable(_tableModel);
            _chartTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
            _tableModel.setDatasetAndChart(_blockDataset, chart);

            getCentralBus().register(this);

            _scrollPanel = new MarkedTableScrollPanel(_chartTable);

            //Add the CSV dump to popup menu.
            final Component csvMenuItem = new OutputDataToCSVFileMenuItem(_chartTable,
                                                                          SwingTools.getGlobalDialogParent(_chartTable));
            final JPopupMenu popupMenu = new JPopupMenu();
            popupMenu.add(csvMenuItem);
            _scrollPanel.setComponentPopupMenu(popupMenu);

            final JTabbedPane tabPane = new JTabbedPane();
            tabPane.addTab("Chart", _currentChartPanel);
            tabPane.addTab("Table", _scrollPanel);

            add(tabPane, BorderLayout.CENTER);

            setupRendererToolTips();
        }
        else
        {
            _tableModel = null;
            _chartTable = null;
            _scrollPanel = null;
        }
    }

    /**
     * Calls the other constructor passing in true for prepareForPanelDisplay.
     */
    public ParameterBlockDiagnosticPanel(final MEFPFullModelParameters fullParameters,
                                         final MEFPForecastSource[] sources,
                                         final Collection<CanonicalEvent> displayedEvents,
                                         final ModelParameterType[] parameterToDisplay)
    {
        this(fullParameters, sources, displayedEvents, parameterToDisplay, true);
    }

    /**
     * @return This will use the central bus underlying the {@link #_chartTable}.
     */
    private EventBus getCentralBus()
    {
        return _chartTable.getCentralBus();
    }

    /**
     * Call to display a new chart within this panel. This keeps the {@link #_currentChartPanel} the same while only
     * changing the chart it displays. Recomputes the {@link #_blockDataset}, builds the chart, and sets the
     * {@link #_currentChartPanel} to display the newly built chart. That will trigger any navigation panel displaying
     * the chart to update automatically.
     * 
     * @param fullParameters The full parameters, typically the same as the previous full parameters.
     * @param sources The new selected sources.
     * @param displayedEvents Events to display.
     * @param parameterToDisplay Parameters to display.
     */
    public void updateForNewChart(final MEFPFullModelParameters fullParameters,
                                  final MEFPForecastSource[] sources,
                                  final Collection<CanonicalEvent> displayedEvents,
                                  final ModelParameterType[] parameterToDisplay)
    {
        final ParameterBlockDiagnosticXYZDataset oldSet = _blockDataset;
        _blockDataset = new ParameterBlockDiagnosticXYZDataset(fullParameters,
                                                               sources,
                                                               displayedEvents,
                                                               parameterToDisplay);

        //Check the old set parameter type against the new set.  If the type is changed, then
        //we cannot use the previous pain scale, so clear out the default and override paint scales.
        if((oldSet != null) && !_blockDataset.getParameterType().equals(oldSet.getParameterType()))
        {
            _defaultPaintScale = null;
            _paintScaleBoundsOverride = null;
        }

        final JFreeChart chart = buildXYBlockChart();
        _currentChartPanel.setChart(chart);
        addAxisChangeListener();
        _tableModel.setDatasetAndChart(_blockDataset, chart);
        setupRendererToolTips();
    }

    /**
     * @return The chart panel for displaying the block chart.
     */
    public OHDFixedChartPanel getChartPanel()
    {
        return _currentChartPanel;
    }

    /**
     * @return The underlying chart.
     */
    public JFreeChart getChart()
    {
        return _currentChartPanel.getChart();
    }

    /**
     * Called to process a legend click.
     * 
     * @param chart The chart, which is acquired from the chart event.
     */
    private void processLegendClick(final JFreeChart chart)
    {
        final LegendBoundsRecordingPaintScaleLegend legend = (LegendBoundsRecordingPaintScaleLegend)chart.getSubtitle(0);

        //Step size is determined using the HNumber computeDecimalPlaces method using the current lower and upper bounds.
        final double stepSize = Math.pow(10.0,
                                         -1
                                             * (HNumber.computeDecimalPlaces(legend.getScale().getUpperBound()
                                                 - legend.getScale().getLowerBound(), 2) + 1));

        //Each spinner is created and placed within a panel.
        final JSpinner lowerSpinner = HSwingFactory.createJSpinner(null,
                                                                   10,
                                                                   legend.getScale().getLowerBound(),
                                                                   null,
                                                                   null,
                                                                   stepSize);
        final JPanel lowerSpinnerPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        lowerSpinnerPanel.add(new JLabel("Specify Lower Bound: "));
        lowerSpinnerPanel.add(lowerSpinner);
        final JSpinner upperSpinner = HSwingFactory.createJSpinner(null,
                                                                   10,
                                                                   legend.getScale().getUpperBound(),
                                                                   null,
                                                                   null,
                                                                   stepSize);
        final JPanel upperSpinnerPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        upperSpinnerPanel.add(new JLabel("Specify Upper Bound: "));
        upperSpinnerPanel.add(upperSpinner);

        //Set to defaults makes use of the default scale stored as an attribute.
        final JButton setToDefaults = HSwingFactory.createJButton("Set to Defaults", null, new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                lowerSpinner.setValue(_defaultPaintScale.getLowerBound());
                upperSpinner.setValue(_defaultPaintScale.getUpperBound());
            }
        });

        //The panel containing the spinners and button and displayed in an option pane.
        final JPanel panel = new JPanel(new GridBagLayout());
        final GridBagConstraints cons = new GridBagConstraints();
        cons.gridy = 0;
        cons.weighty = 0;
        cons.weightx = 1; //XXX Must be positive to ensure that it fills in the mainPanel width with the table.
        cons.anchor = GridBagConstraints.NORTHWEST;
        cons.fill = GridBagConstraints.BOTH;
        panel.add(upperSpinnerPanel, cons);
        cons.gridy = 1;
        panel.add(lowerSpinnerPanel, cons);
        cons.gridy = 2;
        cons.fill = GridBagConstraints.NONE;
        cons.anchor = GridBagConstraints.CENTER;
        panel.add(setToDefaults, cons);

        final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(ParameterBlockDiagnosticPanel.this),
                                                         panel,
                                                         "Specify Scale Bounds",
                                                         JOptionPane.OK_CANCEL_OPTION);

        //If OK is clicked, do something.
        if(option == JOptionPane.OK_OPTION)
        {
            final double lb = ((Number)lowerSpinner.getValue()).doubleValue();
            final double ub = ((Number)upperSpinner.getValue()).doubleValue();
            if(lb >= ub)
            {
                JOptionPane.showMessageDialog(panel, "Lower bound, " + lb + ", must be less than upper bound, " + ub
                    + ".", "Error Specifying Scale", JOptionPane.ERROR_MESSAGE);
            }
            else
            {
                _paintScaleBoundsOverride = new Point2D.Double(lb, ub);
                final JFreeChart newChart = buildXYBlockChart();

                //Copy over the current axis limits, since a legend click should not affect axis limits!
                final XYPlot oldPlot = (XYPlot)((CombinedDomainXYPlot)getChart().getXYPlot()).getSubplots().get(0);
                final XYPlot newPlot = (XYPlot)((CombinedDomainXYPlot)newChart.getXYPlot()).getSubplots().get(0);
                newPlot.getDomainAxis().setRange(oldPlot.getDomainAxis().getRange());
                newPlot.getRangeAxis(0).setRange(oldPlot.getRangeAxis(0).getRange());

                _currentChartPanel.setChart(newChart);
                _tableModel.setDatasetAndChart(_blockDataset, newChart);
                addAxisChangeListener();
                setupRendererToolTips();
            }
        }
    }

    /**
     * Adds needed change listeners to allow the marked scroll panel to show marks.
     */
    private void addAxisChangeListener()
    {
        final XYPlot plot = (XYPlot)((CombinedDomainXYPlot)_currentChartPanel.getChart().getXYPlot()).getSubplots()
                                                                                                     .get(0);
        new GeneralAxisLimitsChangedNotice.PostingListener(plot.getDomainAxis(), _currentChartPanel, this, this, true);
        new GeneralAxisLimitsChangedNotice.PostingListener(plot.getRangeAxis(0), _currentChartPanel, this, this, true);
    }

    /**
     * Opens an {@link EventDaySummaryPanel} for the specified event and day of year.
     */
    private void openEventDaySummaryPanel(final CanonicalEvent event, final int dayOfYear)
    {

        try
        {
            //Construct the panel to use based on checking the parameter type.
            JPanel panel = null;
            final ParameterId parameterId = ParameterId.valueOf(_blockDataset.getParameterType().getParameterId());
            if(parameterId.isPrecipitation())
            {
                panel = new PrecipitationEventDaySummaryPanel(_blockDataset.getFullParameters(),
                                                              _blockDataset.getSourceParameters(0),
                                                              event,
                                                              dayOfYear);
            }
            else if(parameterId.isTemperature())
            {
                panel = new TemperatureEventDaySummaryPanel(_blockDataset.getFullParameters(),
                                                            _blockDataset.getSourceParameters(0),
                                                            event,
                                                            dayOfYear,
                                                            parameterId);
            }
            else
            {
                throw new IllegalStateException("How did this happen: the parameter " + parameterId
                    + " is neither precip nor temp?");
            }

            String titleText = "Parameter Summary Panel for Source "
                + _blockDataset.getSourceParameters(0).getForecastSource()
                + ", Location "
                + _blockDataset.getFullParameters().getIdentifier().buildStringToDisplayInTree()
                + ", Event "
                + determineDomaainAxisTickLabel(event, _blockDataset.getFullParameters()
                                                                    .getIdentifier()
                                                                    .isPrecipitationDataType());
            if(dayOfYear >= 0)
            {
                titleText += ", and Day " + dayOfYear;
            }
            else
            {
                titleText += ", and All Days";
            }

            //Add it to a frame.
            final JFrame frame = new JFrame(titleText);
            frame.setContentPane(panel);
            frame.setLocationRelativeTo(this);
            frame.setSize(800, 600);
            frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
            frame.setVisible(true);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(this),
                                          "Unable to open parameter summary panel for event ("
                                              + event.getStartLeadPeriod() + ", " + event.getEndLeadPeriod()
                                              + ") and day " + dayOfYear + ":\n" + e.getMessage(),
                                          "Error Opening Summary Panel!",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Opens a popup menu to allow for selecting either one day of year or all days when displaying the event summary
     * panel.
     * 
     * @param displayLocation {@link Point} location for popup display relative to this (panel).
     * @param event {@link CanonicalEvent} corresponding to it.
     * @param day Day of year corresponding to it.
     */
    private void openPopupMenu(final Point displayLocation, final CanonicalEvent event, final int day)
    {
        //Display a popup menu to allow for selecting the mode of event-day summary viewing.
        final JPopupMenu menu = new JPopupMenu();
        final SelfListeningMenuItem item1 = new SelfListeningMenuItem("Display for " + MEFPTools.getDayOfYearStr(day)
            + " Only")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                //For one day...
                openEventDaySummaryPanel(event, day);
                //                menu.setVisible(false);
            }
        };
        item1.setRolloverEnabled(true);
        item1.addRolloverShading();
        final SelfListeningMenuItem item2 = new SelfListeningMenuItem("Display for All Days of Year")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                //For all days...
                openEventDaySummaryPanel(event, -1);
                //                menu.setVisible(false);
            }
        };
        item2.setRolloverEnabled(true);
        item2.addRolloverShading();

        //Create and display the selection menu.
        menu.add(item1);
        menu.add(item2);
        menu.setLocation(displayLocation);
        menu.show(this, displayLocation.x, displayLocation.y);
    }

    /**
     * Opens a point-summary panel for viewing data for the clicked event and day of the year.
     * 
     * @param xyEntity
     */
    private void processChartClick(final ChartMouseEvent mouseEvent, final XYItemEntity xyEntity)
    {
        //Get the entity clicked and determine the event and day of year.
        final int item = xyEntity.getItem();
        int eventNum = -1;
        int dayNum = -1;
        if(xyEntity.getDataset() == _questionableDataset)
        {
            eventNum = _questionableDataset.getX(0, item).intValue();
            dayNum = _questionableDataset.getY(0, item).intValue();
        }
        else
        {
            final int[] indices = _blockDataset.computeIndicesOfItemInBlockSeries(item);
            eventNum = indices[0];
            dayNum = indices[1];
        }

        final CanonicalEvent event = _blockDataset.getEvent(eventNum);
        final int day = _blockDataset.getDay(dayNum);

        openPopupMenu(SwingUtilities.convertPoint(mouseEvent.getTrigger().getComponent(), mouseEvent.getTrigger()
                                                                                                    .getPoint(), this),
                      event,
                      day);
    }

    /**
     * Processes a click on a cell in the table.
     */
    private void processTableCellSelection(final int modelRow, final int modelCol)
    {
        if((modelRow < 0) || (modelCol < 0))
        {
            return;
        }

        final Point pt = MouseInfo.getPointerInfo().getLocation();
        SwingUtilities.convertPointFromScreen(pt, this);
        openPopupMenu(pt, _tableModel.getEventForCol(modelCol), _tableModel.getDayOfYearForRow(modelRow));
    }

    /**
     * @return {@link PaintScale} to use for the block renderer. Based on an ESPADP scale.
     */
    private PaintScale generatePaintScale(double lowerBound, double upperBound)
    {
        if(_paintScaleBoundsOverride != null)
        {
            lowerBound = _paintScaleBoundsOverride.getX();
            upperBound = _paintScaleBoundsOverride.getY();
        }
        final LookupPaintScale paintScale = new LookupPaintScale(lowerBound, upperBound, Color.GRAY)
        {
            @Override
            public Paint getPaint(final double arg0)
            {
                if(Double.isNaN(arg0))
                {
                    return new Color(224, 224, 224);
                }
                return super.getPaint(arg0);
            }
        };
        final Color[] colors = ChartTools.buildESPADPColorPalette(101);
        int colorCount = 0;
        for(double d = 0.00; d <= 1.0d; d += 0.01)
        {
            paintScale.add(lowerBound + d * (upperBound - lowerBound), colors[colorCount]);
            colorCount++;
        }

        if(_paintScaleBoundsOverride == null)
        {
            _defaultPaintScale = paintScale;
        }

        return paintScale;
    }

    /**
     * @return A tool tip that is appropriate for the provided block item number (i.e., the {@link #_blockRenderer} and
     *         {@link #_blockDataset}).
     */
    private final String generateToolTipText(final int itemRelativeToBlocks)
    {
        final int[] indices = _blockDataset.computeIndicesOfItemInBlockSeries(itemRelativeToBlocks);
        final String tip = _blockDataset.getQuestionableMessagesToolTipString(indices[0], indices[1]);

        return tip;
    }

    /**
     * Sets up tool tips to be used by all renderers. This must be done after the chart is put in a panel, because the
     * panel will set tool tip generators itself and these must override.
     */
    private void setupRendererToolTips()
    {
        _blockRenderer.setBaseToolTipGenerator(new XYToolTipGenerator()
        {
            @Override
            public String generateToolTip(final XYDataset dataset, final int series, final int item)
            {
                return generateToolTipText(item);
            }
        });

        //Tool tips are translated into an item for the _blockRenderer and then [QUESTIONABLE] is added.
        _questionableRenderer.setBaseToolTipGenerator(new XYToolTipGenerator()
        {
            @Override
            public String generateToolTip(final XYDataset dataset, final int series, final int item)
            {
                final int eventNum = _questionableDataset.getX(series, item).intValue();
                final int dayNum = _questionableDataset.getY(series, item).intValue();
                return generateToolTipText(dayNum * _blockDataset.getNumberOfEvents() + eventNum);
            }
        });
    }

    /**
     * Sets up {@link #_questionableRenderer} for drawing scatter points indicating questionable parameters.
     */
    private void setupQuestionableRenderer()
    {
        _questionableRenderer = new XYLineAndShapeRenderer();

        final Shape shape = ChartConstants.getShape("x", 0.75d);
        _questionableRenderer.setSeriesPaint(0, Color.BLACK);
        _questionableRenderer.setSeriesShape(0, shape);
        _questionableRenderer.setSeriesFillPaint(0, Color.BLACK);
        _questionableRenderer.setSeriesShapesFilled(0, true);
        _questionableRenderer.setSeriesLinesVisible(0, false);
    }

    /**
     * Sets up {@link #_blockRenderer} for drawing blocks.
     */
    private void setupBlockRenderer()
    {
        //Determine min/max
        final double[] minMax = _blockDataset.computeMinimumAndMaximum();

        //Setup the renderer
        _blockRenderer = new GraphGenXYBlockRenderer();
        _blockRenderer.setPaintScale(generatePaintScale(minMax[0], minMax[1]));
        _blockRenderer.setBlockHeight(0.66);
        _blockRenderer.setBlockWidth(0.66);
        _blockRenderer.setBlockAnchor(RectangleAnchor.CENTER);
    }

    /**
     * @return A {@link SymbolAxis} to be used for the domain of the block plot.
     */
    private NumberAxis setupDomainAxis()
    {
        final String[] axisTicks = new String[_blockDataset.getNumberOfEvents()];
        for(int i = 0; i < axisTicks.length; i++)
        {
            axisTicks[i] = determineDomaainAxisTickLabel(_blockDataset.getEvent(i),
                                                         _blockDataset.getFullParameters()
                                                                      .getIdentifier()
                                                                      .isPrecipitationDataType());
        }

        final SymbolAxis xAxis = new SymbolAxis("Event: Aggregation Window Start Lead Time - End Lead Time", axisTicks);

        xAxis.setLowerMargin(0.0);
        xAxis.setUpperMargin(0.0);
        xAxis.setAxisLinePaint(Color.white);
        xAxis.setTickMarkPaint(Color.white);
        xAxis.setAutoRange(true);
        xAxis.setVerticalTickLabels(true);
        xAxis.setTickLabelFont(AXIS_TICK_FONT);
        xAxis.setLabelFont(AXIS_LABEL_FONT);

        return xAxis;
    }

    /**
     * @return A {@link SymbolAxis} specifying the days for which parameters are estiamted.
     */
    private NumberAxis setupRangeAxis()
    {
        final String[] axisTicks = new String[_blockDataset.getNumberOfDays()];

        //Since this will be done often, I'll use a single baseDate as a starting point and then set its day of year appropriately as needed.
        final Calendar baseDate = HCalendar.convertStringToCalendar("2001-01-01 00:00:00",
                                                                    HCalendar.DEFAULT_DATE_FORMAT);
        for(int i = 0; i < axisTicks.length; i++)
        {
            final int dayOfYear = _blockDataset.getDay(i); //starts counting at one, so we subtract one when adding later.
            final Calendar tickCal = (Calendar)baseDate.clone();
            tickCal.add(Calendar.DAY_OF_YEAR, dayOfYear - 1);
            axisTicks[i] = HCalendar.buildDateStr(tickCal, "MMM dd") + " (day " + dayOfYear + ")";
        }
        final SymbolAxis yAxis = new SymbolAxis("Forecast Start Date (Day of Year)", axisTicks);

        yAxis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());
        yAxis.setLowerMargin(0.0);
        yAxis.setUpperMargin(0.0);
        yAxis.setAxisLinePaint(Color.white);
        yAxis.setTickMarkPaint(Color.white);
        yAxis.setAutoRange(true);
        yAxis.setTickLabelFont(AXIS_TICK_FONT);
        yAxis.setLabelFont(AXIS_LABEL_FONT);

        return yAxis;
    }

    /**
     * @return A {@link PaintScaleLegend} built using a {@link NumberAxis}. This makes use of the inner class
     *         {@link LegendBoundsRecordingPaintScaleLegend}.
     */
    private PaintScaleLegend setupLegend()
    {
        final NumberAxis scaleAxis = new NumberAxis("Scale");
        scaleAxis.setAxisLinePaint(Color.white);
        scaleAxis.setTickMarkPaint(Color.white);
        scaleAxis.setTickLabelFont(LEGEND_TICK_FONT);
        scaleAxis.setLabelFont(AXIS_LABEL_FONT);
        scaleAxis.setRange(_blockRenderer.getPaintScale().getLowerBound(), _blockRenderer.getPaintScale()
                                                                                         .getUpperBound());

        final PaintScaleLegend legend = new LegendBoundsRecordingPaintScaleLegend(_blockRenderer.getPaintScale(),
                                                                                  scaleAxis);
        legend.setAxisLocation(AxisLocation.BOTTOM_OR_LEFT);
        legend.setAxisOffset(5.0);
        legend.setMargin(new RectangleInsets(5, 5, 5, 5));
        legend.setFrame(new BlockBorder(Color.red));
        legend.setPadding(new RectangleInsets(10, 10, 10, 10));
        legend.setStripWidth(10);
        legend.setPosition(RectangleEdge.RIGHT);
        legend.setBackgroundPaint(new Color(120, 120, 180));
        return legend;
    }

    /**
     * @return The title of the plot, which is data type, source, and parameter dependent.
     */
    private String buildPlotTitle()
    {
        //Starting point...
        String title = "";
        if(this._blockDataset.getFullParameters().getIdentifier().isPrecipitationDataType())
        {
            title += _blockDataset.getParameterType().getName();
        }
        else
        {
            title += _blockDataset.getParameterType().getParameterId() + " "
                + _blockDataset.getParameterType().getName();
        }

        //Add unit, if exists
        if(_blockDataset.getParameterType().getUnit() != null)
        {
            title += " [" + _blockDataset.getParameterType().getUnit() + "]";
        }

        //Add source information, which may include one or two
        if((_blockDataset.getNumberOfSources() == 1)
            || (_blockDataset.getSourceParameters(0).getForecastSource() == _blockDataset.getSourceParameters(1)
                                                                                         .getForecastSource()))
        {
            //If only one source or two identical sources, only need to output one source.
            title += "\nSource: " + _blockDataset.getSourceParameters(0).getForecastSource();
        }
        else
        {
            //Otherwise, the second source is the difference source.
            title += "\nSource: " + _blockDataset.getSourceParameters(0).getForecastSource() + " (difference: "
                + _blockDataset.getSourceParameters(1).getForecastSource() + ")";
        }

        //Add location
        title += "    Location: " + _blockDataset.getFullParameters().getIdentifier().buildStringToDisplayInTree();
        return title;
    }

    /**
     * @return Builds the {@link JFreeChart} and returns it calling all necessary setup methods.
     */
    private JFreeChart buildXYBlockChart()
    {
        //For now, building the data set on the fly.
        _questionableDataset = _blockDataset.buildQuestionableDataset();

        final NumberAxis xAxis = setupDomainAxis();
        final NumberAxis yAxis = setupRangeAxis();

        setupBlockRenderer();
        setupQuestionableRenderer();
        final XYPlot plot = new XYPlot();
        plot.setDomainAxis(0, xAxis);
        plot.setRangeAxis(0, yAxis);
        plot.setDataset(0, _questionableDataset);
        plot.setRenderer(0, _questionableRenderer);
        plot.setDataset(1, _blockDataset);
        plot.setRenderer(1, _blockRenderer);

        plot.setBackgroundPaint(Color.lightGray);
        //plot.setDomainGridlinesVisible(false);
        plot.setRangeGridlinePaint(Color.white);
        plot.setAxisOffset(new RectangleInsets(0, 0, 0, 0));
        plot.setOutlinePaint(Color.blue);

        final CombinedDomainXYPlot combinedPlot = new CombinedDomainXYPlot(xAxis);
        combinedPlot.add(plot);

        //Plot title.
        final JFreeChart chart = new JFreeChart(buildPlotTitle(), combinedPlot);
        chart.getTitle().setFont(TITLE_FONT);

        //Legend...
        chart.removeLegend();
        chart.addSubtitle(setupLegend());

        chart.setBackgroundPaint(new Color(180, 180, 250));

        return chart;
    }

    /**
     * Special {@link PaintScaleLegend} that records the bounds on the legend when its
     * {@link #draw(Graphics2D, Rectangle2D, Object)} method is called.
     * 
     * @author hankherr
     */
    private class LegendBoundsRecordingPaintScaleLegend extends PaintScaleLegend
    {
        private Rectangle2D _bounds = null;

        public LegendBoundsRecordingPaintScaleLegend(final PaintScale scale, final ValueAxis axis)
        {
            super(scale, axis);
        }

        @Override
        public Object draw(final Graphics2D g2, final Rectangle2D area, final Object params)
        {
            //The _bounds are computed by transforming the area based on g2 to determine the frame that contains the legend.
            final double[] cornerPoint = new double[]{area.getX(), area.getY()};
            final double[] otherCorner = new double[]{area.getX() + area.getWidth(), area.getY() + area.getHeight()};
            final double[] newCornerPoint = new double[2];
            final double[] newOtherCorner = new double[2];
            g2.getTransform().transform(cornerPoint, 0, newCornerPoint, 0, 1);
            g2.getTransform().transform(otherCorner, 0, newOtherCorner, 0, 1);
            _bounds = new Rectangle2D.Double();
            _bounds.setFrame(newCornerPoint[0],
                             newCornerPoint[1],
                             newOtherCorner[0] - newCornerPoint[0],
                             newOtherCorner[1] - newCornerPoint[1]);

            return super.draw(g2, area, params);
        }

        @Override
        public Rectangle2D getBounds()
        {
            return _bounds;
        }
    }

    @Override
    public void post(final Object event)
    {
        getCentralBus().post(event);
    }

    @Override
    @Subscribe
    public void reactToTableCellSelection(final GeneralTableCellSelectedNotice evt)
    {
        processTableCellSelection(evt.getModelRow(), evt.getModelCol());
    }

    @Override
    @Subscribe
    public void reactToGeneralAxisLimitsChanged(final GeneralAxisLimitsChangedNotice evt)
    {
        final XYPlot plot = (XYPlot)((CombinedDomainXYPlot)_currentChartPanel.getChart().getXYPlot()).getSubplots()
                                                                                                     .get(0);

        //Attempt to scroll the table to fit the entire zoomed region.
        //Identify the column to which to scroll to as the upper bound of the region and do it.
        //The min/max stuff ensures the column is valid.
        //The first scroll resets the scroll to the far right so that the scroll to col is consistent regardless
        //of where the starting point is in the scroll pane.
        if(evt.getAxis() == plot.getDomainAxis())
        {
            final double lb = evt.getAxis().getRange().getLowerBound();
            int col = (int)(Math.floor(lb)) + 1;//+1 accounts for date column
            col = Math.min(col, _chartTable.getColumnCount() - 1);
            col = Math.max(col, 0);
            _chartTable.scrollColumnToVisible(_chartTable.getColumnCount());
            _chartTable.scrollColumnToVisible(col);
        }
        //Identify the row to scroll to as the lower bound of the region, 
        //for the furthest south row in the table, and do it.
        //The min/max stuff ensures the row is valid.
        //The first scroll resets the scroll to the  bottom so that the scroll to row is consistent regardless
        //of where the starting point is in the scroll pane.
        else
        {
            final double ub = evt.getAxis().getRange().getUpperBound();
            int row = (int)(_chartTable.getRowCount() - ub) - 1;
            row = Math.min(row, _chartTable.getRowCount() - 1);
            row = Math.max(row, 0);
            _chartTable.scrollRowToVisible(_chartTable.getRowCount() - 1);
            _chartTable.scrollRowToVisible(row);
        }

        _scrollPanel.recomputeMarks();
        _scrollPanel.repaint();
    }

    /**
     * This method may be called by other components in order to display text that matches that shown for an event on
     * the domain axis of block plot displayed within a {@link ParameterBlockDiagnosticPanel}. This calls
     * {@link MEFPTools#getCanonicalEventPeriodStr(CanonicalEvent, boolean, boolean, boolean, DecimalFormat)}.
     * 
     * @param event The {@link CanonicalEvent} for which to generate a domain axis tick label.
     * @param isPrecipitationDataType Boolean indicating if the event is a precipitation event (true) or temperature
     *            event (false).
     * @return A domain axis tick label for the given event, with the provided time step and decimal format.
     */
    public static String determineDomaainAxisTickLabel(final CanonicalEvent event, final boolean isPrecipitationDataType)
    {
        return MEFPTools.getCanonicalEventPeriodStr(event,
                                                    isPrecipitationDataType,
                                                    false,
                                                    false,
                                                    DOMAIN_TICK_LABEL_NUMBER_FORMATTER)
            + " - "
            + MEFPTools.getCanonicalEventPeriodStr(event,
                                                   isPrecipitationDataType,
                                                   true,
                                                   true,
                                                   DOMAIN_TICK_LABEL_NUMBER_FORMATTER);
    }
}
