package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.Point;
import java.text.DecimalFormat;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.model.ModelParameterType;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.NumberTools;

import org.jfree.data.xy.AbstractXYDataset;
import org.jfree.data.xy.AbstractXYZDataset;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYZDataset;

import com.google.common.collect.Lists;

/**
 * An {@link XYZDataset} for use in displaying block data in a JFreeChart.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ParameterBlockDiagnosticXYZDataset extends AbstractXYZDataset
{
    private final MEFPSourceModelParameters[] _sourceParameters;
    private final MEFPFullModelParameters _fullParameters;
    private final ModelParameterType _parameterType;
    private double[][] _eventDayParameterValues;
    private List<Integer> _daysWithData;
    private final List<CanonicalEvent> _eventsWithData;
    private final DecimalFormat _formatter = new DecimalFormat("0.####");

    /**
     * @param fullParameters Full model parameters used to acquire questionable results.
     * @param sourceParameters The source parameters containing data to record here.
     * @param displayedEvents Events for which to store data.
     * @param parametersToDisplay Either one or two parameter types. If two are provided, the displayed data is the
     *            difference between the two: the first - the second.
     */
    public ParameterBlockDiagnosticXYZDataset(final MEFPFullModelParameters fullParameters,
                                              final MEFPForecastSource[] sources,
                                              final Collection<CanonicalEvent> displayedEvents,
                                              final ModelParameterType... parametersToDisplay)
    {
        if(sources.length != parametersToDisplay.length)
        {
            throw new IllegalArgumentException("The number of provided sources does not match the number of model parameter types passed in.");
        }
        if((sources.length != 1) && (sources.length != 2))
        {
            throw new IllegalArgumentException("The number of sources provided must be either 1 or 2, but was "
                + sources.length);
        }

        //Pull the source parameters.
        _sourceParameters = new MEFPSourceModelParameters[sources.length];
        for(int i = 0; i < sources.length; i++)
        {
            _sourceParameters[i] = fullParameters.getSourceModelParameters(sources[i]);
            if(!_sourceParameters[i].wereParametersEstimatedForSource())
            {
                throw new IllegalArgumentException("For the parameters provided, the source " + sources[i].getName()
                    + " does not include estimated parameters.");
            }
        }

        _parameterType = constructDisplayedParameterType(parametersToDisplay);
        _fullParameters = fullParameters;
        if((displayedEvents == null) || (displayedEvents.isEmpty()))
        {
            _eventsWithData = Lists.newArrayList(_sourceParameters[0].getComputedEvents());
        }
        else
        {
            _eventsWithData = Lists.newArrayList(displayedEvents);
        }

        populateValues(parametersToDisplay);
    }

    /**
     * @param types One or two types used to build the block data.
     * @return A {@link ModelParameterType} summarizing the block data.
     */
    private ModelParameterType constructDisplayedParameterType(final ModelParameterType[] types)
    {
        //Check for issues.
        if(types.length == 1)
        {
            return types[0];
        }
        if(types.length != 2)
        {
            throw new IllegalArgumentException("Either one or two types must be provided, but " + types.length
                + " were provided.");
        }
        if(types[1] == null)
        {
            System.err.println("####>> types[1] is NULL %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%");
        }

        //Check compatible units.
        if(types[1] != null)
        {
            if(!GeneralTools.checkForFullEqualityOfObjects(types[0].getUnit(), types[1].getUnit()))
            {
                throw new IllegalArgumentException("Two provided model parameter types must have the same unit: "
                    + types[0].getUnit() + ", " + types[1].getUnit());
            }
        }

        //If two types are provided, then a difference will be computed.  Define an appropriate model parameter type.
        //Used for charting purposes only.
        return new ModelParameterType()
        {
            @Override
            public boolean getSquarePlottedLimits()
            {
                return true; //difference should be plotted so that 0 is the mid point of values.
            }

            @Override
            public Class<? extends Number> getValueTypeClass()
            {
                return types[0].getValueTypeClass();
            }

            @Override
            public String getName()
            {
                if(!types[0].equals(types[1]))
                {
                    return types[0].getName() + " - " + types[1].getName();
                }
                return "Difference Across Sources for " + types[0].getName();
            }

            @Override
            public double computeValue(final double[] forecasts, final double[] observations)
            {
                return Double.NaN;
            }

            @Override
            public String getUnit()
            {
                return types[0].getUnit();
            }

            @Override
            public String getParameterId()
            {
                return types[0].getParameterId();
            }

        };
    }

    /**
     * Populate the {@link #_eventDayParameterValues} attribute.
     * 
     * @param sourceParms The source parameters from which to draw the values.
     * @param parameterType The parameter to display.
     */
    private void populateValues(final ModelParameterType[] types)
    {
        _daysWithData = _fullParameters.generateDaysOfTheYearForWhichToEstimateParameters();
        _eventDayParameterValues = new double[_eventsWithData.size()][_daysWithData.size()];

        for(int eventNum = 0; eventNum < _eventDayParameterValues.length; eventNum++)
        {
            final CanonicalEvent event = _eventsWithData.get(eventNum);
            int dayIndex = 0;
            for(final int day: _daysWithData)
            {
                double value = 0;
                if(types.length == 1)
                {
                    value = _sourceParameters[0].getParameterValues(types[0]).getValue(day, event);
                }
                else
                {
                    if(!_sourceParameters[1].getParameterValues(types[1]).wereParametersComputedForEvent(event))
                    {
                        value = Double.NaN;
                    }
                    else
                    {
                        value = _sourceParameters[0].getParameterValues(types[0]).getValue(day, event)
                            - _sourceParameters[1].getParameterValues(types[1]).getValue(day, event);
                    }
                }
                _eventDayParameterValues[eventNum][dayIndex] = value;
                dayIndex++;
            }
        }
    }

    /**
     * When plotted, the items in the chart are tracked in one long list. This method translates the index in that one
     * long list to row/column indices within the {@link #_eventDayParameterValues} double array.
     * 
     * @param item Overall index of the item to look for.
     * @return Array of two indices: the event index within {@link MEFPSourceModelParameters#getComputedEvents()} and
     *         the day of the year index within {@link #_daysWithData}.
     */
    public int[] computeIndicesOfItemInBlockSeries(final int item)
    {
        final int[] results = new int[2];
        results[0] = item % _eventDayParameterValues.length;
        results[1] = item / _eventDayParameterValues.length;
        return results;
    }

    public String formatNumber(final Number number)
    {
        return NumberTools.formatNumber(_formatter, number.doubleValue());
    }

    /**
     * @return The minimum and maximum of the block data to display, which is used to setup the paint scale. The minimum
     *         is at index 0, and max at 1. Values of NaN imply that no non-missing data was found.
     */
    public double[] computeMinimumAndMaximum()
    {
        final double[] results = new double[2];
        results[0] = Double.MAX_VALUE;
        results[1] = Double.MIN_VALUE;
        for(int eventNum = 0; eventNum < _eventDayParameterValues.length; eventNum++)
        {
            for(int dayNum = 0; dayNum < _eventDayParameterValues[eventNum].length; dayNum++)
            {
                if(!Double.isNaN(_eventDayParameterValues[eventNum][dayNum]))
                {
                    if(_eventDayParameterValues[eventNum][dayNum] < results[0])
                    {
                        results[0] = _eventDayParameterValues[eventNum][dayNum];
                    }
                    if(_eventDayParameterValues[eventNum][dayNum] > results[1])
                    {
                        results[1] = _eventDayParameterValues[eventNum][dayNum];
                    }
                }
            }
        }

        //Put in parameter type specific min and max as appropriate.
        if(!Double.isNaN(_parameterType.getMinimumDisplayValue()))
        {
            results[0] = _parameterType.getMinimumDisplayValue();
        }
        if(!Double.isNaN(_parameterType.getMaximumDisplayValue()))
        {
            results[1] = _parameterType.getMaximumDisplayValue();
        }

        //Check results to make sure everything has a usable value.
        if((results[0] == Double.MAX_VALUE) && (results[1] == Double.MIN_VALUE))
        {
            results[0] = 0.0D;
            results[1] = 1.0D;
        }
        else if(results[0] == Double.MAX_VALUE)
        {
            results[0] = results[1] - 1.0D;
        }
        else if(results[1] == Double.MIN_VALUE)
        {
            results[1] = results[0] + 1.0d;
        }

        //Check for sqaured plotting limits
        if(_parameterType.getSquarePlottedLimits())
        {
            results[0] = -1.0 * Math.max(Math.abs(results[0]), Math.abs(results[1]));
            results[1] = Math.max(Math.abs(results[0]), Math.abs(results[1]));
        }
        return results;
    }

    //_eventDayParameterValues

    public List<Integer> getDaysWithData()
    {
        return _daysWithData;
    }

    public List<CanonicalEvent> getEventsWithData()
    {
        return _eventsWithData;
    }

    public double getParameterValue(final int canonicalEventIndex, final int dayOfYearIndex)
    {
        return _eventDayParameterValues[canonicalEventIndex][dayOfYearIndex];
    }

    public CanonicalEvent getEvent(final int eventIndex)
    {
        return _eventsWithData.get(eventIndex);
    }

    public int getNumberOfEvents()
    {
        return _eventsWithData.size();
    }

    public int getNumberOfDays()
    {
        return _daysWithData.size();
    }

    public int getDay(final int dayIndex)
    {
        return _daysWithData.get(dayIndex);
    }

    public ModelParameterType getParameterType()
    {
        return _parameterType;
    }

    public int getNumberOfSources()
    {
        return _sourceParameters.length;
    }

    public MEFPSourceModelParameters getSourceParameters(final int index)
    {
        return _sourceParameters[index];
    }

    public MEFPFullModelParameters getFullParameters()
    {
        return _fullParameters;
    }

    /**
     * @return {@link Set} of message {@link String}s acquired from {@link #_fullParameters} questionable log returned
     *         by {@link MEFPFullModelParameters#getQuestionableParameterLog()}. If the set is empty, then no
     *         questionable messages were found (i.e., the parameters are not questioanble).
     */
    public Collection<String> getQuestionableMessages(final int eventNum, final int dayNum)
    {
        final List<String> messages = Lists.newArrayList(_fullParameters.getQuestionableParameterLog()
                                                                        .getEntries(_sourceParameters[0].getForecastSource(),
                                                                                    _eventsWithData.get(eventNum),
                                                                                    _daysWithData.get(dayNum)));
        if(_sourceParameters.length == 2)
        {
            final Set<String> otherMessages = _fullParameters.getQuestionableParameterLog()
                                                             .getEntries(_sourceParameters[1].getForecastSource(),
                                                                         _eventsWithData.get(eventNum),
                                                                         _daysWithData.get(dayNum));
            if(!otherMessages.isEmpty())
            {
                messages.add("");
                messages.add("---------- Other Source: " + _sourceParameters[1].getForecastSource().getName());
                messages.addAll(otherMessages);
            }
        }
        return messages;
    }

    /**
     * @param eventNum Index of {@link CanonicalEvent} within {@link #_eventsWithData}.
     * @param dayNum Index of day within {@link #_daysWithData}.
     * @return A standard tool tip string to be used for the parameter value at the provided event and day.
     */
    public String getQuestionableMessagesToolTipString(final int eventNum, final int dayNum)
    {
        final CanonicalEvent event = _eventsWithData.get(eventNum);
        final int dayOfYear = _daysWithData.get(dayNum);
        String tip = "<html>Event: (" + event.getStartLeadPeriod() + ", " + event.getEndLeadPeriod() + "), day "
            + dayOfYear + " = " + formatNumber(_eventDayParameterValues[eventNum][dayNum]);
        final Collection<String> messages = getQuestionableMessages(eventNum, dayNum);
        if((messages != null) && (!messages.isEmpty()))
        {

            tip += " [QUESTIONABLE]:";

            //At this point, it should NOT be null... but just in case...
            if(messages != null)
            {
                for(final String message: messages)
                {
                    tip += "<br>" + message;
                }
            }
        }
        tip += "</html>";

        return tip;
    }

    /**
     * @param eventNum The event number.
     * @param dayNum The day number.
     * @return True if {@link #getQuestionableMessages(int, int)} returns an non-empty, non-null set, indicating that
     *         the parameters are questionable.
     */
    public boolean isQuestionable(final int eventNum, final int dayNum)
    {
        final Collection<String> messages = getQuestionableMessages(eventNum, dayNum);
        return (messages != null) && (!messages.isEmpty());
    }

    /**
     * Builds the {@link #_questionableDataset} which contains values that are to be marked as questionable.
     */
    public XYDataset buildQuestionableDataset()
    {
        final List<Point> questionablePoints = Lists.newArrayList();
        for(int eventNum = 0; eventNum < _eventDayParameterValues.length; eventNum++)
        {
            for(int dayNum = 0; dayNum < _eventDayParameterValues[eventNum].length; dayNum++)
            {
                if(isQuestionable(eventNum, dayNum))
                {
                    questionablePoints.add(new Point(eventNum, dayNum)); //TODO Need to check for questionability at some source location
                }
            }
        }

        final XYDataset questionableDataset = new AbstractXYDataset()
        {
            @Override
            public int getItemCount(final int series)
            {
                return questionablePoints.size();
            }

            @Override
            public Number getX(final int series, final int item)
            {
                return questionablePoints.get(item).x;
            }

            @Override
            public Number getY(final int series, final int item)
            {
                return questionablePoints.get(item).y;
            }

            @Override
            public int getSeriesCount()
            {
                return 1;
            }

            @Override
            public Comparable getSeriesKey(final int series)
            {
                return "Questionable Points";
            }
        };
        return questionableDataset;
    }

    @Override
    public Number getZ(final int series, final int item)
    {
        final int[] indices = computeIndicesOfItemInBlockSeries(item);
        return _eventDayParameterValues[indices[0]][indices[1]];
    }

    @Override
    public int getItemCount(final int series)
    {
        return _eventDayParameterValues.length * _daysWithData.size();
    }

    @Override
    public Number getX(final int series, final int item)
    {
        final int[] indices = computeIndicesOfItemInBlockSeries(item);
        return indices[0];
    }

    @Override
    public Number getY(final int series, final int item)
    {
        final int[] indices = computeIndicesOfItemInBlockSeries(item);
        return indices[1];
    }

    @Override
    public int getSeriesCount()
    {
        return 1;
    }

    @Override
    public Comparable getSeriesKey(final int series)
    {
        return "PARMS";
    }
}
