package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.Color;
import java.awt.Component;

import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;

import ohd.hseb.charter.jfreechartoverride.GraphGenXYBlockRenderer;
import ohd.hseb.charter.panel.AbstractChartEngineTableModel;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.gui.jtable.models.RowColumnMarkingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowHeaderTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.TableAwareTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.ToolTipTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.WrapRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.ForwardingTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;

import org.jdesktop.swingx.renderer.JRendererLabel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.PaintScale;

/**
 * This must be used in conjunction with a {@link ParameterBlockDiagnosticPanel} that shares the same
 * {@link ParameterBlockDiagnosticXYZDataset}.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class ParameterBlockXYZDatasetTableModel extends AbstractTableModel implements RowColumnMarkingTableModel,
WrapRendererTableModel, TableAwareTableModel, ToolTipTableModel, RowHeaderTableModel
{
    public ParameterBlockDiagnosticXYZDataset _dataset;
    public JFreeChart _chart = null;
    public JTable _table = null;

    /**
     * @param dataset The {@link ParameterBlockDiagnosticXYZDataset} to which this is linked.
     */
    public ParameterBlockXYZDatasetTableModel(final ParameterBlockDiagnosticXYZDataset dataset)
    {
        _dataset = dataset;
    }

    /**
     * Records the provided data set and chart and then calls {@link #fireTableStructureChanged()}.
     */
    public void setDatasetAndChart(final ParameterBlockDiagnosticXYZDataset dataset, final JFreeChart chart)
    {
        _dataset = dataset;
        _chart = chart;
        fireTableStructureChanged();
    }

    public JTable getTable()
    {
        return _table;
    }

    /**
     * @return The {@link PaintScale} put in place within {@link ParameterBlockDiagnosticPanel}. It is bound in the
     *         first subplot of the {@link CombinedDomainXYPlot} returned by {@link JFreeChart#getXYPlot()} for
     *         {@link #_chart}.
     */
    public PaintScale getPaintScale()
    {
        final XYPlot plot = (XYPlot)((CombinedDomainXYPlot)_chart.getXYPlot()).getSubplots().get(0);
        final GraphGenXYBlockRenderer renderer = (GraphGenXYBlockRenderer)plot.getRenderer(1);
        return renderer.getPaintScale();
    }

    public CanonicalEvent getEventForCol(final int col)
    {
        if(col == 0)
        {
            throw new IllegalArgumentException("Column 0 is a label column; it has no canonical event.");
        }
        return _dataset.getEvent(col - 1);
    }

    public int getDayOfYearForRow(final int row)
    {
        return _dataset.getDay(getRowCount() - row - 1);
    }

    /**
     * Row and column must be model values!
     * 
     * @return The value turned into a {@link String} by {@link #getValueAt(int, int)}.
     */
    public Object getRawValueAt(final int rowIndex, final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return getDayOfYearForRow(rowIndex);
        }
        else
        {
            return _dataset.getParameterValue(columnIndex - 1, getRowCount() - rowIndex - 1);
        }
    }

    @Override
    public Class<?> getColumnClass(final int columnIndex)
    {
        if(columnIndex == 0)
        {
            return Integer.class;
        }
        return Double.class;
    }

    @Override
    public String getColumnName(final int column)
    {
        if(column == 0)
        {
            return "Day of Year";
        }
        return CanonicalEvent.createXMLAttributeString(getEventForCol(column));
    }

    @Override
    public int getRowCount()
    {
        return _dataset.getDaysWithData().size();
    }

    @Override
    public int getColumnCount()
    {
        return _dataset.getEventsWithData().size() + 1;
    }

    @Override
    public Object getValueAt(final int rowIndex, final int columnIndex)
    {
        final Object rawValue = getRawValueAt(rowIndex, columnIndex);
        return rawValue;
    }

    /**
     * @return Returns the standard {@link AbstractChartEngineTableModel#WITHIN_LIMITS_MARK_COLOR} if the event is being
     *         displayed within the domain axis of the chart.
     */
    @Override
    public Dyad<Color, Integer> getRowMarkColor(final int modelRow)
    {
        //Check to see if the row is within the domain axis limits.  Return standard 
        final XYPlot plot = (XYPlot)((CombinedDomainXYPlot)_chart.getXYPlot()).getSubplots().get(0);
        if(plot.getRangeAxis(0).getRange().contains(getRowCount() - modelRow - 1))
        {
            return new Dyad<Color, Integer>(AbstractChartEngineTableModel.WITHIN_LIMITS_MARK_COLOR, 0);
        }
        return null;
    }

    /**
     * @return Returns the standard {@link AbstractChartEngineTableModel#WITHIN_LIMITS_MARK_COLOR} if the day is being
     *         displayed within the range axis of the chart.
     */
    @Override
    public Dyad<Color, Integer> getColumnMarkColor(final int col)
    {
        final XYPlot plot = (XYPlot)((CombinedDomainXYPlot)_chart.getXYPlot()).getSubplots().get(0);

        //One is subtracted because the first column is a label column.
        if(plot.getDomainAxis().getRange().contains(col - 1))
        {
            return new Dyad<Color, Integer>(AbstractChartEngineTableModel.WITHIN_LIMITS_MARK_COLOR, 0);
        }
        return null;
    }

    @Override
    public void addTable(final JTable table)
    {
        _table = table;
    }

    @Override
    public void removeTable(final JTable table)
    {
        _table = null;
    }

    @Override
    public boolean applyAfterAllOtherRenderers()
    {
        return false;
    }

    @Override
    public String getColumnHeaderToolTip(final int visibleColIndex)
    {
        final int modelColIndex = getTable().convertColumnIndexToModel(visibleColIndex);
        if(modelColIndex == 0)
        {
            return "Days of the year";
        }
        final CanonicalEvent event = getEventForCol(modelColIndex);
        return "Event starting at time period " + event.getStartLeadPeriod() + " and ending at period "
            + event.getEndLeadPeriod() + " (duration of " + event.getDuration() + " periods)";
    }

    @Override
    public String getCellToolTip(final int rowIndex, final int visibleColIndex)
    {
        final int modelColIndex = getTable().convertColumnIndexToModel(visibleColIndex);
        if(modelColIndex == 0)
        {
            return "Day of the year: " + getRawValueAt(rowIndex, modelColIndex);
        }
        return _dataset.getQuestionableMessagesToolTipString(modelColIndex - 1, getRowCount() - rowIndex - 1);
    }

    @Override
    public TableCellRenderer wrapRenderer(final TableCellRenderer baseRenderer)
    {
        return new DefaultChartEngineCellRenderer(baseRenderer);
    }

    /**
     * Renderer fades out cells outside the zoomed in region of chart. It also sets the background color using the
     * {@link PaintScale} returned by {@link #getPaintScale()}.
     * 
     * @author hankherr
     */
    protected class DefaultChartEngineCellRenderer extends ForwardingTableCellRenderer
    {
        public DefaultChartEngineCellRenderer(final TableCellRenderer baseRenderer)
        {
            super(baseRenderer);
        }

        @Override
        public Component getTableCellRendererComponent(final JTable table,
                                                       final Object value,
                                                       final boolean isSelected,
                                                       final boolean hasFocus,
                                                       final int row,
                                                       final int column)
        {
            //This is starting with a default table cell renderer; not the renderers used in super classes.
            final Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            final int modelRow = getTable().convertRowIndexToModel(row);
            final int modelCol = getTable().convertColumnIndexToModel(column);

            //Add questionmark to the label if the data is questionable.
            if((modelCol > 0) && (_dataset.isQuestionable(modelCol - 1, getRowCount() - modelRow - 1)))
            {
                if(c.getClass().isAssignableFrom(JRendererLabel.class))
                {
                    ((JRendererLabel)c).setText(((JRendererLabel)c).getText() + " (?)");
                }
            }

            //For the first column, mix the background color with light gray if the row is selected.
            if(column == 0)
            {
                //IF selected, mix the color with light gray.
                if(getTable().isRowSelected(row))
                {
                    c.setBackground(ColorTools.mixColors(c.getBackground(), Color.LIGHT_GRAY));
                }
            }

            //For all other columns... 
            if(column > 0)
            {
                //If the cell is within the axis limits, then make the foreground bold.
                //Otherwise, use a plain foreground with a color mixed with dark gray, lightening it a bit from black.
                if((getRowMarkColor(modelRow) == null) || (getColumnMarkColor(modelCol) == null))
                {
                    //Only a slight fade.
                    c.setForeground(ColorTools.mixColors(c.getForeground(), Color.GRAY));
                }
                else
                {
                    //Only make the text bold if the cell is a JRendererLabel.  This should always be the case,
                    //but I still want to do an if check first to be safe.
                    if(c.getClass().isAssignableFrom(JRendererLabel.class))
                    {
                        ((JRendererLabel)c).setText("<html><b/>" + ((JRendererLabel)c).getText() + "</html>");
                    }
                }

                //If the cell is not selected, then use a mixing background for cell for which either the column
                //or row is the same as the selected cell (i.e., table cross hairs).  If the cell is not within
                //the chart's axis limits, fade the background to make it obvious.
                if(!isSelected)
                {
                    //If selected, mix the color with light gray.
                    Color color = (Color)getPaintScale().getPaint((Double)getRawValueAt(modelRow, modelCol));
                    if((getTable().isColumnSelected(column)) || (getTable().isRowSelected(row)))
                    {
                        color = ColorTools.mixColors(color, Color.LIGHT_GRAY);
                    }

                    c.setBackground(new Color(color.getRed(), color.getGreen(), color.getBlue()));

                    //If not in the axis limits, fade the background color toward white.
                    if((getRowMarkColor(modelRow) == null) || (getColumnMarkColor(modelCol) == null))
                    {
                        c.setBackground(ColorTools.mixColors(c.getBackground(), Color.WHITE));
                    }
                }
            }
            return c;
        }
    }

    @Override
    public int getRowHeaderColumn()
    {
        return 0;
    }
}
