package ohd.hseb.hefs.mefp.pe.estimation.diag;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.RepaintManager;

import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.panel.CombinedDomainChartNavigationPanel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.GlobalDialogParent;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tools.ParameterId;

import org.jfree.chart.JFreeChart;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * Displays a block diagram on the left and a selection panel on the right. It allows for selecting sources, parameters,
 * and events for which to display the summary block panel. In addition, a navigation panel is provided in the upper
 * right for allowing the user to more readily see the zoomed in area.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
@GlobalDialogParent
@HelpFile("PDFHELP:helpManual.pdf#ParameterDiagnosticsPanelRef")
public class ParameterQCDiagnosticPanel extends ParameterEstimatorDiagnosticPanel implements
DisplayedParameterSelectedNotice.Subscriber, CanonicalEventsSelectedNotice.Subscriber,
ForecastSourceSelectedNotice.Subscriber
{
    private final MEFPFullModelParameters _fullParameters;
    private final JSplitPane _splitPane;
    private final ForecastSourceSelectionPanel _selectSourcePanel;
    private final DisplayedParameterSelectionPanel _selectParmPanel;
    private final CanonicalEventSelectionPanel _selectEventPanel;

    /**
     * Will contain the navigation panel.
     */
    private final JPanel _navPanelHolder;

    /**
     * The navigation component.
     */
    private CombinedDomainChartNavigationPanel _navPanel;

    /**
     * Used to prevent reacting to parameter and event selections when making changes for a new source selection.
     * Selection changes will be processed all at once.
     */
    private boolean _currentProcessingSelectedSource = false;

    /**
     * The currently displayed block panel.
     */
    private ParameterBlockDiagnosticPanel _blockPanel = null;

    /**
     * Used to communicate events where the user requests that the diagnostic block plot be created for many locations.
     */
    private final EventBus _eventBus = new EventBus();

    /**
     * @param fullParameters The complete set of parameters associated with this panel.
     */
    public ParameterQCDiagnosticPanel(final MEFPFullModelParameters fullParameters)
    {
        _fullParameters = fullParameters;

        _selectSourcePanel = new ForecastSourceSelectionPanel(fullParameters);
        _selectParmPanel = new DisplayedParameterSelectionPanel(fullParameters, getSelectedSourceModelParameters());
        _selectEventPanel = new CanonicalEventSelectionPanel(fullParameters,
                                                             getSelectedSourceModelParameters().getForecastSource());
        _selectEventPanel.setMinimumSize(new Dimension(200, 100));
        _selectEventPanel.setPreferredSize(new Dimension(250, 100));

        _navPanelHolder = new JPanel(new BorderLayout());
        _navPanelHolder.setPreferredSize(new Dimension(225, 160));

        //Diag plot button
        final JButton createPlotsButton = new JButton("Create For Selected Locations");
        createPlotsButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                _eventBus.post(new CreateBlockPlotForSelectedLocationsNotice(ParameterQCDiagnosticPanel.this));
            }
        });

        final JPanel rightPanel = new JPanel(new GridBagLayout());
        final GridBagConstraints cons = new GridBagConstraints();
        cons.gridy = 0;
        cons.weighty = 0;
        cons.weightx = 1; //XXX Must be positive to ensure that it fills in the mainPanel width with the table.
        cons.anchor = GridBagConstraints.NORTHWEST;
        cons.fill = GridBagConstraints.BOTH;
        rightPanel.add(_navPanelHolder, cons);
        cons.gridy = 1;
        rightPanel.add(_selectSourcePanel, cons);
        cons.gridy = 2;
        rightPanel.add(_selectParmPanel, cons);
        cons.gridy = 3;
        cons.weighty = 1;
        rightPanel.add(_selectEventPanel, cons);
        cons.gridy = 4;
        cons.weighty = 0;
        cons.fill = GridBagConstraints.NONE;
        cons.anchor = GridBagConstraints.CENTER;
        rightPanel.add(createPlotsButton, cons);

        _splitPane = HSwingFactory.createJSPlitPane(JSplitPane.HORIZONTAL_SPLIT, new JPanel(), rightPanel, false);

        rightPanel.setMinimumSize(new Dimension(275, 200));

        this.setLayout(new BorderLayout());
        this.add(_splitPane, BorderLayout.CENTER);

        _splitPane.setResizeWeight(1);

        _selectParmPanel.registerWithBus(this);
        _selectEventPanel.registerWithBus(this);
        _selectSourcePanel.registerWithBus(this);

        _selectParmPanel.selectInitialParameter();
    }

    /**
     * @return {@link MEFPSourceModelParameters} corresponding to the selected source specified by
     *         {@link #_selectSourcePanel}.
     */
    private MEFPSourceModelParameters getSelectedSourceModelParameters()
    {
        return _fullParameters.getSourceModelParameters(_selectSourcePanel.getSelectedSource());
    }

    /**
     * Determine the selected sources. The first one is always the primary one, or that associated with the
     * _selectedSourcePanel. The second one only exists if the number of parameter ModelParameterTypes is two and is the
     * secondary source maintained by the _selectParmPanel. If it is null, it is set equal to the primary source.
     * 
     * @return Array specifying the selected sources.
     */
    private MEFPForecastSource[] determineSelectedSources()
    {
        final MEFPForecastSource[] selectedSources = new MEFPForecastSource[_selectParmPanel.getSelectedParameters().length];
        selectedSources[0] = _selectSourcePanel.getSelectedSource();
        if(selectedSources.length == 2)
        {
            selectedSources[1] = _selectParmPanel.getSecondarySource();
            if(selectedSources[1] == null)
            {
                selectedSources[1] = selectedSources[0];
            }
        }
        return selectedSources;
    }

    /**
     * Uses the current selections to build an image for the provided parameters. The image will have the same size as
     * the size of the panel currently displaying the block chart.
     * 
     * @param parameters The parameters for which to build the block plot.
     * @param imageFileToCreate The image file to create.
     * @throws Exception If a problem occurs, such as not being able to write to a file.
     */
    public void createDiagnosticChart(final MEFPFullModelParameters parameters, final File imageFileToCreate) throws Exception
    {
        //Determine the selected sources.
        final MEFPForecastSource[] selectedSources = determineSelectedSources();
        final ParameterBlockDiagnosticPanel blockPanel = new ParameterBlockDiagnosticPanel(parameters,
                                                                                           selectedSources,
                                                                                           _selectEventPanel.getSelectedEvents(),
                                                                                           _selectParmPanel.getSelectedParameters(),
                                                                                           false);
        final JFreeChart chart = blockPanel.getChart();
        final Dimension size = _blockPanel.getSize();

        ChartTools.generateOutputImageFile(imageFileToCreate,
                                                       chart,
                                                       (int)size.getWidth(),
                                                       (int)size.getHeight());
    }

    /**
     * Updates the block panel display to match the selected.
     */
    private void updateDisplay()
    {
        //Determine the selected sources.
        final MEFPForecastSource[] selectedSources = determineSelectedSources();

        //Construct the block panel if necessary.
        if(_blockPanel == null)
        {
            _blockPanel = new ParameterBlockDiagnosticPanel(_fullParameters,
                                                            selectedSources,
                                                            _selectEventPanel.getSelectedEvents(),
                                                            _selectParmPanel.getSelectedParameters());

            //Create a navigation panel and add it to the _navPanel container.
            _navPanel = new CombinedDomainChartNavigationPanel(_blockPanel.getChartPanel(),
                                                               _navPanelHolder.getPreferredSize());

            //Remove and read navPanel.  The holder is only repainted after the add.
            _navPanelHolder.removeAll();
            _navPanelHolder.add(_navPanel, BorderLayout.CENTER);
            RepaintManager.currentManager(_navPanelHolder).markCompletelyClean(_navPanelHolder);

            //Setup the split pane
            _splitPane.setTopComponent(_blockPanel);
        }
        //Otherwise, just update it for a new chart.  Note that we do NOT want to create it from scratch, because the navigation
        //panel works better (i.e., no flicker from sudden resizes of panels).
        else
        {
            _blockPanel.updateForNewChart(_fullParameters,
                                          selectedSources,
                                          _selectEventPanel.getSelectedEvents(),
                                          _selectParmPanel.getSelectedParameters());
        }
    }

    /**
     * Register the supplied {@link Subscriber} to the {@link EventBus} that underlies this panel.
     */
    public void registerWithBus(final CreateBlockPlotForSelectedLocationsNotice.Subscriber registree)
    {
        _eventBus.register(registree);
    }

    /**
     * Unregister the supplied {@link Subscriber} to the {@link EventBus} that underlies this panel.
     */
    public void unregisterWithBus(final CreateBlockPlotForSelectedLocationsNotice.Subscriber registree)
    {
        _eventBus.unregister(registree);
    }

    /**
     * @return The parameterId used in the currently selected parameter to view based on {@link #_selectParmPanel}.
     */
    public String getSelectedParameterDataType()
    {
        return _selectParmPanel.getSelectedParameters()[0].getParameterId();
    }

    /**
     * @return The {@link ParameterId.Type} associated with the {@link #_fullParameters} passed into this panel.
     */
    public ParameterId.Type getParameterFileDataType()
    {
        return _fullParameters.getIdentifier().getParameterIdType();
    }

    @Override
    @Subscribe
    public void reactToDisplayedParameterSelected(final DisplayedParameterSelectedNotice evt)
    {
        if(!_currentProcessingSelectedSource)
        {
            updateDisplay();
        }

    }

    @Override
    @Subscribe
    public void reactToCanonicalEventsSelected(final CanonicalEventsSelectedNotice evt)
    {
        if(!_currentProcessingSelectedSource)
        {
            updateDisplay();
        }
    }

    @Override
    @Subscribe
    public void reactToForecastSourceSelected(final ForecastSourceSelectedNotice evt)
    {
        _currentProcessingSelectedSource = true;
        _selectParmPanel.setSourceModelParameters(_fullParameters.getSourceModelParameters(evt.getSelectedSource()));
        _selectEventPanel.setSourceModelParameters(_fullParameters.getSourceModelParameters(evt.getSelectedSource()));
        _currentProcessingSelectedSource = false;

        updateDisplay();
    }
}
