package ohd.hseb.hefs.mefp.pe.setup;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.util.Calendar;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.filechooser.FileFilter;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventTableModel;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventType;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventsManager;
import ohd.hseb.hefs.pe.core.ParameterEstimatorSubPanel;
import ohd.hseb.hefs.pe.notice.CanonicalEventsCopiedNotice;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.notice.StepStatusRefreshAllNotice;
import ohd.hseb.hefs.utils.filechooser.FileExtensionFileFilter;
import ohd.hseb.hefs.utils.filechooser.HGlobalFileChooser;
import ohd.hseb.hefs.utils.gui.jtable.GenericTable;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HCalendar;

import org.jfree.util.Log;

import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;

/**
 * Panel for editing canonical events.
 * 
 * @author alexander.garbarino
 */
@SuppressWarnings("serial")
public class CanonicalEventPanel extends ParameterEstimatorSubPanel implements CanonicalEventsCopiedNotice.Subscriber
{
    private final JTabbedPane _tabbedPane;
    private final CanonicalEventSubPanel _basePanel;
    private final CanonicalEventSubPanel _modPanel;

    private final CanonicalEventsManager _manager;

    /**
     * Table model listener that posts {@link StepStatusRefreshAllNotice} instances whenever the table changes.
     */
    private final TableModelListener _tableModelListener = new TableModelListener()
    {
        @Override
        public void tableChanged(final TableModelEvent e)
        {
            getRunInfo().post(new StepStatusRefreshAllNotice(this));
        }
    };

    /**
     * Restore default events button.
     */
    private final JButton _restoreButton = new SelfListeningButton("Restore Default",
                                                                   "Restore the default canonical events")
    {

        @Override
        public void actionPerformed(final ActionEvent e)
        {
            try
            {
                final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                                                 "Do you wish to save the current canonical events?",
                                                                 "Save Current Events?",
                                                                 JOptionPane.YES_NO_CANCEL_OPTION);
                if(option == JOptionPane.YES_OPTION)
                {
                    if(!processSave())
                    {
                        JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                                      "Save was unsuccessful.  Aborting restore defaults!",
                                                      "Aborting Restore Defaults!",
                                                      JOptionPane.ERROR_MESSAGE);
                        return;
                    }
                }
                else if(option == JOptionPane.CANCEL_OPTION)
                {
                    return;
                }

                restoreCanonicalEvent();
            }
            catch(final Exception exception)
            {
                Log.error("Error restoring to default Canonical Events:" + exception.getMessage());
            }
        }
    };

    /**
     * Save current events button.
     */
    private final JButton _saveButton = new SelfListeningButton("Save", "Save events for current data type to file.")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {
            processSave();
        }
    };

    /**
     * Load current events button.
     */
    private final JButton _loadButton = new SelfListeningButton("Load", "Load events for current data type from file.")
    {
        @Override
        public void actionPerformed(final ActionEvent e)
        {

            final int option = JOptionPane.showConfirmDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                                             "Do you wish to save the current canonical events?",
                                                             "Save Current Events?",
                                                             JOptionPane.YES_NO_CANCEL_OPTION);
            if(option == JOptionPane.YES_OPTION)
            {
                if(!processSave())
                {
                    JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                                  "Save was unsuccessful.  Aborting restore defaults!",
                                                  "Aborting Restore Defaults!",
                                                  JOptionPane.ERROR_MESSAGE);
                    return;
                }
            }
            else if(option == JOptionPane.CANCEL_OPTION)
            {
                return;
            }

            processLoad();
        }
    };

    /**
     * @param runInfo Used only for message posting and to acquire the {@link CanonicalEventsManager}.
     */
    public CanonicalEventPanel(final MEFPParameterEstimatorRunInfo runInfo)
    {
        super(runInfo);
        _manager = runInfo.getCanonicalEventsMgr();
        _tabbedPane = new JTabbedPane();
        _basePanel = new CanonicalEventSubPanel(CanonicalEventType.BASE);
        _modPanel = new CanonicalEventSubPanel(CanonicalEventType.MODULATION);

        initializeDisplay();
    }

    /**
     * Call to open a save dialog for users to save events.
     * 
     * @return True if save was successful, false otherwise.
     */
    private boolean processSave()
    {
        File saveEventsDir = FileTools.newFile(getRunInfo().getBaseDirectory(), "savedCanonicalEvents");
        try
        {
            FileTools.mkdirIfItDoesNotExist(saveEventsDir);
        }
        catch(final IOException e1)
        {
            //Ignore it.  Instead, the saveEventsDir will be set to ".".
            saveEventsDir = new File(".");
        }

        final HGlobalFileChooser chooser = HGlobalFileChooser.getFileChooser("Save "
                                                                                 + getRunInfo().getSelectedDataType()
                                                                                 + " Events Only to XML File",
                                                                             "saveCanonicalEvents",
                                                                             saveEventsDir,
                                                                             Lists.newArrayList((FileFilter)new FileExtensionFileFilter("xml",
                                                                                                                                        "XML file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("xml.gz",
                                                                                                                                        "Gzipped XML file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("fi",
                                                                                                                                        "FastInfoset file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("fi.gz",
                                                                                                                                        "Gzipped FastInfoset file")));

        File toSave = chooser.saveFile(FileTools.newFile(getRunInfo().getSelectedDataType() + ".canonicalEvents."
                                           + HCalendar.buildDateStr(Calendar.getInstance()) + ".xml"),
                                       SwingTools.getGlobalDialogParent(this));
        toSave = FileTools.ensureValidExtension(toSave, "xml", "xml", "xml.gz", "fi", "fi.gz");
        if(toSave == null)
        {
            return false;
        }
        try
        {
            if((getRunInfo().getSelectedDataType() == ParameterId.Type.PRECIPITATION))
            {
                XMLTools.writeXMLFileFromXMLWriter(toSave, _manager.getWriterForPrecipitationOnly(), true);
            }
            else if((getRunInfo().getSelectedDataType() == ParameterId.Type.TEMPERATURE))
            {
                XMLTools.writeXMLFileFromXMLWriter(toSave, _manager.getWriterForTemperatureOnly(), true);
            }
        }
        catch(final Throwable t)
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                          t.getMessage(),
                                          "Error output canonical event data to file!",
                                          JOptionPane.ERROR_MESSAGE);
            return false;
        }
        return true;
    }

    /**
     * Call to open a dialog for the user to load canonical events.
     */
    private void processLoad()
    {
        File loadEventsDir = FileTools.newFile(getRunInfo().getBaseDirectory(), "savedCanonicalEvents");
        if(!loadEventsDir.exists())
        {
            loadEventsDir = new File(".");
        }

        final HGlobalFileChooser chooser = HGlobalFileChooser.getFileChooser("Load "
                                                                                 + getRunInfo().getSelectedDataType()
                                                                                 + " Events Only from XML File",
                                                                             "loadCanonicalEvents",
                                                                             loadEventsDir,
                                                                             Lists.newArrayList((FileFilter)new FileExtensionFileFilter("xml",
                                                                                                                                        "XML file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("xml.gz",
                                                                                                                                        "Gzipped XML file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("fi",
                                                                                                                                        "FastInfoset file"),
                                                                                                (FileFilter)new FileExtensionFileFilter("fi.gz",
                                                                                                                                        "Gzipped FastInfoset file")));
        final File toLoad = chooser.chooseFile(SwingTools.getGlobalDialogParent(this));
        if(toLoad == null)
        {
            return;
        }
        try
        {
            //Initialize event lists to be read in and call the getCompatibleReaderForSingleDataType to turn it into a compatible single data type reader.
            final CanonicalEventList baseEvents = new CanonicalEventList(getRunInfo().getSelectedDataType() == ParameterId.Type.PRECIPITATION);
            final CanonicalEventList modEvents = new CanonicalEventList(getRunInfo().getSelectedDataType() == ParameterId.Type.PRECIPITATION);
            XMLTools.readXMLFromFile(toLoad,
                                     _manager.getCompatibleReaderForSingleDataType(baseEvents,
                                                                                   modEvents,
                                                                                   getRunInfo().getSelectedDataType()));

            //If empty, then there is nothing to do.  Let the user know via an Exception.
            if(baseEvents.isEmpty() && modEvents.isEmpty())
            {
                throw new Exception("No " + getRunInfo().getSelectedDataType()
                    + " events were found in the file.\nWas the file saved for the other data type?");
            }

            //Read was successful, so put the stuff in place.  This will trigger a CanonicalEventsCopiedNotice which this panel listens to in order to update the tables.
            _manager.copyEvents(baseEvents);
            _manager.copyEvents(modEvents);
        }
        catch(final Throwable t)
        {
            JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(CanonicalEventPanel.this),
                                          t.getMessage(),
                                          "Error output canonical event data to file!",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Restore the canonical events.
     * 
     * @throws Exception
     */
    private void restoreCanonicalEvent() throws Exception
    {
        //Get the default canonical events manager and copy the events from it into _manager.
        final CanonicalEventsManager canonicalEventsMgr = MEFPParameterEstimatorRunInfo.loadDefaultRunTimeInformation()
                                                                                       .getCanonicalEventsMgr();
        _manager.copyEvents(canonicalEventsMgr, getRunInfo().getSelectedDataType());
    }

    @Override
    protected void initializeDisplay()
    {
        this.setLayout(new BorderLayout());
        this.add(_tabbedPane, BorderLayout.CENTER);
        _tabbedPane.add("Base", _basePanel);
        _tabbedPane.add("Modulation", _modPanel);

        final JPanel buttonWrapPanel = new JPanel(new FlowLayout());
        buttonWrapPanel.add(_restoreButton);
        buttonWrapPanel.add(_saveButton);
        buttonWrapPanel.add(_loadButton);
        this.add(buttonWrapPanel, BorderLayout.SOUTH);
    }

    @Override
    public MEFPParameterEstimatorRunInfo getRunInfo()
    {
        return (MEFPParameterEstimatorRunInfo)super.getRunInfo();
    }

    /**
     * Panel associated with ether base or modulation events.
     * 
     * @author Hank.Herr
     */
    private class CanonicalEventSubPanel extends JPanel implements SelectedIdentifiersChangedNotice.Subscriber
    {
        private final CanonicalEventType _type;
        private CanonicalEventTableModel _precipModel;
        private CanonicalEventTableModel _tempModel;
        private final JTable _table;

        public CanonicalEventSubPanel(final CanonicalEventType type)
        {
            _type = type;
            initializeModels();

            _table = new GenericTable();
            _table.setFillsViewportHeight(true);
            _table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

            this.setLayout(new BorderLayout());
            this.add(new JScrollPane(_table), BorderLayout.CENTER);

            changeToDataType(getRunInfo().getSelectedDataType());
            getRunInfo().register(this);
        }

        protected void initializeModels()
        {
            initializePrecipitationModel(_manager.getEventList(_type, ParameterId.Type.PRECIPITATION));
            initializeTemperatureModel(_manager.getEventList(_type, ParameterId.Type.TEMPERATURE));
        }

        protected void initializePrecipitationModel(final CanonicalEventList events)
        {
            _precipModel = new CanonicalEventTableModel(events);
            _precipModel.addTableModelListener(_tableModelListener);
        }

        protected void initializeTemperatureModel(final CanonicalEventList events)
        {
            _tempModel = new CanonicalEventTableModel(events);
            _tempModel.addTableModelListener(_tableModelListener);
        }

        @Override
        public String getName()
        {
            return _type.toString();
        }

        private void changeToDataType(final ParameterId.Type type)
        {
            switch(type)
            {
                case PRECIPITATION:
                    _table.setModel(_precipModel);
                    break;
                case TEMPERATURE:
                    _table.setModel(_tempModel);
                    break;
                default:
                    throw new IllegalArgumentException("Unsupported data type " + type);
            }
        }

        @Override
        @Subscribe
        public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice notice)
        {
            changeToDataType(notice.getDataType());
        }
    }

    private void reactToCopiedEventList(final CanonicalEventList list)
    {
        //This is exactly what is done in the restore defaults method.  Update the table!
        if(list.isPrecipitation())
        {
            if(list.isBaseEventType())
            {
                _basePanel.initializePrecipitationModel(_manager.getPrecipitationBaseCanonicalEventList());
            }
            else
            {
                _modPanel.initializePrecipitationModel(_manager.getPrecipitationModulationCanonicalEventList());
            }
        }
        else
        {
            if(list.isBaseEventType())
            {
                _basePanel.initializeTemperatureModel(_manager.getTemperatureBaseCanonicalEventList());
            }
            else
            {
                _modPanel.initializeTemperatureModel(_manager.getTemperatureModulationCanonicalEventList());
            }
        }
    }

    @Override
    @Subscribe
    public void reactToCopiedCanonicalEvents(final CanonicalEventsCopiedNotice notice)
    {
        reactToCopiedEventList(notice.getCopiedEventsList1());
        if(notice.getCopiedEventsList2() != null)
        {
            reactToCopiedEventList(notice.getCopiedEventsList2());
        }
        _basePanel.changeToDataType(getRunInfo().getSelectedDataType());
        _modPanel.changeToDataType(getRunInfo().getSelectedDataType());

        getRunInfo().post(new StepStatusRefreshAllNotice(this));
    }
}
