package ohd.hseb.hefs.mefp.sources;

import ohd.hseb.hefs.mefp.models.MEFPBaseModelControlOptions;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelControlOptions;
import ohd.hseb.hefs.mefp.models.temperature.MEFPTemperatureModelControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventValuesGatherer;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.pe.sources.AbstractForecastSource;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;

/**
 * Abstract implementation of {@link MEFPForecastSource} that extends {@link AbstractForecastSource}.
 * 
 * @author hankherr
 */
public abstract class AbstractMEFPForecastSource extends AbstractForecastSource implements MEFPForecastSource
{

    @Override
    public SourceCanonicalEventValues constructSourceCanonicalEventValues(final CanonicalEventList fullEventList,
                                                                          final int numberOfForecastDays)
    {
        return constructSourceCanonicalEventValues(fullEventList.subList(fullEventList.determineNumberOfApplicableEvents(numberOfForecastDays)));
    }

    @Override
    public MEFPSourceDataHandler getSourceDataHandler()
    {
        return (MEFPSourceDataHandler)super.getSourceDataHandler();
    }

    @Override
    public boolean canEPTModelBeUsedForSource()
    {
        return true;
    }

    @Override
    public SourceCanonicalEventValues constructSourceCanonicalEventValues(final CanonicalEventList eventsToCompute)
    {
        return new SourceCanonicalEventValues(this, eventsToCompute);
    }

    /**
     * Returns gatherer for IPT events, by default.
     */
    @Override
    public CanonicalEventValuesGatherer constructCanonicalEventValuesGatherer(final SourceCanonicalEventValues eventValues,
                                                                              final MEFPBaseModelControlOptions modelControlOptions,
                                                                              final MEFPSourceControlOptions sourceControlOptions)
    {
        if(modelControlOptions instanceof MEFPPrecipitationModelControlOptions)
        {
            //The gatherer is an IPT gatherer.  For EPT, the both minimum number should be DO_NOT_CHECK and the fraction defining wet events is different.
            //Those attributes can be set in the returned gatherer after construction.
            final MEFPPrecipitationModelControlOptions modelCtlParms = (MEFPPrecipitationModelControlOptions)modelControlOptions;
            return new StandardCanonicalEventValuesGatherer(eventValues.getForecastValues(),
                                                            eventValues.getObservedValues(),
                                                            modelCtlParms.getMinimumWidthOfDataWindowInDays(),
                                                            modelCtlParms.getMaximumWidthOfDataWindowInDays(),
                                                            sourceControlOptions.getInitialYear(),
                                                            sourceControlOptions.getFinalYear(),
                                                            modelCtlParms.getIPT().getFractionDefiningWetEvents(),
                                                            modelCtlParms.getMinimumNumberOfObservationsRequired(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveObservations(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveForecasts(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveObservations());
        }
        else
        {

            final MEFPTemperatureModelControlOptions modelCtlParms = (MEFPTemperatureModelControlOptions)modelControlOptions;
            return new StandardCanonicalEventValuesGatherer(eventValues.getForecastValues(),
                                                            eventValues.getObservedValues(),
                                                            modelCtlParms.getMinimumWidthOfDataWindowInDays(),
                                                            modelCtlParms.getMaximumWidthOfDataWindowInDays(),
                                                            sourceControlOptions.getInitialYear(),
                                                            sourceControlOptions.getFinalYear(),
                                                            modelCtlParms.getMinimumNumberOfObservationsRequired());
        }
    }

    @Override
    public int getOperationalLagInHoursWhenEstimatingParameters(final LocationAndDataTypeIdentifier identifier)
    {
        return 0;
    }

    /**
     * @return By default, this returns the same lag used when estimating parameters,
     *         {@link #getOperationalLagInHoursWhenEstimatingParameters(LocationAndDataTypeIdentifier)}.
     */
    @Override
    public int getOperationalLagInHoursWhenAcquringReforecast(final LocationAndDataTypeIdentifier identifier)
    {
        return getOperationalLagInHoursWhenEstimatingParameters(identifier);
    }

    @Override
    public boolean isClimatologySource()
    {
        return false;
    }

    @Override
    public int getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration()
    {
        return 1;
    }

    @Override
    public int getDefaultNumberOfForecastDays()
    {
        return -1;
    }

    @Override
    public int getDefaultInitialYear()
    {
        return Integer.MIN_VALUE;
    }

    @Override
    public int getDefaultLastYear()
    {
        return Integer.MAX_VALUE;
    }

}
