package ohd.hseb.hefs.mefp.sources;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.sources.AbstractSourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.collect.Lists;

/**
 * Provides a default {@link #getSingleObservedTimeSeries(LocationAndDataTypeIdentifier)} and
 * {@link #loadHindcastTimeSeries(LocationAndDataTypeIdentifier, ParameterId, long)}.
 * 
 * @author hankherr
 */
public abstract class AbstractMEFPSourceDataHandler extends AbstractSourceDataHandler implements MEFPSourceDataHandler
{

    @Override
    public TimeSeriesArray getSingleObservedTimeSeries(final LocationAndDataTypeIdentifier identifier) throws ParameterEstimationException
    {
        final Collection<TimeSeriesArray> TSs = getLoadedObservedTimeSeries(identifier);
        if(TSs.size() != 1)
        {
            throw new ParameterEstimationException("Expected one observed time series corresponding to identifier "
                + identifier.buildStringToDisplayInTree() + ", but found " + TSs.size());
        }
        return TSs.iterator().next();
    }

    /**
     * Default version loads all time series and then picks out the one it wants. This is not efficient, as it would be
     * best to load only the needed time series directly. This requires additional testing to confirm that its getting
     * the same time series as is gotten in Fortran and a possible rewrite to a binary search algorithm. <br>
     * <br>
     * Note that a time series found is accepted if either it is an exact match for the forecast time -OR- the first 12Z
     * time AFTER the reforecast T0 is an exact match. It does this because canonical computations performs the same
     * first 12Z search during parameter estimation, so it is best to do it the same here.
     */
    @Override
    public Collection<TimeSeriesArray> loadHindcastTimeSeries(final LocationAndDataTypeIdentifier identifier,
                                                              final ParameterId dataTypeToLoad,
                                                              final long desiredForecastTime) throws Exception
    {
        final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();

        //Load all time series.
        loadPreparedTimeSeries(Lists.newArrayList(identifier));

        //Add any time series with the desired forecasst time to the results.
        for(final TimeSeriesArray ts: this.getAllLoadedForecastTimeSeries())
        {
            //Here is the check for an exact forecast time match or a match to the first 12Z time after the reforecast T0.
            if(((ts.getHeader().getForecastTime() == desiredForecastTime) || (DateTools.findFirstExact12ZTimeAfterOrOn(ts.getHeader()
                                                                                                                         .getForecastTime()) == desiredForecastTime))
                && (ParameterId.of(ts.getHeader()).equals(dataTypeToLoad)))
            {
                results.add(ts);
            }
        }

        //Throw an exception if none are found.  Return the results otherwise.
        if(results.isEmpty())
        {
            throw new Exception("Unable to find a time series for the desired forecast time, "
                + HCalendar.buildDateTimeTZStr(desiredForecastTime) + ", for the parameterId "
                + dataTypeToLoad.toString() + ".");

        }
        return results;
    }

    /**
     * @param identifier The identifier to translate.
     * @return The translated identifier. By default, the passed in identifier is returned.
     */
    @Override
    public LocationAndDataTypeIdentifier translateIdentifier(final LocationAndDataTypeIdentifier identifier)
    {
        return identifier;
    }

    /**
     * Default is to return null, indicating there is no questionable data.
     * 
     * @param identifier - identifier used to construct the questionable file name
     * @return The map created by {@link QuestionableTools#toHash(java.io.File)}.
     * @throws Exception
     */
    @Override
    public HashMap<String, HashMap<Long, HashMap<Long, String>>> loadQuestionableHash(final LocationAndDataTypeIdentifier identifier) throws Exception
    {
        return null;
    }
}
