package ohd.hseb.hefs.mefp.sources;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.precipitation.PrecipitationParameterEstimationModel;
import ohd.hseb.hefs.mefp.models.temperature.TemperatureParameterEstimationModel;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.cfsv2.CFSv2PEStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepOptionsPanel;
import ohd.hseb.hefs.pe.core.StepTableModel;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.notice.SelectedIdentifiersChangedNotice;
import ohd.hseb.hefs.pe.tools.DefaultStepTableStatusProvider;
import ohd.hseb.hefs.pe.tools.GenericSummaryTablePanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tools.StringTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;

/**
 * Super class to use for gridded forecast sources, compatible with {@link GriddedPEStepProcessor}. It uses a
 * {@link GenericSummaryTablePanel} to summarize the status for various locations. In addition to standard buttons, a
 * view button tied to {@link MEFPSourceDataHandler#getAllLoadedForecastTimeSeries()} is provided for viewing the
 * gridded time series found for a location.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public abstract class GriddedPEStepOptionsPanel extends ParameterEstimatorStepOptionsPanel
{
    private static final Logger LOG = LogManager.getLogger(GriddedPEStepOptionsPanel.class);

    private GenericSummaryTablePanel _locationPanel;
//    private GFSLocationTableModel _locationModel;

    private final JButton _viewButton = new SelfListeningButton("chart20x20", "View Forecasts")
    {

        @Override
        public void actionPerformed(final ActionEvent e)
        {
            viewSelectedForecasts();
        }
    };

    public GriddedPEStepOptionsPanel(final MEFPParameterEstimatorRunInfo runInfo,
                                     final GriddedPEStepProcessor stepProcessor)
    {
        super(runInfo, stepProcessor);
        initializeDisplay();
        updateEnablednessOfViewButton();
    }

    /**
     * @return A name to use for the time series that are viewed via {@link #viewSelectedForecasts()}, which is
     *         triggered with the button {@link #_viewButton}. This string is also displayed in the tool tip.
     */
    protected abstract String getSourceTimeSeriesShortName();

    /**
     * Instantiates a diagnostic panel for viewing the time series viewed via {@link #viewSelectedForecasts()}.
     * 
     * @param identifier
     * @param tsc The time series to view.
     * @return A instance of {@link ParameterEstimatorDiagnosticPanel} that displays the time series.
     */
    protected abstract ParameterEstimatorDiagnosticPanel instantiateDiagnosticPanel(LocationAndDataTypeIdentifier identifier,
                                                                                    Collection<TimeSeriesArray> tsc);

    /**
     * @return A list of components to add to the {@link GenericSummaryTablePanel} tool bar. Override if, for example,
     *         you have another view button to add, as is the case for {@link CFSv2PEStepOptionsPanel}.
     */
    protected List<Component> buildAdditionalToolBarComponents()
    {
        return Lists.newArrayList();
    }

    /**
     * @return The identifier selected in the {@link #_locationPanel}. Call as needed; there should be no need to
     *         override.
     */
    protected LocationAndDataTypeIdentifier getSelectedIdentifier()
    {
        return _locationPanel.getSelectedIdentifier();
    }

    /**
     * Updates the enabledness of the {@link #_viewButton}. Override if you have other buttons that must have their
     * enabledness overridden whenever a selection is made in the {@link #_locationPanel}.
     */
    protected void updateEnablednessOfViewButton()
    {
        final LocationAndDataTypeIdentifier identifier = getSelectedIdentifier();
        if(identifier != null)
        {
            _viewButton.setEnabled(getHandler().havePreparedDataFilesBeenCreatedAlready(identifier));
            if(_viewButton.isEnabled())
            {
                _viewButton.setToolTipText("View the " + getSourceTimeSeriesShortName()
                    + " archive/reforecast time series for " + identifier.buildStringToDisplayInTree() + ".");
            }
            else
            {
                _viewButton.setToolTipText("<html>No " + getSourceTimeSeriesShortName()
                    + " archive/reforecast files are available for " + identifier.buildStringToDisplayInTree()
                    + ".<br>Click the Download Files Button to download them.</html>");
            }
        }
        else
        {
            _viewButton.setEnabled(false);
            _viewButton.setToolTipText("Select one row to view the " + getSourceTimeSeriesShortName()
                + " archive/reforecast time series for that location and data type.");
        }
    }

    /**
     * Override if you need to do something special for the options panel. For example, GEFS needs to include ensembles
     * as well as mean time series.
     * 
     * @param identifiers Identifiers for which to load forecasts.
     * @return Loaded time series.
     * @throws Exception
     */
    protected Collection<TimeSeriesArray> loadDiagnosticTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        //Forecast TS
        final Collection<TimeSeriesArray> tsc = new ArrayList<TimeSeriesArray>();
        getHandler().loadPreparedTimeSeries(identifiers);
        tsc.addAll(getHandler().getAllLoadedForecastTimeSeries());

        //Observed TS
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            try
            {
                if(identifier.isPrecipitationDataType())
                {
                    tsc.add(PrecipitationParameterEstimationModel.retrieveHistoricalData((MEFPParameterEstimatorRunInfo)getRunInfo(),
                                                                                         identifier));
                }
                else if(identifier.isTemperatureDataType())
                {
                    tsc.addAll(TemperatureParameterEstimationModel.retrieveHistoricalData((MEFPParameterEstimatorRunInfo)getRunInfo(),
                                                                                          identifier));
                }
                else
                {
                    throw new IllegalArgumentException("Identifier " + identifier.buildStringToDisplayInTree()
                        + " is neither precip nor temp.  How did that happen?");
                }
            }
            catch(final Exception e)
            {
                LOG.warn("Unable to load observed data for display in diagnostic plot: " + e.getMessage());
                LOG.warn("Be sure that the processed historical data file has been generated.");
            }
        }

        return tsc;
    }

    /**
     * Calls the {@link MEFPSourceDataHandler#getAllLoadedForecastTimeSeries()} method to acquire time series to view.
     */
    protected void viewSelectedForecasts()
    {
        final LocationAndDataTypeIdentifier selectedIdent = _locationPanel.getSelectedIdentifier();
        if(selectedIdent == null)
        {
            return;
        }
        final List<LocationAndDataTypeIdentifier> identifiers = Lists.newArrayList(selectedIdent);

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading " + getSourceTimeSeriesShortName() + " time series...");

                Collection<TimeSeriesArray> tsc = new ArrayList<TimeSeriesArray>();
                try
                {
                    tsc = loadDiagnosticTimeSeries(identifiers);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Error loading " + getSourceTimeSeriesShortName()
                                              + " time series for location "
                                              + selectedIdent.buildStringToDisplayInTree() + ": " + e.getMessage()),
                                          true);
                    return;
                }

                updateNote("Constructing chart...");
                final ParameterEstimatorDiagnosticPanel panel = instantiateDiagnosticPanel(_locationPanel.getSelectedIdentifier(),
                                                                                           tsc);
                fireDiagnostic(panel);

                endTask();

                //Dump the time series to a file for reading
                //            TimeSeriesArray[] usedTS = new TimeSeriesArray[100];
                //            for(int i = 0; i < 100; i++)
                //            {
                //                usedTS[i] = ts.get(i);
                //            }
                //            ts = new TimeSeriesArrays(usedTS);
                //            TimeSeriesArraysTools.writeToFile(new File("testdata/diagnosticProducts/gfsTemperatureForecastDisplayDiagnosticsTS.xml"),
                //                                              ts);
            }
        };
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this, "Building Chart to Display "
            + getSourceTimeSeriesShortName() + " Reforecasts", buildPanelJob, false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying " + getSourceTimeSeriesShortName()
                    + " archive/reforecasts:", exc.getMessage());
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(GriddedPEStepOptionsPanel.this),
                                              StringTools.wordWrap(exc.getMessage(), 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    public MEFPForecastSource getSource()
    {
        return getStepProcessor().getSource();
    }

    public MEFPSourceDataHandler getHandler()
    {
        return getSource().getSourceDataHandler();
    }

    @Override
    protected void initializeDisplay()
    {
        _viewButton.setToolTipText("View the " + getSource().getSourceId()
            + " archive/reforecasts for a single, selected location and data type.");
        final List<Component> addedComponents = Lists.newArrayList((Component)_viewButton);
        addedComponents.addAll(buildAdditionalToolBarComponents());
        _locationPanel = new GenericSummaryTablePanel(getRunInfo(),
                                                      "Summary of Available " + getSource().getSourceId()
                                                          + " Forecast Data",
                                                      new StepTableModel(getRunInfo(),
                                                                         getStepProcessor().getClass(),
                                                                         new DefaultStepTableStatusProvider(getStepProcessor())),
                                                      true,
                                                      addedComponents);
        _locationPanel.addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                updateEnablednessOfViewButton();
            }
        });

        setLayout(new BorderLayout());
        add(_locationPanel, BorderLayout.CENTER);
    }

    @Override
    public GriddedPEStepProcessor getStepProcessor()
    {
        return (GriddedPEStepProcessor)super.getStepProcessor();
    }

    @Override
    public List<LocationAndDataTypeIdentifier> getStepUnitsToPerform()
    {
        return _locationPanel.getSelectedIdentifiers();
    }

    @Override
    public void reactToSelectedIdentifiersChanged(final SelectedIdentifiersChangedNotice evt)
    {
        _locationPanel.getModel().setIdentifiers(evt.getIdentifiers());
        _locationPanel.refreshTable();
    }

    @Override
    public void gotoUnit(final Collection<StepUnit> units)
    {
        _locationPanel.selectIdentifiers(ListTools.convertCollection(units, (LocationAndDataTypeIdentifier)null));
    }

}
