package ohd.hseb.hefs.mefp.sources;

import java.io.File;
import java.io.IOException;
import java.util.List;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.tools.MEFPPESFTPSettings;
import ohd.hseb.hefs.pe.core.GenericParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.core.StepUnit;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.ftp.SFTPConductor;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.status.StatusLabel;
import ohd.hseb.util.misc.HString;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;

/**
 * General super class for gridded forecast source step processors. This step processor is designed to SFTP to server to
 * acquire ASCII grid files that contain the archive or reforecasts for a specific grid point. It is up to the source's
 * {@link MEFPSourceDataHandler} to identify the correct files to acquire via
 * {@link MEFPSourceDataHandler#generateListOfPreparedDataFiles(LocationAndDataTypeIdentifier)} method. The sftp
 * information is acquired via {@link MEFPParameterEstimatorRunInfo#getGriddedDataSFTPSettings()}. Name information
 * about the forecast source is acquired via {@link MEFPForecastSource#getSourceId()}. This uses the return value of
 * {@link MEFPForecastSource#getSourceDataHandler()} as its data handler to use, except when constructing an SFTP server
 * data handler.
 * 
 * @author hankherr
 */
public abstract class GriddedPEStepProcessor extends GenericParameterEstimatorStepProcessor
{
    private static final Logger LOG = LogManager.getLogger(GriddedPEStepProcessor.class);
    private final MEFPForecastSource _source;
    private final MEFPParameterEstimatorRunInfo _runInformation;
    private SFTPConductor _sftpConductor = null;
    private final List<String> _alreadyDownloadedServerFileNames = Lists.newArrayList();

    public GriddedPEStepProcessor(final MEFPParameterEstimatorRunInfo runInformation, final MEFPForecastSource source)
    {
        _source = source;
        _runInformation = runInformation;
    }

    /**
     * @return An {@link MEFPSourceDataHandler} initialize for use in SFTP-ing files. The base directory for this
     *         handler should be acquired via {@link #getRunInfo()} method
     *         {@link MEFPParameterEstimatorRunInfo#getGriddedDataSFTPSettings()} and the returned objects method
     *         {@link MEFPPESFTPSettings#getBaseDirectoryOnServer()}.
     * @throws Exception If the data handler cannot be constructed.
     */
    protected abstract MEFPSourceDataHandler instantiateSFTPHandler() throws Exception;

    public MEFPForecastSource getSource()
    {
        return _source;
    }

    /**
     * @return Shortcut to {@link #_source} method {@link MEFPForecastSource#getSourceDataHandler()}.
     */
    public MEFPSourceDataHandler getHandler()
    {
        return _source.getSourceDataHandler();
    }

    public MEFPParameterEstimatorRunInfo getRunInfo()
    {
        return _runInformation;
    }

    /**
     * @return Shortcut to {@link #getHandler()} method
     *         {@link MEFPSourceDataHandler#havePreparedDataFilesBeenCreatedAlready(LocationAndDataTypeIdentifier)}.
     */
    protected boolean doFilesExist(final LocationAndDataTypeIdentifier identifier)
    {
        return getHandler().havePreparedDataFilesBeenCreatedAlready(identifier);
    }

    @Override
    public StatusIndicator getStatus(final StepUnit unit)
    {
        if(!(unit instanceof LocationAndDataTypeIdentifier))
        {
            throw new IllegalArgumentException("Must be passed an Identifier.");
        }
        final LocationAndDataTypeIdentifier identifier = (LocationAndDataTypeIdentifier)unit;

        if(doFilesExist(identifier))
        {
            return StatusLabel.make(true, "<html>Files containing " + getSource().getSourceId()
                + " processed ASCII ensemble mean reforecasts were found.<br>"
                + "Contents were not checked for completeness.</html>");
        }
        else
        {
            return StatusLabel.make(false, "No files were found that contain the " + getSource().getSourceId()
                + " processed ASCII ensemble mean reforecasts.");
        }
    }

    @Override
    public void prepareStep(final boolean skipUserInput) throws Exception
    {
        _alreadyDownloadedServerFileNames.clear();
    }

    @Override
    public void performStep(final StepUnit unit) throws Exception
    {
        //Get the file lists
        final List<File> localFileLocations = getHandler().generateListOfPreparedDataFiles((LocationAndDataTypeIdentifier)unit);
        final MEFPSourceDataHandler ftpDataHandler = instantiateSFTPHandler();
        final List<File> serverFileLocations = ftpDataHandler.generateListOfPreparedDataFiles((LocationAndDataTypeIdentifier)unit);

        JobMessenger.newMonitorSubJob();
        JobMessenger.setMaximumNumberOfSteps(serverFileLocations.size() + 1);

        //Need to make sure the monitor sub job is cleared, so I wrapped the below in a try-finally.
        try
        {
            //Connect
            if(_sftpConductor == null)
            {
                JobMessenger.updateNote("Connecting to "
                    + _runInformation.getGriddedDataSFTPSettings().getServerName()
                    + " or one of its backups ("
                    + HString.buildStringFromList(_runInformation.getGriddedDataSFTPSettings().getBackupServerNames(),
                                                  ",") + ") as user '"
                    + _runInformation.getGriddedDataSFTPSettings().getUserName() + "'...");
                try
                {
                    _sftpConductor = _runInformation.getGriddedDataSFTPSettings().startFTPSession();
                }
                catch(final Exception e)
                {
                    throw e;
                }
            }
            else
            {
                JobMessenger.updateNote("Using already established connection to ftp server.");
            }

            //Retrieve files
            for(int fileIndex = 0; fileIndex < serverFileLocations.size(); fileIndex++)
            {
                //Remove any leading d:/ from the server file name.
                String serverFileName = serverFileLocations.get(fileIndex).getAbsolutePath();
                serverFileName = serverFileName.substring(serverFileName.indexOf(File.separator));

                //Check to see if this set of step runs has already included downloading this file for a different unit.
                if(_alreadyDownloadedServerFileNames.contains(serverFileName))
                {
                    LOG.debug("Skipping download... the file has already been acquired: " + serverFileName);
                    JobMessenger.madeProgress("Already acquired file " + serverFileName + "...");
                    continue;
                }

                JobMessenger.madeProgress("Getting file " + serverFileName + "...");

                try
                {
                    _sftpConductor.getFileFromServer(localFileLocations.get(fileIndex).getAbsolutePath(),
                                                     serverFileName);
                    _alreadyDownloadedServerFileNames.add(serverFileName);
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    throw new IOException("Error downloading file " + serverFileName + " to "
                        + localFileLocations.get(fileIndex).getAbsolutePath() + ":" + e.getMessage());
                }
            }

            JobMessenger.madeProgress("Done.");
        }
        finally
        {
            JobMessenger.clearMonitorSubJob();
        }
    }

    @Override
    public void cleanupAfterStep()
    {
        if(_sftpConductor != null)
        {
            _sftpConductor.closeConnection();
            _sftpConductor = null;
        }
        _alreadyDownloadedServerFileNames.clear();
    }

    @Override
    public String getToolTipTextForIdentifierTableColumnHeader()
    {
        return "<html>Displays if the processed ASCII file for the location and data type can be found under <br>"
            + getHandler().getDataHandlerBaseDirectory().getAbsolutePath() + ".</html>";
    }

    @Override
    public String getToolTipTextDescribingStep()
    {
        return "Download " + getSource().getSourceId() + " gridded reforecast/hindcast files from SFTP server.";
    }

    @Override
    public String getShortNameForIdentifierTableColumn()
    {
        return getSource().getSourceId();
    }

    @Override
    public String getTabNameForStep()
    {
        return getSource().getSourceId();
    }

    @Override
    public String getStepNameForRunButton()
    {
        return "Download " + getSource().getSourceId() + " Reforecast/Archive Files";
    }

    @Override
    public String getPerformStepPrefix()
    {
        return "Downloading " + getSource().getSourceId() + " processed ASCII hindcast files from ftp server";
    }
}
