package ohd.hseb.hefs.mefp.sources;

import ohd.hseb.hefs.mefp.models.MEFPBaseModelControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventValuesGatherer;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;

public interface MEFPForecastSource extends ForecastSource
{

    @Override
    public MEFPSourceDataHandler getSourceDataHandler();

    @Override
    public MEFPSourceControlOptions createControlOptions(ParameterId.Type type);

    /**
     * @return True if the EPT (explicit precipitation treatment; old EPP2 model) precipitation model can be used with
     *         this data source, or only the IPT (implicit precipitation treatment) model can be used.
     */
    public boolean canEPTModelBeUsedForSource();

    /**
     * Calls the other version using the full event list and number of forecast days to determine the applicable events.
     * 
     * @param fullEventList Complete list of events.
     * @param numberOfForecastDays The number of forecast days to use for the source.
     * @return An initialized SourceCanonicalEventValues via calling
     *         {@link #constructSourceCanonicalEventValues(CanonicalEventList)}.
     */
    public SourceCanonicalEventValues constructSourceCanonicalEventValues(CanonicalEventList fullEventList,
                                                                          int numberOfForecastDays);

    /**
     * @param eventsToCompute the {@link CanonicalEventList} of {@link CanonicalEvent}s to compute.
     * @return An instance of {@link SourceCanonicalEventValues} that can be used to compute canonical events for this
     *         {@link MEFPForecastSource}.
     */
    public SourceCanonicalEventValues constructSourceCanonicalEventValues(CanonicalEventList eventsToCompute);

    /**
     * @param eventValues the event values for which values are to be gathered.
     * @param modelControlOptions Control parameters specifying how to gather the parameters.
     * @param sourceControlOptions Control parameters specifying how to gather the events.
     * @return An instance of {@link CanonicalEventValuesGatherer}, usually {@link StandardCanonicalEventValuesGatherer}
     *         , that can be used to gather the canonical event values for the days of the year.
     */
    public CanonicalEventValuesGatherer constructCanonicalEventValuesGatherer(SourceCanonicalEventValues eventValues,
                                                                              MEFPBaseModelControlOptions modelControlOptions,
                                                                              MEFPSourceControlOptions sourceControlOptions);

    /**
     * @return The delay in hours between a desired forecast time when running MEFP operationally and the time of the
     *         forecast from the source forecast that is used. This is applied when computing both the forecast
     *         canonical event values. Specifically, the lag is applied to the reforecast T0 to compute the effective
     *         computational T0 in order to mimic the operational lag. Then the first 12Z time after that is used as the
     *         starting point for canonical event computations. <br>
     * <br>
     *         When deciding on a lag to use for a source, remember that the first 12Z value after the reforecast T0 is
     *         what is actually used for computing canonical events and used for storing those events, AFTER the lag is
     *         applied. Thus,if the only lag is a 12-h lag because the reforecasts are for 0Z but the operational
     *         forecasts are at 12Z, then you don't have to worry... this lag is already handled when events are
     *         computed.<br>
     * <br>
     *         For example, for CFSv2, an operational forecast for Jan 1 12Z uses forecasts no more recent that Dec 31
     *         at 6Z (there is a 24+ hour lag between a CFSv2 forecast T0 and the T0 for which it will be available). To
     *         apply this lag to parameter estimation... A CFSv2 reforecast would be effectively used no sooner than 24
     *         hours AFTER its T0. This means that the lag is applied to the reforecast so that events are computed
     *         starting 24 hours AFTER its T0 and that computational T0 is also used to compute the corresponding
     *         observed events. Note that canonical event values are stored based on the computational T0, so that the
     *         lag does not show up when computing observed canonical event values; it is already applied in the
     *         computational T0. <br>
     * <br>
     *         NOTE: As discussed in the CFSv2 code, we had to ditch the 24-h lag because of there only being EXACTLY
     *         270 days of reforecast monthly data. If any lag is applied, then we would not need the 270 days needed
     *         for some events. Hence, event CFSv2 now uses a 0-hour lag. <br>
     * <br>
     *         One more note: this should always be 0 for climatology forecast sources.
     */
    public int getOperationalLagInHoursWhenEstimatingParameters(LocationAndDataTypeIdentifier identifier);

    /**
     * @return See {@link #getOperationalLagInHoursWhenEstimatingParameters(LocationAndDataTypeIdentifier)}. This lag is
     *         applied when the MEFP is run in hindcast mode and the desired canonical events cannot be found in
     *         parameter event files, so that the reforecast files must be looked at in order to find the needed
     *         reforecast. The same logic as with the other method applies: when searching the reforecasts for which one
     *         to use at a given 12Z T0, its backs up the time by this lag before looking for an appropriate reforecast.<br>
     * <br>
     *         This is NOT used when finding the events to use for a forecast time via the events parameter file. The
     *         reason is because those events are stored based on event computational time, which is the effective T0
     *         for when the event values would be applied. That time should already account for the operational lag.
     */
    public int getOperationalLagInHoursWhenAcquringReforecast(LocationAndDataTypeIdentifier identifier);

    /**
     * @return true If this is a climatology source, false if not.
     * @return
     */
    public boolean isClimatologySource();

    /**
     * @return The number of time series required for ensemble generation for this forecast source. Example, GFS and RFC
     *         require one time series, while CFSv2 requires 16 (the lagged ensemble).
     */
    public int getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration();

    /**
     * @return The default number of forecast days estimation option to use for this source. This number will be used if
     *         (1) it is not negative, and (2) the defaultRunTimeInformation.xml file loaded at run time does not
     *         override it.
     */
    public int getDefaultNumberOfForecastDays();

    /**
     * @return The default initial year estimation option to use for this source. This number will be used if (1) it is
     *         neither {@link Integer#MIN_VALUE} or {@link Integer#MAX_VALUE}, and (2) the defaultRunTimeInformation.xml
     *         file loaded at run time does not override it.
     */
    public int getDefaultInitialYear();

    /**
     * @return The default last year estimation option to use for this source. This number will be used if (1) it is
     *         neither {@link Integer#MIN_VALUE} or {@link Integer#MAX_VALUE}, and (2) the defaultRunTimeInformation.xml
     *         file loaded at run time does not override it.
     */
    public int getDefaultLastYear();
}
