package ohd.hseb.hefs.mefp.sources;

import java.util.Collection;
import java.util.HashMap;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.tools.QuestionableTools;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;

/**
 * Extension of SourceDataHandler adds methods associated with running the underlying Fortran prototypes. It will go
 * away once the MEFPPE algorithms are recoded into Java.
 * 
 * @author hank.herr
 */
public interface MEFPSourceDataHandler extends SourceDataHandler
{
    /**
     * @param identifier The identifier for which one time series should be found.
     * @return The one observed time series. This presumes that {@link #loadOriginalTimeSeries(java.util.List)} and/or
     *         {@link #loadPreparedTimeSeries(java.util.List)} has already been called, loading the time series into
     *         memory. This should only need to call {@link #getLoadedObservedTimeSeries(LocationAndDataTypeIdentifier)}
     *         , check the returned collection's size, and return the one object in the collection.
     * @throws {@link ParameterEstimationException} if 0 or more than 1 time series is found.
     */
    public TimeSeriesArray getSingleObservedTimeSeries(LocationAndDataTypeIdentifier identifier) throws ParameterEstimationException;

    /**
     * Used in hindcasting, this must go to the reforecast/archive files and pull out the time series that should be
     * used in hindcasting for the given identifier, parameter id, and forecast time.
     * 
     * @param identifier Identifier for which to acquire reforecast/archive time series.
     * @param dataTypeToLoad The parameter id of the data to load.
     * @param desiredForecastTime The desired hindcast forecast time.
     * @return {@link Collection} of time series for the given forecast time; if more than one, it means an ensemble is
     *         used.
     * @throws Exception If a hindcast with the given precise forecastTime cannot be found.
     */
    public Collection<TimeSeriesArray> loadHindcastTimeSeries(LocationAndDataTypeIdentifier identifier,
                                                              ParameterId dataTypeToLoad,
                                                              long desiredForecastTime) throws Exception;

    // This is a default version that will work for historical data handler within the AbstractMEFPSourceDataHandler.  
    // For the RFCForecastDataHandler, override this method to make it work.

    /**
     * Used in tool tip to display questionable values. Reads a questionable XML file and returns a triple HashMap. See
     * QuestionableTool.toHash() for a description of the triple hash.
     * 
     * @param identifier - identifier used to construct the questionable file name
     * @return The map created by {@link QuestionableTools#toHash(java.io.File)}.
     */
    public HashMap<String, HashMap<Long, HashMap<Long, String>>> loadQuestionableHash(final LocationAndDataTypeIdentifier identifier) throws Exception;
}
