package ohd.hseb.hefs.mefp.sources.cfsv2;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.graphgen.utils.GraphGenCalculatorTools;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPSourceDataHandler;
import ohd.hseb.hefs.mefp.sources.GriddedPEStepOptionsPanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.geo.CoordinateGrid;
import ohd.hseb.hefs.utils.geo.CoordinateGridPoint;
import ohd.hseb.hefs.utils.geo.CoordinateSystemFilesHandler;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.LaggedEnsemble;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.agg.AggregationTools;
import ohd.hseb.hefs.utils.tsarrays.agg.TimeSeriesAggregationException;
import ohd.hseb.util.misc.HCalendar;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;

public class CFSv2DataHandler extends AbstractMEFPSourceDataHandler
{
    private static final Logger LOG = LogManager.getLogger(CFSv2DataHandler.class);

    /**
     * Stores the location of the processed ASCII file directories
     */
    private File _preparedDataFileDirectory = null;

    /**
     * The file handler to use for dealing with Gaussian grids.
     */
    private final CoordinateGrid _gaussianGrid;

    /**
     * These are {@link ArrayListMultimap}, so that the order of the returned list matches the order of the time series
     * added, which should match the order in the files.
     */
    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _identifierToSubmonTSMap;

    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _identifierToDailySubmonTSMap;

    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesArrays> _identifierToMonEnsemblesMap;

    private final ArrayListMultimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _identifierToMonMeanTSMap;

    public CFSv2DataHandler() throws Exception
    {
        _gaussianGrid = CoordinateSystemFilesHandler.GAUSSIAN_T126_COORDINATES.loadCoordinates();

        _identifierToSubmonTSMap = ArrayListMultimap.create();
        _identifierToMonEnsemblesMap = ArrayListMultimap.create();
        _identifierToMonMeanTSMap = ArrayListMultimap.create();
        _identifierToDailySubmonTSMap = ArrayListMultimap.create();
    }

    public CFSv2DataHandler(final File baseDirectory) throws Exception
    {
        this();
        setDataHandlerBaseDirectory(baseDirectory);
    }

    private String constructLatLonComponentOfFileNames(final CoordinateGridPoint cfsv2Pt)
    {
        final DecimalFormat lonNF = new DecimalFormat("000.0000");
        lonNF.setNegativePrefix("");

        final DecimalFormat latNF = new DecimalFormat("00.0000");
        latNF.setNegativePrefix("");

        //lat/lon component of file names.
        String latLonStr = lonNF.format(cfsv2Pt.getLongitude());
        if(cfsv2Pt.getLongitude() < 0)
        {
            latLonStr += "W";
        }
        else
        {
            latLonStr += "E";
        }
        latLonStr += "_" + latNF.format(cfsv2Pt.getLatitude());
        if(cfsv2Pt.getLatitude() > 0)
        {
            latLonStr += "N";
        }
        else
        {
            latLonStr += "S";
        }
        return latLonStr;
    }

    public boolean areSubmonthlyFilesPresent(final LocationAndDataTypeIdentifier identifier)
    {
        //Every other file starting with the first is a submonthly file (i.e., either index 0
        //for precip or both 0 and 2 for temp).
        final List<File> files = this.generateListOfPreparedDataFiles(identifier);
        for(int i = 0; i < files.size(); i += 2)
        {
            if(!files.get(i).exists())
            {
                return false;
            }
        }
        return true;
    }

    public boolean areMonthlyFilesPresent(final LocationAndDataTypeIdentifier identifier)
    {
        //Every other file starting with the second is a submonthly file (i.e., either index 1
        //for precip or both 1 and 3 for temp).
        final List<File> files = this.generateListOfPreparedDataFiles(identifier);
        for(int i = 1; i < files.size(); i += 2)
        {
            if(!files.get(i).exists())
            {
                return false;
            }
        }
        return true;
    }

    private void loadPrecipitationPreparedTimeSeries(final LocationAndDataTypeIdentifier identifier,
                                                     final File submonFile,
                                                     final File monFile)
    {
        List<TimeSeriesArray> ts = new ArrayList<TimeSeriesArray>();
        List<TimeSeriesArrays> monEnsembles = new ArrayList<TimeSeriesArrays>();
        List<TimeSeriesArray> monMeanTS = new ArrayList<TimeSeriesArray>();

        try
        {
            ts = ProcessedCFSv2FileTools.readPrecipitationSubmonthlyFile(identifier, submonFile);
            this._identifierToSubmonTSMap.putAll(identifier, ts);
        }
        catch(final Exception e)
        {
            LOG.info("Unable to load precipitation submonthly data for " + identifier.buildStringToDisplayInTree()
                + ": " + e.getMessage());
        }

        try
        {
            monEnsembles = ProcessedCFSv2FileTools.readPrecipitationMonthlyFile(identifier, monFile);
            _identifierToMonEnsemblesMap.putAll(identifier, monEnsembles);

            monMeanTS = GraphGenCalculatorTools.computeEnsembleMeanTimeSeries(monEnsembles);
            _identifierToMonMeanTSMap.putAll(identifier, monMeanTS);
        }
        catch(final Exception e)
        {
            LOG.info("Unable to load precipitation monthly ensembles for " + identifier.buildStringToDisplayInTree()
                + " and/or compute means: " + e.getMessage());
        }
    }

    private void loadTemperaturePreparedTimeSeries(final boolean minimumTemperature,
                                                   final LocationAndDataTypeIdentifier identifier,
                                                   final File submonFile,
                                                   final File monFile)
    {
        List<TimeSeriesArray> ts = new ArrayList<TimeSeriesArray>();
        List<TimeSeriesArrays> monEnsembles = new ArrayList<TimeSeriesArrays>();
        List<TimeSeriesArray> monMeanTS = new ArrayList<TimeSeriesArray>();

        try
        {
            ts = ProcessedCFSv2FileTools.readTemperatureSubmonthlyFile(minimumTemperature, identifier, submonFile);
            this._identifierToSubmonTSMap.putAll(identifier, ts);
        }
        catch(final Exception e)
        {
            LOG.info("Unable to load temperature submonthly data for " + identifier.buildStringToDisplayInTree() + ": "
                + e.getMessage());
        }

        try
        {
            monEnsembles = ProcessedCFSv2FileTools.readTemperatureMonthlyFile(minimumTemperature, identifier, monFile);
            _identifierToMonEnsemblesMap.putAll(identifier, monEnsembles);

            monMeanTS = GraphGenCalculatorTools.computeEnsembleMeanTimeSeries(monEnsembles);
            _identifierToMonMeanTSMap.putAll(identifier, monMeanTS);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.info("Unable to load temperature monthly ensembles for " + identifier.buildStringToDisplayInTree()
                + " and/or compute means: " + e.getMessage());
        }
    }

    public void clearDailySubmonthlyMap()
    {
        _identifierToDailySubmonTSMap.clear();
    }

    /**
     * Builds the {@link #_identifierToDailySubmonTSMap} based on the {@link #_identifierToSubmonTSMap} by aggregating
     * each to a daily 12Z to 12Z map. Note that the forecastTime of the resulting time series is the same as the base
     * time series, so that the forecast time reflects what was in the submonthly file, but the data will be 12Z
     * aggregated starting from 24h before the time series start time.
     */
    public void computeDailySubmonthlyMap()
    {
        TimeSeriesArray aggTS = null; //Stores the aggregated ts.
        for(final LocationAndDataTypeIdentifier identifier: this._identifierToSubmonTSMap.keySet())
        {
            final List<TimeSeriesArray> submonTS = _identifierToSubmonTSMap.get(identifier);
            for(int i = 0; i < submonTS.size(); i++)
            {
                //Start time is first 12Z time after forecast time.  End time is first 12Z before end time.
                final long start = DateTools.computeTimeWithDesiredHourAfterBase(submonTS.get(i)
                                                                                         .getHeader()
                                                                                         .getForecastTime(), 12);
                final long end = DateTools.computeTimeWithDesiredHourBeforeBase(submonTS.get(i).getEndTime(), 12);
                try
                {
                    if(ParameterId.of(submonTS.get(i).getHeader()).isMax())
                    {
                        aggTS = AggregationTools.aggregateToMaximum(submonTS.get(i),
                                                                    start,
                                                                    end,
                                                                    "1 day",
                                                                    null,
                                                                    null,
                                                                    false,
                                                                    false);
                    }
                    else if(ParameterId.of(submonTS.get(i).getHeader()).isMin())
                    {
                        aggTS = AggregationTools.aggregateToMinimum(submonTS.get(i),
                                                                    start,
                                                                    end,
                                                                    "1 day",
                                                                    null,
                                                                    null,
                                                                    false,
                                                                    false);
                    }
                    else
                    {
                        throw new IllegalArgumentException("Submonthly time series parameter id, "
                            + submonTS.get(i).getHeader().getParameterId()
                            + ", is not valid to determine if min or max temperature.");
                    }

                    //Recover the original parameter id. The aggregation changes the parameter id.
                    ((DefaultTimeSeriesHeader)aggTS.getHeader()).setParameterId(submonTS.get(i)
                                                                                        .getHeader()
                                                                                        .getParameterId());
                }
                catch(final TimeSeriesAggregationException e)
                {
                    throw new IllegalStateException("Unable to aggregate submonthly time series to 24-hours.");
                }

                this._identifierToDailySubmonTSMap.put(identifier, aggTS);
            }
        }
    }

    /**
     * WARNING!!! This returns only the submonthly forecasts. It is included only so that the standard
     * {@link GriddedPEStepOptionsPanel} default view button operations can load the submonthly time series. DO NOT USE
     * THIS METHOD FOR ANY OTHER REASON! Call the methods {@link #getAllLoadedMonthlyEnsembles()},
     * {@link #getAllLoadedMonthlyMeanTimeSeries()}, {@link #getAllLoadedSubmonthlyTimeSeries()} instead.
     */
    @Override
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeries()
    {
        return getAllLoadedSubmonthlyTimeSeries();
    }

    public Collection<TimeSeriesArray> getAllLoadedSubmonthlyTimeSeries()
    {
        return _identifierToSubmonTSMap.values();
    }

    public Collection<TimeSeriesArrays> getAllLoadedMonthlyEnsembles()
    {
        return _identifierToMonEnsemblesMap.values();
    }

    public Collection<TimeSeriesArray> getAllLoadedMonthlyMeanTimeSeries()
    {
        return _identifierToMonMeanTSMap.values();
    }

    /**
     * This algorithm makes an assumption that the monthlytime series to which this must correspond will have a 0Z
     * forecast time (the data will be 12Z-12Z, but the forecast time will be 0Z; see {@link ProcessedCFSv2FileTools}).
     * That 0Z forecast time MUST be passed in as the forecastTime, here. That 0Z forecast time will be the time of the
     * first submonthly time series for a day, so that when the {@link LaggedEnsemble#LaggedEnsemble(long, List, int)}
     * takes over, it will correctly account for the five day skips.
     * 
     * @param forecastTime Forecast time for which to construct the submonthly ensembles. It must be the 0Z-0Z forecast
     *            time of the corresponding monthly lagged ensemble.
     * @return A {@link LaggedEnsemble} is constructed using the submonthly time series for the given identifier. The
     *         returned ensemble is what will be used for canonical events. Note that the forecast time will be 0Z
     *         based, not 12Z, if this matters (which it will for hindcasting).
     */
    public LaggedEnsemble constructSubmonthlyLaggedEnsembleForCanonicalEventComputation(final LocationAndDataTypeIdentifier identifier,
                                                                                        final ParameterId dataType,
                                                                                        final long forecastTime,
                                                                                        final int numberOfMembers)
    {
        List<TimeSeriesArray> submonTS = null;
        if(identifier.isTemperatureDataType())
        {
            submonTS = _identifierToDailySubmonTSMap.get(identifier);
        }
        else
        {
            submonTS = _identifierToSubmonTSMap.get(identifier);
        }
        if(dataType != null)
        {
            submonTS = TimeSeriesSorter.subListByParameterId(submonTS, dataType);
        }
        return new LaggedEnsemble(forecastTime, submonTS, numberOfMembers);
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        super.setDataHandlerBaseDirectory(directory);
        _preparedDataFileDirectory = new File(directory.getAbsolutePath() + "/processedASCIIGrids");
    }

    @Override
    public File getPreparedDataFilesDirectory()
    {
        return _preparedDataFileDirectory;
    }

    @Override
    public List<File> generateListOfPreparedDataFiles(final LocationAndDataTypeIdentifier identifier)
    {
        final CoordinateGridPoint cfsv2Pt = _gaussianGrid.findNearestPoint(new CoordinateGridPoint(identifier.getUsedLatitude(),
                                                                                                   identifier.getUsedLongitude()),
                                                                           true);

//        System.err.println("####>> --- " + identifier.buildStringToDisplayInTree() + " -- " + cfsv2Pt.toString());

        final List<File> results = new ArrayList<File>();
        if(cfsv2Pt == null) //Coordinates are outside of CONUS! -- shouldn't happen unless its APRFC
        {
            //GeneralTools.dumpStackTrace();
            return results;
        }

        //The files
        final String latLonStr = constructLatLonComponentOfFileNames(cfsv2Pt);
        if(identifier.isPrecipitationDataType())
        {
            File forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_psubmon/PRCP_6HR_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
            forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_pmon/PSUM_30D_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
        }
        else if(identifier.isTemperatureDataType())
        {
            File forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_tsubmon/TMAX_6HR_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
            forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_tmon/TMAX_30D_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
            forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_tsubmon/TMIN_6HR_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
            forecastFile = new File(getPreparedDataFilesDirectory().getAbsolutePath() + "/cfs_tmon/TMIN_30D_"
                + latLonStr + ".CFSv2.gz");
            results.add(forecastFile);
        }
        return results;
    }

    @Override
    public void loadPreparedTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        clearLoadedTimeSeries();

        LOG.info("Loading CFSv2 hindcasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            final List<File> forecastFiles = generateListOfPreparedDataFiles(identifier);
            if(identifier.isPrecipitationDataType())
            {
                loadPrecipitationPreparedTimeSeries(identifier, forecastFiles.get(0), forecastFiles.get(1));
            }
            else if(identifier.isTemperatureDataType())
            {
                //Tmax
                loadTemperaturePreparedTimeSeries(false, identifier, forecastFiles.get(0), forecastFiles.get(1));

                //Tmin
                loadTemperaturePreparedTimeSeries(true, identifier, forecastFiles.get(2), forecastFiles.get(3));
            }
        }
        LOG.info("Done loading CFSv2 hindcasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        //This method is used when computing canonical events in a default manner.  However, for CFSv2, the events are computed
        //using a method specific to CFSv2.  Therefore, for this data handler, nothing needs to be done.  Normally, it would need
        //to return the lagged ensemble mean after combining monthly and submonthly data. 
        return Lists.newArrayList();
    }

    @Override
    public void clearLoadedTimeSeries()
    {
        _identifierToMonEnsemblesMap.clear();
        _identifierToMonMeanTSMap.clear();
        _identifierToSubmonTSMap.clear();
        clearDailySubmonthlyMap();
    }

    @Override
    public Collection<LocationAndDataTypeIdentifier> getIdentifiersWithData()
    {
        return this._identifierToSubmonTSMap.keySet();
    }

    @Override
    public void prepareDataFiles() throws Exception
    {
        throw new Exception("CFSv2DataHandler does not prepare data.  "
            + "The processed ASCII grid files are prepared and delivered by OHD.");
    }

    @Override
    public boolean havePreparedDataFilesBeenCreatedAlready(final LocationAndDataTypeIdentifier identifier)
    {
        final List<File> files = generateListOfPreparedDataFiles(identifier);
        if(files.isEmpty())
        {
            return false;
        }
        for(final File file: generateListOfPreparedDataFiles(identifier))
        {
            if(!file.exists())
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean arePreparedDataFilesUpToDate(final LocationAndDataTypeIdentifier identifier)
    {
        //No up-to-date checking needed.
        return true;
    }

    @Override
    public Collection<TimeSeriesArray> loadHindcastTimeSeries(final LocationAndDataTypeIdentifier identifier,
                                                              final ParameterId dataTypeToLoad,
                                                              final long forecastTime) throws Exception
    {
        //Load reforecast time series
        loadPreparedTimeSeries(Lists.newArrayList(identifier));

        //The monthly time series have 0Z forecast times, but 12Z-12Z data.  For hindcasting, we need to put that data on the correct 
        //0Z-0Z clock implied by the forecast time.  Hence, I need to shift all time series just read in.  However, I need to make 
        //sure that the forecastTime is not changed, so I shift the forecast time back so that its a 0Z-0Z system.
        for(final TimeSeriesArrays monthlyEns: getAllLoadedMonthlyEnsembles())
        {
            TimeSeriesArraysTools.shift(monthlyEns, -1
                * ProcessedCFSv2FileTools.MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT);
            for(int i = 0; i < monthlyEns.size(); i++)
            {
                ((DefaultTimeSeriesHeader)monthlyEns.get(i).getHeader()).setForecastTime(monthlyEns.get(i)
                                                                                                   .getHeader()
                                                                                                   .getForecastTime()
                    + ProcessedCFSv2FileTools.MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT);
            }
        }

        //XXX This should be doable as a binary search, though it would make things a bit more complicated to understand.
        for(final TimeSeriesArrays monthlyEns: getAllLoadedMonthlyEnsembles())
        {
//DEBUG: This if clause is triggered when the desired year is found.  Add a breakpoint on the System.out to stop at the time and step through.
//            if(HCalendar.computeCalendarFromMilliseconds(monthlyEns.get(0).getHeader().getForecastTime())
//                        .get(Calendar.YEAR) == 1993)
//            {
//                System.out.println("####>> HERE!!! ");
//            }

            //See ProcessedCFSv2FileTools.readMonthlyProcessedASCIIForecastFile... It turns out that the header time is on a 0Z clock, but the data
            //is viewed as 12Z-12Z (same day, 12 hours later).  Hence, here, the actually forecast time I want to use to compute lag is the 
            //header time + 12 hours This will not work right if the ASCII files are not 0Z time series for monthly data!
            final long monthlyForecastTime = monthlyEns.get(0).getHeader().getForecastTime();
            final long lag = forecastTime
                - (monthlyForecastTime + ProcessedCFSv2FileTools.MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT);

//            final String forecastTimeStr = HCalendar.buildDateTimeTZStr(forecastTime);
//            if(forecastTimeStr.contains("1993-05"))
//            {
//                System.out.println("####>> CHECKING -- "
//                    + HCalendar.buildDateTimeTZStr(forecastTime)
//                    + " --- "
//                    + HCalendar.buildDateTimeTZStr(monthlyForecastTime)
//                    + " ---> "
//                    + HCalendar.buildDateTimeTZStr(monthlyForecastTime
//                        + ProcessedCFSv2FileTools.MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT));
//            }

            //If the parameterId check passes (necessary for temperature tmin/tmax data) and the lag is within 5 days of the needed date (not inclusive), then...
            if((ParameterId.of(monthlyEns.get(0).getHeader()).equals(dataTypeToLoad))
                && ((lag >= 0) && (lag < 5 * 24 * HCalendar.MILLIS_IN_HR)))
            {
                final LaggedEnsemble monthlyLaggedEns = new LaggedEnsemble(TimeSeriesArraysTools.convertTimeSeriesArraysToList(monthlyEns));

                //Find the submonthly lagged ensemble to use.
                LaggedEnsemble submonLaggedEns = null;
                try
                {
                    //This should yield an exact match, if not then the submonthly is not synchronized with the monthly, which is bad.
                    submonLaggedEns = constructSubmonthlyLaggedEnsembleForCanonicalEventComputation(identifier,
                                                                                                    dataTypeToLoad,
                                                                                                    monthlyForecastTime,
                                                                                                    CFSv2ForecastSource.LAGGED_ENSEMBLE_SIZE);
                }
                catch(final IllegalArgumentException e)
                {
                    LOG.warn("Unable to constructed submonthly lagged ensemble for "
                        + HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATETZ_FORMAT) + " and data type "
                        + dataTypeToLoad + ": " + e.getMessage());
                    continue;
                }

                //Verify that the number of members is identical (should never be a problem, but just in case...).
                if(monthlyLaggedEns.size() != submonLaggedEns.size())
                {
                    throw new Exception("For forecast time "
                        + HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATETZ_FORMAT) + " and data type "
                        + dataTypeToLoad + ", the number of submonthly members loaded, " + submonLaggedEns.size()
                        + ", does not equal the number of monthly members loaded, " + monthlyLaggedEns.size() + ".");
                }

                //Disaggregate each monthly member to the time step of the corresponding submonthly member and use it to extend
                //the submonthly time series to the full length of the monthly.
                final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();
                for(int i = 0; i < monthlyLaggedEns.size(); i++)
                {
                    results.add(TimeSeriesArrayTools.copyTimeSeries(submonLaggedEns.get(i)));
                    final TimeSeriesArray disaggregatedMonthlyTS = AggregationTools.disaggregateAccumulativeTimeSeries(monthlyLaggedEns.get(i),
                                                                                                                       monthlyForecastTime,
                                                                                                                       submonLaggedEns.get(i)
                                                                                                                                      .getHeader()
                                                                                                                                      .getTimeStep()
                                                                                                                                      .getStepMillis());
                    TimeSeriesArrayTools.extendFromOtherTimeSeries(results.get(i), disaggregatedMonthlyTS);
                }

                //Return the extended submonthly ensemble.  The submonthly ensemble at this point is a 0Z ensemble with 0Z-0Z data.
                //We want it to be a 12Z-12Z ensemble and forecast time.
                TimeSeriesArraysTools.shift(results,
                                            +ProcessedCFSv2FileTools.MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT);
                return new LaggedEnsemble(results);
            }
        }

        throw new Exception("Unable to find a monthly time series for "
            + HCalendar.buildDateStr(forecastTime, HCalendar.DEFAULT_DATETZ_FORMAT) + " and data type "
            + dataTypeToLoad + ".");
    }
}
