package ohd.hseb.hefs.mefp.sources.cfsv2;

import java.io.File;

import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.GenericMEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId;

/**
 * NOTE: CFSv2 prate data is the 6h average rate.
 * 
 * @author hank.herr
 */
public class CFSv2ForecastSource extends AbstractMEFPForecastSource
{
    /**
     * This ===MUST=== match the portion of the source name before 'ForecastSource', so that it matches the default
     * return of {@link #getSourceId()}.
     */
    public final static String SOURCE_ID = "CFSv2";
    public final static int LAGGED_ENSEMBLE_SIZE = 16;

    @Override
    public String getName()
    {
        return "CFS Forecasts";
    }

    @Override
    public MEFPSourceModelParameters getSourceModelParameters(final LocationAndDataTypeIdentifier identifier,
                                                              final AlgorithmModelParameters algorithmParameters)
    {
        final MEFPSourceModelParameters parameters = new MEFPSourceModelParameters(identifier,
                                                                                   this,
                                                                                   algorithmParameters,
                                                                                   false);
        return parameters;
    }

    @Override
    public CFSv2DataHandler getSourceDataHandler()
    {
        return (CFSv2DataHandler)super.getSourceDataHandler();
    }

    @Override
    protected void setSourceDataHandler(final SourceDataHandler handler)
    {
        if(!(handler instanceof CFSv2DataHandler))
        {
            throw new IllegalArgumentException("The SourceDataHandler is not a CFSv2DataHandler.");
        }
        super.setSourceDataHandler(handler);
    }

    @Override
    public MEFPSourceControlOptions createControlOptions(final ParameterId.Type type)
    {
        switch(type)
        {
            case TEMPERATURE:
                return new GenericMEFPSourceControlOptions("cfsv2TemperatureControlOptions", this);
            case PRECIPITATION:
                return new MEFPPrecipitationSourceControlOptions("cfsv2PrecipitationControlOptions", this);
            default:
                throw new IllegalArgumentException(String.format("%s not supported.", type));
        }
    }

    @Override
    public void initializeSourceDataHandler(final File baseDirectory, final NoticePoster poster) throws Exception
    {
        this.setSourceDataHandler(new CFSv2DataHandler(baseDirectory));
    }

    @Override
    public SourceCanonicalEventValues constructSourceCanonicalEventValues(final CanonicalEventList eventsToCompute)
    {
        return new CFSv2SourceCanonicalEventValues(this, eventsToCompute);
    }

    /**
     * Operationally, the CFSv2 forecast available at a 12Z time is the one from 24-hours (or more) ago. To account for
     * this in estimating parameters, the 12Z reforecast monthly/submonthly events must be lined up with observed data
     * from 24-hours AFTER the first 12Z value after the CFSv2 reforecast time, since that is how it will be applied
     * operationally. Hence return 24-hours.<br>
     * <br>
     * BUT... the reforecast monthly files are provided for 9 30-day months EXACTLY after the CFSv2 reforecast T0. If I
     * lag any, at all, the events that include day 270 will be incomplete (after the end of month 9) and values for
     * that event cannot be computed. Hence, even though 24-hours is technically correct, we have to use a lag of 0.
     * This lag also cannot be applied when finding hindcasts, because, again, the last event will not be computable.
     */
    @Override
    public int getOperationalLagInHoursWhenEstimatingParameters(final LocationAndDataTypeIdentifier identifier)
    {
        //XXX I've removed this after talking with Limin.  The 24-hour CFSv2 lag exists for both precip and temp.  The Fortran appears to only account
        //for the lag in precip, which is incorrect.
//        if(identifier.isTemperatureDataType())  
//        {
//            return 0;
//        }
        return 0;
    }

    /**
     * @return See the superclass method. The operational lag when hindcasting in this case is the same as that when
     *         estimating parameters, and that is what the super class assumes. I'm including this method here
     *         overridden solely so that I can add a comment emphasizing the fact that I have thought about it and came
     *         to this conclusion.
     */
    @Override
    public int getOperationalLagInHoursWhenAcquringReforecast(final LocationAndDataTypeIdentifier identifier)
    {
        return super.getOperationalLagInHoursWhenAcquringReforecast(identifier);
    }

    @Override
    public int getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration()
    {
        return LAGGED_ENSEMBLE_SIZE;
    }

    @Override
    public ParameterEstimatorStepProcessor getSourceStepProcessor(final ParameterEstimatorRunInfo runInfo)
    {
        return new CFSv2PEStepProcessor((MEFPParameterEstimatorRunInfo)runInfo);
    }
}
