package ohd.hseb.hefs.mefp.sources.cfsv2;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.GriddedPEStepOptionsPanel;
import ohd.hseb.hefs.pe.core.ParameterEstimatorDiagnosticPanel;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.gui.help.HelpFile;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.jobs.GenericJob;
import ohd.hseb.hefs.utils.jobs.HJobMonitorDialog;
import ohd.hseb.hefs.utils.jobs.JobListener;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import com.google.common.collect.Lists;

/**
 * A {@link GriddedPEStepOptionsPanel} that includes an extra button in the toolbar: {@link #_viewMonthlyButton}. It is
 * compatible with the {@link CFSv2PEStepProcessor}.
 * 
 * @author hank.herr
 */
@HelpFile("PDFHELP:helpManual.pdf#GEFSSubpanelRef")
public class CFSv2PEStepOptionsPanel extends GriddedPEStepOptionsPanel
{
    private static final long serialVersionUID = 1L;

    private JButton _viewMonthlyButton;

    public CFSv2PEStepOptionsPanel(final MEFPParameterEstimatorRunInfo runInfo, final CFSv2PEStepProcessor stepProcessor)
    {
        super(runInfo, stepProcessor);
    }

    @SuppressWarnings("serial")
    @Override
    protected void initializeDisplay()
    {
        _viewMonthlyButton = new SelfListeningButton("chartM20x20", "View Forecasts")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                viewSelectedMonthlyCFSv2Hindcasts();
            }
        };
        super.initializeDisplay();
    }

    private void viewSelectedMonthlyCFSv2Hindcasts()
    {
        final List<LocationAndDataTypeIdentifier> identifiers = new ArrayList<LocationAndDataTypeIdentifier>();
        final LocationAndDataTypeIdentifier selectedIdent = getSelectedIdentifier();
        if(selectedIdent == null)
        {
            return;
        }
        identifiers.add(selectedIdent);

        final GenericJob buildPanelJob = new GenericJob()
        {
            @Override
            public void processJob()
            {
                setIndeterminate(true);
                updateNote("Loading time series from source...");

                final Collection<TimeSeriesArray> tsc = new ArrayList<TimeSeriesArray>();
                final Collection<TimeSeriesArrays> ensembles = new ArrayList<TimeSeriesArrays>();
                try
                {
                    getHandler().loadPreparedTimeSeries(identifiers);
                    tsc.addAll(getHandler().getAllLoadedMonthlyMeanTimeSeries());

                    ensembles.addAll(getHandler().getAllLoadedMonthlyEnsembles());
                    final TimeSeriesArrays arrays = TimeSeriesArraysTools.convertTimeSeriesArraysCollections(ensembles);
                    tsc.addAll(TimeSeriesArraysTools.convertTimeSeriesArraysToList(arrays));
                }
                catch(final Exception e)
                {
                    e.printStackTrace();
                    fireProcessJobFailure(new Exception("Error loading CFSv2 monthly time series for location "
                        + selectedIdent.buildStringToDisplayInTree() + ": " + e.getMessage()), true);
                    return;
                }

                updateNote("Constructing chart...");
                final CFSv2MonthlyDiagnosticPanel panel = new CFSv2MonthlyDiagnosticPanel(getSelectedIdentifier(), tsc);
                fireDiagnostic(panel);

                endTask();
            }
        };
        final HJobMonitorDialog jobDialog = new HJobMonitorDialog(this,
                                                                  "Building Chart to Display CFSv2 Monthly Reforecasts",
                                                                  buildPanelJob,
                                                                  false);
        buildPanelJob.addListener(new JobListener()
        {
            @Override
            public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
            {
                //exc.printStackTrace();
                fireDiagnostic("Unable to build chart displaying CFSv2 monthly reforecasts:", exc.getMessage());
                JOptionPane.showMessageDialog(SwingTools.getGlobalDialogParent(CFSv2PEStepOptionsPanel.this),
                                              StringTools.wordWrap(exc.getMessage(), 80),
                                              "Unable to Build Diagnostic Panel",
                                              JOptionPane.ERROR_MESSAGE);
            }

            @Override
            public void processSuccessfulJobCompletion(final GenericJob theJob)
            {
            }
        });
        buildPanelJob.startJob();
        jobDialog.setMinimumSize(new Dimension(350, 10));
        jobDialog.setModal(true);
        jobDialog.setVisible(true);
    }

    @Override
    public CFSv2DataHandler getHandler()
    {
        return (CFSv2DataHandler)getSource().getSourceDataHandler();
    }

    @Override
    protected void updateEnablednessOfViewButton()
    {
        super.updateEnablednessOfViewButton();

        final LocationAndDataTypeIdentifier identifier = getSelectedIdentifier();
        if(identifier != null)
        {
            _viewMonthlyButton.setEnabled(getHandler().areMonthlyFilesPresent(identifier));
            if(_viewMonthlyButton.isEnabled())
            {
                _viewMonthlyButton.setToolTipText("View the CFSv2 monthly hindcasts for "
                    + identifier.buildStringToDisplayInTree() + ".");
            }
            else
            {
                _viewMonthlyButton.setToolTipText("<html>No CFSv2 monthly files are available for "
                    + identifier.buildStringToDisplayInTree()
                    + ".<br>Click the Download Files Button to download them.</html>");
            }
        }
        else
        {
            _viewMonthlyButton.setEnabled(false);
            _viewMonthlyButton.setToolTipText("Select one row to view the CFSv2 monthly hindcasts for that location and data type.");
        }
    }

    @Override
    protected List<Component> buildAdditionalToolBarComponents()
    {
        return Lists.newArrayList((Component)_viewMonthlyButton);
    }

    @Override
    protected ParameterEstimatorDiagnosticPanel instantiateDiagnosticPanel(final LocationAndDataTypeIdentifier identifier,
                                                                           final Collection<TimeSeriesArray> tsc)
    {
        return new CFSv2SubmonthlyDiagnosticPanel(identifier, tsc);
    }

    @Override
    protected String getSourceTimeSeriesShortName()
    {
        return "CFSv2 submonthly";
    }
}
