package ohd.hseb.hefs.mefp.sources.cfsv2;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.LaggedEnsemble;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * CFSv2 requires a special
 * {@link #computeCanonicalEventValues(LocationAndDataTypeIdentifier, TimeSeriesArray, SourceDataHandler, int, int)}
 * method because it must combine monthly and submonthly time series. This will use submonthly time series for computing
 * events for which the end time is 30 days or less. Monthly time series are used for events where the end time is
 * larger than 30 days. Monthly and submonthly data are never mixed: if monthly is to be used, it will be used for all
 * time steps of the canonical event, even those before 30 days.
 * 
 * @author hank.herr
 */
public class CFSv2SourceCanonicalEventValues extends SourceCanonicalEventValues
{
    private static final Logger LOG = LogManager.getLogger(CFSv2SourceCanonicalEventValues.class);
    public final static int NUMBER_OF_DAYS_FOR_SUBMONTHLY = 30;

    public CFSv2SourceCanonicalEventValues(final MEFPForecastSource forecastSource,
                                           final CanonicalEventList eventsToCompute)
    {
        super(forecastSource, eventsToCompute);
    }

    @Override
    public void computeCanonicalEventValues(final LocationAndDataTypeIdentifier identifier,
                                            final ParameterId dataType,
                                            final TimeSeriesArray observedTS,
                                            final SourceDataHandler handler)
    {
        final CFSv2DataHandler cfsv2Handler = (CFSv2DataHandler)handler;
        final List<LaggedEnsemble> submonTimeSeriesUsed = new ArrayList<LaggedEnsemble>();
        final List<LaggedEnsemble> monthlyTimeSeriesUsed = new ArrayList<LaggedEnsemble>();

        //For temperature data, the daily submonthly map must be populated.
        if(identifier.isTemperatureDataType())
        {
            cfsv2Handler.clearDailySubmonthlyMap();
            cfsv2Handler.computeDailySubmonthlyMap();
        }

        //For each monthly ensemble found, we need to construct a corresponding submonthly ensemble.
        for(final TimeSeriesArrays monthlyEns: cfsv2Handler.getAllLoadedMonthlyEnsembles())
        {
            //Skip the ensemble if it does not match the desired parameter type.
            if((dataType != null) && (!ParameterId.of(monthlyEns.get(0).getHeader()).equals(dataType)))
            {
                continue;
            }

            //For monthly data, the forecast times are 0Z, but the data is tied into 12Z times 
            //(i.e, the start time of the first value will be 12Z).  The 0Z forecast time will match the
            //forecast time required of the first submonthly ensemble member (also 0Z).  12Z-12Z day
            //used in computations for the submon ensemble is then enforced within the canonical event
            //calculations, which run on a 12Z clock.
            final long forecastTime = monthlyEns.get(0).getHeader().getForecastTime();

            //In the Fortran code, a submonthly ensemble is only used to compute any events
            //if it contains a full 16 ensemble members, even if the event uses less than 16.
            //This will mimic that.  
            //If the lagged ensemble cannot be constructed, continue to the next monthlyEns.
            LaggedEnsemble submonEns = null;
            try
            {
                submonEns = cfsv2Handler.constructSubmonthlyLaggedEnsembleForCanonicalEventComputation(identifier,
                                                                                                       dataType,
                                                                                                       forecastTime,
                                                                                                       CFSv2ForecastSource.LAGGED_ENSEMBLE_SIZE);
            }
            catch(final IllegalArgumentException e)
            {
                LOG.warn("Unable to constructed lagged ensemble for " + new Date(forecastTime) + ": " + e.getMessage());
                continue;
            }

            //Add to the first and second time series lists.  At this point, I know submonEns cannot be null, because
            //we have continued if there is an exception.
            submonTimeSeriesUsed.add(submonEns);
            monthlyTimeSeriesUsed.add(new LaggedEnsemble(TimeSeriesArraysTools.convertTimeSeriesArraysToList(monthlyEns)));
        }

        //Compute the events.
        final int submonEventCount = getForecastValues().getEventsToCompute()
                                                        .determineNumberOfApplicableEvents(NUMBER_OF_DAYS_FOR_SUBMONTHLY);
        getForecastValues().computeEvents(submonEventCount,
                                          submonTimeSeriesUsed,
                                          monthlyTimeSeriesUsed,
                                          getForecastSource().getOperationalLagInHoursWhenEstimatingParameters(identifier));
        getObservedValues().computeEvents(observedTS, getForecastValues().getComputationalTimes(), 0);
    }
}
