package ohd.hseb.hefs.mefp.sources.cfsv2;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.zip.GZIPInputStream;

import nl.wldelft.fews.pi.PiTimeSeriesHeader;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * This assumes that all units are metric in the files. I does NOT process the files to find the units. Note that for
 * monthly data, the forecast times for the time series will be 0Z, while the data will use 12Z-12Z days. This is done
 * on purpose to allow for canonical event computation.
 * 
 * @author Hank.Herr
 */
public abstract class ProcessedCFSv2FileTools
{
    public static String MONTHLY_ENSEMBLE_ID = "CFSv2 Monthly";

    public static long MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT = 12 * HCalendar.MILLIS_IN_HR;

    private static DefaultTimeSeriesHeader prepareSubmonthlyHeader(final LocationAndDataTypeIdentifier identifier,
                                                                   final String dateString,
                                                                   final String parameterId,
                                                                   final String unitStr)
    {
        //Making sure minutes and seconds are set to 0.
        final Calendar t0 = HCalendar.convertStringToCalendar(dateString + " 00:00", "CCYYMMDDhh mm:ss");

        //DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)timeSeriesToPrepare.getHeader();
        final DefaultTimeSeriesHeader editableHeader = new DefaultTimeSeriesHeader();

        editableHeader.setForecastTime(t0.getTimeInMillis());
        editableHeader.setLocationDescription(identifier.getLocationId());
        editableHeader.setLocationId(identifier.getLocationId());
        editableHeader.setLocationName(identifier.getLocationId());

        editableHeader.setParameterId(parameterId);
        editableHeader.setParameterName(parameterId);
        editableHeader.setParameterType(ParameterType.ACCUMULATIVE);
        editableHeader.setUnit(unitStr);

        //editableHeader.setEnsembleId("EVENTS");
        //editableHeader.setEnsembleMemberIndex(eventIndex);

        return editableHeader;
    }

    private static DefaultTimeSeriesHeader prepareMonthlyHeader(final LocationAndDataTypeIdentifier identifier,
                                                                final String dateString,
                                                                final String parameterId,
                                                                final String unitStr,
                                                                final int ensembleMemberIndex)
    {
        //Making sure minutes and seconds are set to 0.  The hour of day should always be 0Z for monthly data.
        final Calendar t0 = HCalendar.convertStringToCalendar(dateString + " 00:00", "CCYYMMDDhh mm:ss");

        //DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)timeSeriesToPrepare.getHeader();
        final PiTimeSeriesHeader editableHeader = new PiTimeSeriesHeader();

        editableHeader.setForecastTime(t0.getTimeInMillis());
        editableHeader.setLocationDescription(identifier.getLocationId());
        editableHeader.setLocationId(identifier.getLocationId());
        editableHeader.setLocationName(identifier.getLocationId());

        editableHeader.setParameterId(parameterId);
        editableHeader.setParameterName(parameterId);
        editableHeader.setParameterType(ParameterType.ACCUMULATIVE);
        editableHeader.setUnit(unitStr);

        editableHeader.setEnsembleId(MONTHLY_ENSEMBLE_ID);
        editableHeader.setEnsembleMemberIndex(ensembleMemberIndex);

        return editableHeader;
    }

    public static List<TimeSeriesArray> readSubmonthlyProcessedASCIIForecastFile(final LocationAndDataTypeIdentifier identifier,
                                                                                 final String assignedParameterId,
                                                                                 final String assignedUnitStr,
                                                                                 final int numberFieldWidth,
                                                                                 final File gzipForecastFile) throws Exception
    {
        final GZIPInputStream gzipInputStream = new GZIPInputStream(new FileInputStream(gzipForecastFile));
        final InputStreamReader reader = new InputStreamReader(gzipInputStream);
        final BufferedReader buffReader = new BufferedReader(reader);

        //Segmented line positions
        final int[] positions = new int[242];
        positions[0] = 10;
        positions[1] = 13;
        for(int i = 2; i < positions.length; i++)
        {
            positions[i] = positions[i - 1] + numberFieldWidth;
        }

        String line = null;
        final List<TimeSeriesArray> allTS = new ArrayList<TimeSeriesArray>();

        try
        {
            //First line is a header line, but I don't think I need to do anything with it.
            line = buffReader.readLine();

            //While I have lines to read...
            while(((line = buffReader.readLine()) != null) && (!line.trim().isEmpty()))
            {
                final SegmentedLine segLine = new SegmentedLine(line, positions);

                //Skip empty lines, which may be at the end of the file.
                if(segLine.getNumberOfSegments() <= 0)
                {
                    continue;
                }

                //Prepare a header, time step, and ts.  Populate it based on segLine and add it to allTS.
                final DefaultTimeSeriesHeader header = prepareSubmonthlyHeader(identifier,
                                                                               segLine.getSegment(0),
                                                                               assignedParameterId,
                                                                               assignedUnitStr);
                header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 6));
                final TimeSeriesArray ts = new TimeSeriesArray(header);
                for(int i = 2; i < segLine.getNumberOfSegments(); i++)
                {
                    final float number = Float.parseFloat(segLine.getSegment(i).trim()) / 100f;

                    //Assumes -90 or less is missing and stores as Float.NaN.
                    if((number <= -90.0f) || (Float.isNaN(number)))
                    {
                        ts.putValue(header.getForecastTime() + (i - 1) * header.getTimeStep().getStepMillis(),
                                    Float.NaN);
                    }
                    else
                    {
                        ts.putValue(header.getForecastTime() + (i - 1) * header.getTimeStep().getStepMillis(), number);
                    }
                }
                allTS.add(ts);
            }
        }
        finally
        {
            buffReader.close();
        }

        return allTS;
    }

    public static List<TimeSeriesArray> readPrecipitationSubmonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                        final File gzipForecastFile) throws Exception
    {
        return readSubmonthlyProcessedASCIIForecastFile(identifier,
                                                        HEFSTools.FORECAST_PRECIP_PARAMETER_ID,
                                                        "MM",
                                                        8,
                                                        gzipForecastFile);
    }

    public static List<TimeSeriesArray> readMaxTemperatureSubmonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                         final File gzipForecastFile) throws Exception
    {
        final List<TimeSeriesArray> results = readSubmonthlyProcessedASCIIForecastFile(identifier,
                                                                                       HEFSTools.FORECAST_TMAX_PARAMETER_ID,
                                                                                       "DEGC",
                                                                                       6,
                                                                                       gzipForecastFile);
        return results;
    }

    public static List<TimeSeriesArray> readMinTemperatureSubmonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                         final File gzipForecastFile) throws Exception
    {
        final List<TimeSeriesArray> results = readSubmonthlyProcessedASCIIForecastFile(identifier,
                                                                                       HEFSTools.FORECAST_TMIN_PARAMETER_ID,
                                                                                       "DEGC",
                                                                                       6,
                                                                                       gzipForecastFile);
        return results;
    }

    public static List<TimeSeriesArray> readTemperatureSubmonthlyFile(final boolean minimumTemperature,
                                                                      final LocationAndDataTypeIdentifier identifier,
                                                                      final File gzipForecastFile) throws Exception
    {
        if(minimumTemperature)
        {
            return readMinTemperatureSubmonthlyFile(identifier, gzipForecastFile);
        }
        else
        {
            return readMaxTemperatureSubmonthlyFile(identifier, gzipForecastFile);
        }
    }

    public static List<TimeSeriesArrays> readMonthlyProcessedASCIIForecastFile(final LocationAndDataTypeIdentifier identifier,
                                                                               final String assignedParameterId,
                                                                               final String assignedUnitStr,
                                                                               final int numberFieldWidth,
                                                                               final File gzipForecastFile) throws Exception
    {
        final GZIPInputStream gzipInputStream = new GZIPInputStream(new FileInputStream(gzipForecastFile));
        final InputStreamReader reader = new InputStreamReader(gzipInputStream);
        final BufferedReader buffReader = new BufferedReader(reader);

        //Segmented line positions
        final int[] positions = new int[11];
        positions[0] = 10;
        positions[1] = 13;
        for(int i = 2; i < positions.length; i++)
        {
            positions[i] = positions[i - 1] + numberFieldWidth;
        }

        String line = null;
        final List<TimeSeriesArrays> allTSs = new ArrayList<TimeSeriesArrays>();
        TimeSeriesArrays workingTSs = null;
        String workingDateStr = null;

        try
        {
            //First line is a header line, but I don't think I need to do anything with it.
            line = buffReader.readLine();

            //While I have lines to read...
            while(((line = buffReader.readLine()) != null) && (!line.trim().isEmpty()))
            {
                final SegmentedLine segLine = new SegmentedLine(line, positions);

                //Skip empty lines, which may be at the end of the file.
                if(segLine.getNumberOfSegments() <= 0)
                {
                    continue;
                }

                //Member index
                int memberIndex = -1;
                try
                {
                    memberIndex = Integer.parseInt(segLine.getSegment(1).trim());
                }
                catch(final NumberFormatException e)
                {
                    throw new Exception("Second component is not valid member index in this line: '" + line + "'.");
                }

                //Set the date string, assuming that member 1 is always the first member of each lagged ensemble 
                //listed in the file.
                if(memberIndex == 1)
                {
                    workingDateStr = segLine.getSegment(0);
                }

                //Prepare a header, time step, and ts.  Populate it based on segLine and add it to allTS.
                final DefaultTimeSeriesHeader header = prepareMonthlyHeader(identifier,
                                                                            workingDateStr,
                                                                            assignedParameterId,
                                                                            assignedUnitStr,
                                                                            memberIndex);
                header.setTimeStep(SimpleEquidistantTimeStep.getInstance(HCalendar.MILLIS_IN_HR * 24 * 30));
                final TimeSeriesArray ts = new TimeSeriesArray(header);
                for(int i = 2; i < segLine.getNumberOfSegments(); i++)
                {
                    final float number = Float.parseFloat(segLine.getSegment(i).trim()) / 100f;

                    //Assumes -90 or less is missing and stores as Float.NaN.
                    //============================= T0 NOTE ==============================
                    //NOTE: the header forecast time matches what is in the file: always a 0Z time of day.
                    //However, when the monthly values are used, they must be used on a 12Z - 12Z clock.  To handle
                    //this, we add 12 extra hours to the hour of the day.  This means the first value in the time series
                    //is actually 30days 12hours after T0, not just 30 days.  This will NOT work properly if time of day
                    //provided by the ASCII files is not 0Z.  We do this in order to guarantee that the monthly ts
                    //has the same T0 as the corresponding submonthly TS that is the first member of the lagged ensemble.
                    //====================================================================
                    if((number <= -90.0f) || (Float.isNaN(number)))
                    {
                        ts.putValue(header.getForecastTime() + (i - 1) * header.getTimeStep().getStepMillis()
                            + MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT, Float.NaN);
                    }
                    else
                    {
                        ts.putValue(header.getForecastTime() + (i - 1) * header.getTimeStep().getStepMillis()
                            + MONTHLY_HEADER_FORECAST_TIME_TO_VIEWED_FORECAST_TIME_ADJUSTMENT, number);
                    }
                }

                //Check the member index and re-initialize the working arrays if needed.
                if(memberIndex == 1)
                {
                    workingTSs = new TimeSeriesArrays(ts);
                    allTSs.add(workingTSs);
                }
                else
                {
                    workingTSs.add(ts);
                }
            }
        }
        finally
        {
            buffReader.close();
        }

        return allTSs;
    }

    public static List<TimeSeriesArrays> readPrecipitationMonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                      final File gzipForecastFile) throws Exception
    {
        return readMonthlyProcessedASCIIForecastFile(identifier,
                                                     HEFSTools.FORECAST_PRECIP_PARAMETER_ID,
                                                     "MM",
                                                     8,
                                                     gzipForecastFile);
    }

    public static List<TimeSeriesArrays> readMaxTemperatureMonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                       final File gzipForecastFile) throws Exception
    {
        return readMonthlyProcessedASCIIForecastFile(identifier,
                                                     HEFSTools.FORECAST_TMAX_PARAMETER_ID,
                                                     "DEGC",
                                                     8,
                                                     gzipForecastFile);
    }

    public static List<TimeSeriesArrays> readMinTemperatureMonthlyFile(final LocationAndDataTypeIdentifier identifier,
                                                                       final File gzipForecastFile) throws Exception
    {
        return readMonthlyProcessedASCIIForecastFile(identifier,
                                                     HEFSTools.FORECAST_TMIN_PARAMETER_ID,
                                                     "DEGC",
                                                     8,
                                                     gzipForecastFile);
    }

    public static List<TimeSeriesArrays> readTemperatureMonthlyFile(final boolean minimumTemperature,
                                                                    final LocationAndDataTypeIdentifier identifier,
                                                                    final File gzipForecastFile) throws Exception
    {
        if(minimumTemperature)
        {
            return readMinTemperatureMonthlyFile(identifier, gzipForecastFile);
        }
        else
        {
            return readMaxTemperatureMonthlyFile(identifier, gzipForecastFile);
        }
    }

}
