package ohd.hseb.hefs.mefp.sources.gefs;

import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.graphgen.utils.GraphGenCalculatorTools;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPSourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.geo.CoordinateGridPoint;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

/**
 * GEFS grids are 1 degree grids. Just use rounding to find closest point. For testing using lx4, the files must be
 * under /fs/hefs/testdata/ftp/processedASCIIGrids. The directory names are gefs_ascii_[type]_days1to8 or days9to16.
 * Links are used: to create the links see the file ./createGEFSLinks.sh in the grids directory.
 * 
 * @author hankherr
 */
public class GEFSDataHandler extends AbstractMEFPSourceDataHandler
{
    private static final Logger LOG = LogManager.getLogger(GEFSDataHandler.class);

    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArrays> _identifierToEnsembleMap;
    private final Multimap<LocationAndDataTypeIdentifier, TimeSeriesArray> _identifierToEnsembleMeanMap;

    /**
     * Stores the location of the processed ASCII file directories
     */
    private File _preparedDataFileDirectory = null;

//    private final CoordinateGrid _gaussianGrid190Days9to16;
//    private final CoordinateGrid _gaussianGrid254Days1to8;

    private int _testNumberToRead = -1;

    public GEFSDataHandler() throws Exception
    {
//        _gaussianGrid190Days9to16 = CoordinateSystemFilesHandler.GAUSSIAN_T190_COORDINATES.loadCoordinates();
//        _gaussianGrid254Days1to8 = CoordinateSystemFilesHandler.GAUSSIAN_T254_COORDINATES.loadCoordinates();

        _identifierToEnsembleMap = ArrayListMultimap.create();
        _identifierToEnsembleMeanMap = ArrayListMultimap.create();
    }

    public GEFSDataHandler(final File baseDirectory) throws Exception
    {
        this();
        this.setDataHandlerBaseDirectory(baseDirectory);
    }

    public void setTestNumberToRead(final int num)
    {
        this._testNumberToRead = num;
    }

    private String constructLatLonComponentOfFileNames(final CoordinateGridPoint gefsPt)
    {
        final DecimalFormat lonNF = new DecimalFormat("000.000000");
        lonNF.setNegativePrefix("");

        final DecimalFormat latNF = new DecimalFormat("00.000000");
        latNF.setNegativePrefix("");

        //lat/lon component of file names.
        String latLonStr = lonNF.format(gefsPt.getLongitude());
        if(gefsPt.getLongitude() < 0)
        {
            latLonStr += "W";
        }
        else
        {
            latLonStr += "E";
        }
        latLonStr += "_" + latNF.format(gefsPt.getLatitude());
        if(gefsPt.getLatitude() > 0)
        {
            latLonStr += "N";
        }
        else
        {
            latLonStr += "S";
        }
        return latLonStr;
    }

    private void loadPrecipitationPreparedTimeSeries(final LocationAndDataTypeIdentifier identifier,
                                                     final File days1to8File,
                                                     final File days9to16File)
    {
        try
        {
            final List<TimeSeriesArrays> ensembles = ProcessedGEFSFileTools.readPrecipitationFile(identifier,
                                                                                                  days1to8File,
                                                                                                  days9to16File,
                                                                                                  _testNumberToRead);
            _identifierToEnsembleMap.putAll(identifier, ensembles);
            _identifierToEnsembleMeanMap.putAll(identifier,
                                                GraphGenCalculatorTools.computeEnsembleMeanTimeSeries(ensembles));
        }
        catch(final Exception e)
        {
            LOG.info("Unable to load GEFS precipitation reforecasts for " + identifier.buildStringToDisplayInTree()
                + ": " + e.getMessage());
        }
    }

    private void loadTemperaturePreparedTimeSeries(final boolean minimumTemperature,
                                                   final LocationAndDataTypeIdentifier identifier,
                                                   final File days1to8File,
                                                   final File days9to16File)
    {
        try
        {
            final List<TimeSeriesArrays> ensembles = ProcessedGEFSFileTools.readTemperatureFile(minimumTemperature,
                                                                                                identifier,
                                                                                                days1to8File,
                                                                                                days9to16File,
                                                                                                _testNumberToRead);
            _identifierToEnsembleMap.putAll(identifier, ensembles);
            _identifierToEnsembleMeanMap.putAll(identifier,
                                                GraphGenCalculatorTools.computeEnsembleMeanTimeSeries(ensembles));
        }
        catch(final Exception e)
        {
            LOG.info("Unable to load GEFS temperature reforecasts (minimum temperature = " + minimumTemperature
                + ") for " + identifier.buildStringToDisplayInTree() + ": " + e.getMessage());
        }
    }

    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeriesForDiagnostics()
    {
        return ListTools.concat(TimeSeriesArraysTools.convertTimeSeriesArraysToList(TimeSeriesArraysTools.convertTimeSeriesArraysCollections(_identifierToEnsembleMap.values())),
                                _identifierToEnsembleMeanMap.values());
    }

    @Override
    public void setDataHandlerBaseDirectory(final File directory)
    {
        super.setDataHandlerBaseDirectory(directory);
        _preparedDataFileDirectory = new File(directory.getAbsolutePath() + "/processedASCIIGrids");
    }

    @Override
    public File getPreparedDataFilesDirectory()
    {
        return _preparedDataFileDirectory;
    }

    @Override
    public List<File> generateListOfPreparedDataFiles(final LocationAndDataTypeIdentifier identifier)
    {
//        final CoordinateGridPoint gefsPtDays1to8 = _gaussianGrid254Days1to8.findNearestPoint(new CoordinateGridPoint((double)Math.round(identifier.getUsedLatitude()),
//                                                                                                                     (double)Math.round(identifier.getUsedLongitude())),
//                                                                                             true);
//        final CoordinateGridPoint gefsPtDays9to16 = _gaussianGrid190Days9to16.findNearestPoint(new CoordinateGridPoint((double)Math.round(identifier.getUsedLatitude()),
//                                                                                                                       (double)Math.round(identifier.getUsedLongitude())),
//                                                                                               true);

        //Round to find nearest lat and lon. Changes refer to Fogbuz 1369
        final CoordinateGridPoint gefsPtDays1to8 = new CoordinateGridPoint(Math.ceil(identifier.getUsedLatitude() - 0.5),
                                                                           (double)Math.round(identifier.getUsedLongitude()));
        final CoordinateGridPoint gefsPtDays9to16 = new CoordinateGridPoint(Math.ceil(identifier.getUsedLatitude() - 0.5),
                                                                            (double)Math.round(identifier.getUsedLongitude()));

        final List<File> results = new ArrayList<File>();
        if((gefsPtDays1to8 == null) || (gefsPtDays9to16 == null)) //Coordinates are outside of defined area
        {
            LOG.warn("Location " + identifier.toString()
                + " has coordinates for which Gaussian grid points cannot be determined... weird.");
            return results;
        }

        final String days1to8LatLonStr = constructLatLonComponentOfFileNames(gefsPtDays1to8);
        final String days9to16LatLonStr = constructLatLonComponentOfFileNames(gefsPtDays9to16);

        if(identifier.isPrecipitationDataType())
        {
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_prcp_days1to8", "PRCP_6HR_"
                + days1to8LatLonStr + "_days1to8.GEFS.gz"));
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_prcp_days9to16", "PRCP_6HR_"
                + days9to16LatLonStr + "_days9to16.GEFS.gz"));
        }
        else if(identifier.isTemperatureDataType())
        {
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_tmin_days1to8", "TMIN_6HR_"
                + days1to8LatLonStr + "_days1to8.GEFS.gz"));
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_tmin_days9to16", "TMIN_6HR_"
                + days9to16LatLonStr + "_days9to16.GEFS.gz"));
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_tmax_days1to8", "TMAX_6HR_"
                + days1to8LatLonStr + "_days1to8.GEFS.gz"));
            results.add(FileTools.newFile(getPreparedDataFilesDirectory(), "gefs_ascii_tmax_days9to16", "TMAX_6HR_"
                + days9to16LatLonStr + "_days9to16.GEFS.gz"));
        }
        return results;
    }

    @Override
    public void loadOriginalTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        throw new Exception("GEFSDataHandler does not load the original grib2 time series.");
    }

    @Override
    public void loadPreparedTimeSeries(final List<LocationAndDataTypeIdentifier> identifiers) throws Exception
    {
        clearLoadedTimeSeries();

        LOG.info("Loading GEFS reforecasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
        for(final LocationAndDataTypeIdentifier identifier: identifiers)
        {
            final List<File> forecastFiles = generateListOfPreparedDataFiles(identifier);
            if(identifier.isPrecipitationDataType())
            {
                loadPrecipitationPreparedTimeSeries(identifier, forecastFiles.get(0), forecastFiles.get(1));
            }
            else if(identifier.isTemperatureDataType())
            {
                //TMIN
                loadTemperaturePreparedTimeSeries(true, identifier, forecastFiles.get(0), forecastFiles.get(1));

                //TMAX
                loadTemperaturePreparedTimeSeries(false, identifier, forecastFiles.get(2), forecastFiles.get(3));
            }
        }
        LOG.info("Done loading GEFS reforecasts for "
            + LocationAndDataTypeIdentifier.createListOfIdentifierStrings(identifiers) + "...");
    }

    @Override
    public void prepareDataFiles() throws Exception
    {
        throw new Exception("GEFSDataHandler does not prepare data.  "
            + "The processed ASCII grid files are prepared and delivered by OHD.");
    }

    @Override
    public Collection<TimeSeriesArray> getLoadedForecastTimeSeries(final LocationAndDataTypeIdentifier identifier)
    {
        return _identifierToEnsembleMeanMap.get(identifier);
    }

    @Override
    public Collection<TimeSeriesArray> getAllLoadedForecastTimeSeries()
    {
        return _identifierToEnsembleMeanMap.values();
    }

    @Override
    public void initialize() throws Exception
    {
    }

    @Override
    public boolean havePreparedDataFilesBeenCreatedAlready(final LocationAndDataTypeIdentifier identifier)
    {
        for(final File file: generateListOfPreparedDataFiles(identifier))
        {
            if(!file.exists())
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean arePreparedDataFilesUpToDate(final LocationAndDataTypeIdentifier identifier)
    {
        // No up-to-date checking needed.
        return true;
    }

    @Override
    public void clearLoadedTimeSeries()
    {
        _identifierToEnsembleMap.clear();
        _identifierToEnsembleMeanMap.clear();
    }

    @Override
    public Collection<LocationAndDataTypeIdentifier> getIdentifiersWithData()
    {
        return _identifierToEnsembleMeanMap.keys();
    }

//    @Override
//    public Collection<TimeSeriesArray> loadHindcastTimeSeries(final LocationAndDataTypeIdentifier identifier,
//                                                              final ParameterId dataTypeToLoad,
//                                                              final long forecastTime) throws Exception
//    {
//        //NOTE: the super method will catch problems loading time series below.  This wrapper is only needed because the precip forecast must
//        //be converted to 6h.
//        if(identifier.isTemperatureDataType())
//        {
//            return super.loadHindcastTimeSeries(identifier, dataTypeToLoad, forecastTime);
//        }
//        final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();
//        for(final TimeSeriesArray ts: super.loadHindcastTimeSeries(identifier, dataTypeToLoad, forecastTime))
//        {
//            final TimeSeriesArray ts6h = convert24HPrecipTo6H(ts);
//            results.add(ts6h);
//        }
//        return results;
//    }
}
