package ohd.hseb.hefs.mefp.sources.gefs;

import java.io.File;

import ohd.hseb.hefs.mefp.models.MEFPBaseModelControlOptions;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.MEFPPrecipitationModelControlOptions;
import ohd.hseb.hefs.mefp.models.temperature.MEFPTemperatureModelControlOptions;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.GenericMEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationSourceControlOptions;
import ohd.hseb.hefs.mefp.sources.AbstractMEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.MEFPSourceControlOptions;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventValuesGatherer;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.mefp.tools.canonical.StandardCanonicalEventValuesGatherer;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.core.ParameterEstimatorStepProcessor;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.sources.SourceDataHandler;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.notify.NoticePoster;
import ohd.hseb.hefs.utils.tools.ParameterId;

public class GEFSForecastSource extends AbstractMEFPForecastSource
{
    /**
     * This ===MUST=== match the portion of the source name before 'ForecastSource', so that it matches the default
     * return of {@link #getSourceId()}.
     */
    public final static String SOURCE_ID = "GEFS";

    @Override
    public String getName()
    {
        return "GEFS Forecasts";
    }

    @Override
    public MEFPSourceModelParameters getSourceModelParameters(final LocationAndDataTypeIdentifier identifier,
                                                              final AlgorithmModelParameters algorithmParameters)
    {
        final MEFPSourceModelParameters parameters = new MEFPSourceModelParameters(identifier,
                                                                                   this,
                                                                                   algorithmParameters,
                                                                                   false);
        return parameters;
    }

    @Override
    public GEFSDataHandler getSourceDataHandler()
    {
        return (GEFSDataHandler)super.getSourceDataHandler();
    }

    @Override
    protected void setSourceDataHandler(final SourceDataHandler handler)
    {
        if(!(handler instanceof GEFSDataHandler))
        {
            throw new IllegalArgumentException("The SourceDataHandler is not a GEFSDataHandler.");
        }
        super.setSourceDataHandler(handler);
    }

    @Override
    public MEFPSourceControlOptions createControlOptions(final ParameterId.Type type)
    {
        //XXX sensitivity analysis - uncomment the following to use daysBetweenT0s option.  See additional marked lines below.
//        final IntegerEstimationOption daysBetweenT0sOption = new IntegerEstimationOption("daysBetweenT0s",
//                                                                                         "Number of days between T0s",
//                                                                                         1,
//                                                                                         1,
//                                                                                         1,
//                                                                                         null)
//        {
//            @Override
//            public JComponent makeEditorEditor()
//            {
//                if(System.getProperties().getProperty("SHOW_DAYS_BETWEEN_T0S") == null)
//                {
//                    return null;
//                }
//                return super.makeEditorEditor();
//            }
//        };

        switch(type)
        {
            case TEMPERATURE:
                return new GenericMEFPSourceControlOptions("gefsTemperatureControlOptions", this);
//                                                           daysBetweenT0sOption);//XXX sensitivity analysis - uncomment to use and modify line above
            case PRECIPITATION:
                return new MEFPPrecipitationSourceControlOptions("gefsPrecipitationControlOptions", this);
//                                                                 daysBetweenT0sOption);//XXX sensitivity analysis - uncomment to use and modify line above
            default:
                throw new IllegalArgumentException(String.format("%s not supported.", type));
        }
    }

    @Override
    public void initializeSourceDataHandler(final File baseDirectory, final NoticePoster poster) throws Exception
    {
        this.setSourceDataHandler(new GEFSDataHandler(baseDirectory));
    }

    /**
     * I'm overriding this here only so that I can explain why it returns 0 (which the super class returns). <br>
     * <br>
     * Operationally, at a 12Z forecast time, the ensemble generated for 0Z is available (but not 6 or 12). Hence, there
     * is a 12-hour lag in operational forecast availability. But... that 12h lag is already accounted for in the
     * canonical event values. Those values computed from a 0Z reforecast time series are stored based on the first
     * following 12Z computational time (see {@link CanonicalEventValues}). Since all reforecasts are 0Z forecasts, the
     * 12h lag is therefore accounted for.
     */
    @Override
    public int getOperationalLagInHoursWhenEstimatingParameters(final LocationAndDataTypeIdentifier identifier)
    {
        return super.getOperationalLagInHoursWhenEstimatingParameters(identifier);
    }

    /**
     * I'm overriding this so that I can emphasize that I have thought about it and decided that this should return what
     * {@link #getOperationalLagInHoursWhenEstimatingParameters(LocationAndDataTypeIdentifier)} returns, which is what
     * is done in the super class.
     */
    @Override
    public int getOperationalLagInHoursWhenAcquringReforecast(final LocationAndDataTypeIdentifier identifier)
    {
        return super.getOperationalLagInHoursWhenAcquringReforecast(identifier);
    }

    /**
     * Returns gatherer for IPT events, by default. <br>
     * XXX sensitivity analysis
     */
    @Override
    public CanonicalEventValuesGatherer constructCanonicalEventValuesGatherer(final SourceCanonicalEventValues eventValues,
                                                                              final MEFPBaseModelControlOptions modelControlOptions,
                                                                              final MEFPSourceControlOptions sourceControlOptions)
    {
        if(modelControlOptions instanceof MEFPPrecipitationModelControlOptions)
        {
            //The gatherer is an IPT gatherer.  For EPT, the both minimum number should be DO_NOT_CHECK and the fraction defining wet events is different.
            //Those attributes can be set in the returned gatherer after construction.
            final MEFPPrecipitationModelControlOptions modelCtlParms = (MEFPPrecipitationModelControlOptions)modelControlOptions;
            return new StandardCanonicalEventValuesGatherer(eventValues.getForecastValues(),
                                                            eventValues.getObservedValues(),
                                                            modelCtlParms.getMinimumWidthOfDataWindowInDays(),
                                                            modelCtlParms.getMaximumWidthOfDataWindowInDays(),
                                                            sourceControlOptions.getInitialYear(),
                                                            sourceControlOptions.getFinalYear(),
//                                                            (Integer)((GenericMEFPSourceControlOptions)sourceControlOptions).getOptionWithName("daysBetweenT0s")
//                                                                                                                            .get(), //XXX sensitivity analysis - uncomment to use
                                                            modelCtlParms.getIPT().getFractionDefiningWetEvents(),
                                                            modelCtlParms.getMinimumNumberOfObservationsRequired(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveObservations(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveForecasts(),
                                                            modelCtlParms.getMinimumNumberOfRequiredPositiveObservations());
        }
        else
        {

            final MEFPTemperatureModelControlOptions modelCtlParms = (MEFPTemperatureModelControlOptions)modelControlOptions;
            return new StandardCanonicalEventValuesGatherer(eventValues.getForecastValues(),
                                                            eventValues.getObservedValues(),
                                                            modelCtlParms.getMinimumWidthOfDataWindowInDays(),
                                                            modelCtlParms.getMaximumWidthOfDataWindowInDays(),
                                                            sourceControlOptions.getInitialYear(),
                                                            sourceControlOptions.getFinalYear(),
//                                                            (Integer)((GenericMEFPSourceControlOptions)sourceControlOptions).getOptionWithName("daysBetweenT0s")
//                                                                                                                            .get(), //XXX sensitivity analysis - uncomment to use
                                                            modelCtlParms.getMinimumNumberOfObservationsRequired());
        }
    }

    @Override
    public ParameterEstimatorStepProcessor getSourceStepProcessor(final ParameterEstimatorRunInfo runInfo)
    {
        return new GEFSPEStepProcessor((MEFPParameterEstimatorRunInfo)runInfo);
    }
}
